package QBit::Application::Model::DBManager::_Utils::Fields;

use qbit;

use base qw(QBit::Class);

__PACKAGE__->mk_ro_accessors(
    {
        model                       => '__MODEL__',
        get_fields                  => '__FIELDS__',
        get_proc_field_names_sorted => '__PROC_FIELD_NAMES_SORTED__',
        get_field_need_delete_names => '__FIELD_NEED_DELETE_NAMES__',
        all_langs                   => '__ALL_LANGS__',
        get_related_models          => '__RELATED_MODELS__',
    }
);

__PACKAGE__->mk_accessors({stash => '__STASH__'});

sub get_db_fields {
    my ($self, $table) = @_;

    my %res    = ();
    my $fields = $self->{'__FIELDS__'};

    if (defined($table)) {
        foreach my $field_name (keys(%$fields)) {
            next
              unless ($fields->{$field_name}{'db'} || $fields->{$field_name}{'db_generated'} || '') eq ($table || '');
            $res{$field_name} = defined($fields->{$field_name}{'db_expr'}) ? $fields->{$field_name}{'db_expr'} : '';
        }
    } else {
        foreach my $field_name (keys(%$fields)) {
            next unless $fields->{$field_name}{'db'} || $fields->{$field_name}{'db_generated'};
            $res{$field_name} = defined($fields->{$field_name}{'db_expr'}) ? $fields->{$field_name}{'db_expr'} : '';
        }
    }

    return \%res;
}

sub init_check_rights {
    my ($self, $fields) = @_;

    foreach my $name (keys(%$fields)) {
        my $fld = $fields->{$name};
        my @rights;

        if ($fld->{'depends_on'}) {
            @rights = map {
                defined($fields->{$_}{'check_rights'})
                  ? (
                    ref($fields->{$_}{'check_rights'}) eq 'ARRAY'
                    ? @{$fields->{$_}{'check_rights'}}
                    : $fields->{$_}{'check_rights'}
                  )
                  : ()
            } @{$fld->{'depends_on'}};
        }

        if (@rights) {
            $fld->{'check_rights'} =
              array_uniq(@rights, (defined($fld->{'check_rights'}) ? $fld->{'check_rights'} : ()));

        } elsif (defined($fld->{'check_rights'}) && ref($fld->{'check_rights'}) ne 'ARRAY') {

            $fld->{'check_rights'} = [$fld->{'check_rights'}];
        }
    }

    return $fields;
}

sub init_field_sort {
    my ($self, $fields) = @_;

    my $orders = {};
    $orders->{$_} = _init_field_sort($orders, $fields, $_, 0) foreach keys(%$fields);

    return $orders;
}

sub init_fields {
    my ($fields_class, $original_fields, $model_class) = @_;

    _init_opts_fields($model_class, $original_fields) if grep {$_->{from_opts}} values(%$original_fields);

    my $fields = clone($original_fields);

    foreach my $name (keys(%$fields)) {
        _init_field_deps($model_class, $name, $fields);

        my $fld = $fields->{$name};
        if (exists($fld->{'get'}) || exists($fld->{'submodel'})) {
            $fld->{'_proc'} = 1;
        }
    }

    return $fields;
}

sub need {
    my ($self, $name) = @_;

    return exists($self->{'__FIELDS__'}{$name});
}

sub new {
    my ($class, $fields, $orders, $field_names, $model, %opts) = @_;

    my %res_fields;
    my %need_delete;
    my @unknown_fields;
    my $all_langs = $opts{'all_langs'};
    my %related_models;

    $field_names = [grep {$fields->{$_}{'default'}} keys(%$fields)] unless defined($field_names);
    $field_names = [$field_names] if ref($field_names) ne 'ARRAY';
    if (($field_names->[0] // '') eq '*') {
        $field_names = [grep {!$fields->{$_}->{lazy}} keys(%$fields)];
    }

  LOOP:
    foreach my $field (@$field_names) {
        unless (exists($fields->{$field})) {
            push(@unknown_fields, $field);
            next;
        }

        if ($fields->{$field}{'check_rights'}) {
            foreach my $right (@{$fields->{$field}{'check_rights'}}) {
                next LOOP if !$model->check_rights($right);
            }
        }
        $res_fields{$field} = $fields->{$field};
    }
    throw gettext('In model %s not found follows fields: %s', ref($model), join(', ', @unknown_fields))
      if @unknown_fields;

    foreach my $field (keys(%res_fields)) {
        if (exists($fields->{$field}{'depends_on'}) || exists($fields->{$field}{'forced_depends_on'})) {
            foreach
              my $dep_field (@{$fields->{$field}{'depends_on'} || []}, @{$fields->{$field}{'forced_depends_on'} || []})
            {
                if ($dep_field =~ /^(.+)\.(.+)$/) {
                    $related_models{$1}{$2} = 1;
                    next;
                }
                unless (exists($res_fields{$dep_field})) {
                    $res_fields{$dep_field}  = $fields->{$dep_field};
                    $need_delete{$dep_field} = TRUE;
                }
            }
        }
    }

    my @proc_fields = sort {($orders->{$a} || 0) <=> ($orders->{$b} || 0) || $a cmp $b}
      grep {$res_fields{$_}{'_proc'} || $all_langs && $res_fields{$_}{'i18n'}} keys(%res_fields);

    my $self = $class->SUPER::new(
        __MODEL__                   => $model,
        __FIELDS__                  => \%res_fields,
        __PROC_FIELD_NAMES_SORTED__ => \@proc_fields,
        __FIELD_NEED_DELETE_NAMES__ => [sort keys(%need_delete)],
        __ALL_LANGS__               => $all_langs,
        __RELATED_MODELS__          => \%related_models,
        __STASH__                   => {},
    );

    weaken($self->{'model'});

    return $self;
}

sub process_data {
    my ($self, $data) = @_;

    my $fields         = $self->{'__FIELDS__'};
    my $proc_fields    = $self->{'__PROC_FIELD_NAMES_SORTED__'};
    my $need_delete    = $self->{'__FIELD_NEED_DELETE_NAMES__'};
    my $related_models = $self->{'__RELATED_MODELS__'};

    #throw gettext('Cannot get field "%s"', $field);

    foreach my $model (keys(%$related_models)) {
        my $def = $self->model->related_models->{$model};

        my $accessor = $def->{'accessor'};
        throw
          sprintf("No 'accessor' defined for related model \"%s.related_models.%s\" ", $self->model->accessor, $model)
          unless defined $accessor;
        throw sprintf("No accessor in related model \"%s.%s\"", $self->model->accessor, $accessor)
          unless $self->model->can($accessor);

        my $tmp_grant_rights;
        if (my $grant_rights = $def->{grant_rights}) {
            $tmp_grant_rights = $self->model->$accessor->app->add_tmp_rights(@$grant_rights);
        }

        # TODO: optimize this foreach
        # and optimize 2 cases, one index and many indexes:
        #   $_->{$def->{'key_fields'}[0]}
        #   join($;, map {$row->{$_} // chr(2)} @{$def->{'key_fields'}})
        #
        foreach my $row (
            @{
                $self->model->$accessor->get_all(
                    fields => [sort keys %{$related_models->{$model}}, @{$def->{'key_fields'}}],
                    filter => $def->{'filter'}->($self,                $data),
                )
            }
          )
        {
            if ($def->{'value_type'} && $def->{'value_type'} eq 'array') {
                # Autovivification in 'push' also works: push(@{$h->{'a','b','c'}}, 'bbb').
                push(@{$self->{$model}{join($;, map {$row->{$_} // chr(2)} @{$def->{'key_fields'}})}}, $row);
            } elsif ($def->{'value_type'} && $def->{'value_type'} =~ /^array_(.+)$/) {
                push(@{$self->{$model}{join($;, map {$row->{$_} // chr(2)} @{$def->{'key_fields'}})}}, $row->{$1});
            } else {
                $self->{$model}{join($;, map {$row->{$_} // chr(2)} @{$def->{'key_fields'}})} = $row;
            }
        }
    }

    # prefetch data for "submodel"
    foreach my $field (@$proc_fields) {
        if (exists($fields->{$field}{'submodel'})) {
            my $sm = $fields->{$field}{'submodel'};
            $sm->{depends_on} = [grep((ref($sm->{$_}) eq 'CODE'), keys(%$sm))];
            $sm->{model} = $sm->{$sm->{depends_on}->[0]}->($self);
            rows_expand($data, $sm->{depends_on}->[0] => $sm->{model} => {$field => $sm->{fields}});
        }
    }

    my @locale_names;
    my $locales_fetched;

    foreach my $rec (@$data) {
        foreach my $field (@$proc_fields) {
            if (exists($fields->{$field}{'get'})) {
                $rec->{$field} = $fields->{$field}{'get'}($self, $rec);
            } elsif (exists($fields->{$field}{'submodel'})) {
                # SSP tests expect undef for non-existing submodel fields.
                # Maybe this defaulting should be moved above to submodel prefetch.
                $rec->{$field} //= undef;
            } elsif ($fields->{$field}{'i18n'}) {
                # This case should be active only if all_langs is true.
                unless ($locales_fetched) {
                    @locale_names = keys(%{$self->model->get_option('locales', {})}) if $self->model;
                    $locales_fetched++;
                }
                $rec->{$field} = {map {$_ => $rec->{"${field}_${_}"}} @locale_names};
            }
        }

        # PI-25944 Костыль, чтобы сохранить типизацию, на которую фронт завязан (можно убрать после выпиливания rosetta)
        $self->_adjust_type($fields, $rec);

        foreach my $field (@$need_delete) {
            delete($rec->{$field});
        }
    }

    return $data;
}

# PI-25944 Костыль, чтобы сохранить типизацию, на которую фронт завязан (можно убрать после выпиливания rosetta)
sub _adjust_type {
    my ($self, $fields, $rec) = @_;

    return if ($ENV{SYSTEM} && $ENV{SYSTEM} eq 'restapi' && !($ENV{SUBSYSTEM} && $ENV{SUBSYSTEM} eq 'rosetta'));

    foreach my $field (keys %$fields) {
        my $type = $fields->{$field}{adjust_type};
        if (defined($rec->{$field}) && $type) {
            if ($type eq 'int') {
                $rec->{$field} += 0;
            } elsif ($type eq 'str') {
                $rec->{$field} .= '';
            } elsif ($type eq 'hash_int') {
                if (ref($rec->{$field}) eq 'HASH') {
                    foreach (values %{$rec->{$field}}) {
                        $_ += 0 if (defined $_);
                    }
                }
            } elsif ($type eq 'array_str') {
                if (ref($rec->{$field}) eq 'ARRAY') {
                    foreach (@{$rec->{$field}}) {
                        $_ .= '' if (defined $_);
                    }
                }
            } elsif ($type eq 'array_hash_str') {
                if (ref($rec->{$field}) eq 'ARRAY') {
                    foreach my $hash (@{$rec->{$field}}) {
                        if (ref($hash) eq 'HASH') {
                            foreach (values %{$hash}) {
                                $_ .= '' if (defined $_);
                            }
                        }
                    }
                }
            }
        }
    }
}

sub _init_field_deps {
    my ($model_class, $name, $fields, $orig_field_name, $deps_name) = @_;

    # Related models' fields not processed here.
    if ($name =~ /\./) {
        return {
            depends_on        => [],
            forced_depends_on => [],
        };
    }

    unless (exists $fields->{$name}) {
        my $all_fields_names = join(', ', map("\"$_\"", keys(%{$fields || {}})));

        my $field_name =
          $deps_name
          ? join('.', $orig_field_name, $deps_name, $name)
          : $name;

        throw gettext('%s: field "%s" does not exists in set: %s', $model_class, $field_name, $all_fields_names);
    }

    my $fld = $fields->{$name};

    if ($fld->{'submodel'}) {
        my $deps = [grep((ref($fld->{'submodel'}->{$_}) eq 'CODE'), keys(%{$fld->{'submodel'}}))];
        $fld->{'depends_on'} = $fld->{'submodel_depends_on'} = $deps;
    }

    my @result    = {};
    my $init_data = {};
    foreach my $init_prop_name (qw( submodel_depends_on   depends_on  forced_depends_on )) {
        my @data = map {ref($_) ? @$_ : $_} $fld->{$init_prop_name};
        my $data_prop_name = $init_prop_name eq 'submodel_depends_on' ? 'depends_on' : $init_prop_name;
        push @{$init_data->{$data_prop_name}}, @data;
        if (@data) {
            my $prop_name = $init_prop_name eq 'submodel_depends_on' ? 'submodel' : $init_prop_name;
            push @result, map {_init_field_deps($model_class, $_, $fields, $name, $prop_name)} grep {$_} @data;
        }
    }

    delete $fld->{'submodel_depends_on'};

    my $deps        = array_uniq($init_data->{depends_on}        // [], map {$_->{depends_on}        // []} @result);
    my $forced_deps = array_uniq($init_data->{forced_depends_on} // [], map {$_->{forced_depends_on} // []} @result);

    if (@$deps) {
        $fld->{'depends_on'} = $deps;
    } else {
        delete $fld->{'depends_on'};
    }

    if (@$forced_deps) {
        $fld->{'forced_depends_on'} = $forced_deps;
    } else {
        delete $fld->{'forced_depends_on'};
    }

    return {
        depends_on        => $deps,
        forced_depends_on => $forced_deps,
    };
}

sub _init_field_sort {
    my ($orders, $fields, $name, $level) = @_;

    return $orders->{$name} + $level if exists($orders->{$name});

    my @foreign_fields = (@{$fields->{$name}{'depends_on'} || []}, @{$fields->{$name}{'forced_depends_on'} || []});

    return @foreign_fields
      ? array_max(map {_init_field_sort($orders, $fields, $_, $level + 1)} @foreign_fields)
      : $level;
}

sub _setup_get {
    my ($field, $name) = @_;

    weaken($field);
    my $orig_get = delete $field->{get};

    $field->{get} = sub {
        my ($fields, $row) = @_;
        my $value = $row->{$name};

        my $v;
        if (defined $value) {
            # if quoted / unquoted (-> | ->>)
            try {
                $v = from_json($value);
            }
            catch {
                $v = $value;
            };
        }
        $row->{$name} = $v;
        $row->{$name} = $orig_get->($fields, $row) if $orig_get;
        return $row->{$name};
    };
}

sub _init_opts_fields {
    my ($model_class, $fields) = @_;

    $fields->{opts} //= {
        db  => TRUE,
        get => sub {
            my ($fields, $row) = @_;
            utf8::decode($row->{'opts'});
            from_json($row->{'opts'});
        },
        type       => 'complex',
        need_check => {type => 'ref', optional => TRUE},
    };

    throw "Please define json schema for $model_class opts field and put it's name inside get_opts_schema_name method."
      unless $model_class->can('get_opts_schema_name');

    foreach my $name (keys(%$fields)) {
        my $field = $fields->{$name};
        my $query_type = $field->{from_opts} // next;

        if ($query_type eq 'from_hash') {
            $field->{depends_on} //= [];
            push @{$field->{depends_on}}, 'opts';

            my $orig_get = delete $field->{get};
            $field->{get} = sub {
                my ($fields, $row) = @_;
                $row->{$name} = $row->{opts}{$name};
                $row->{$name} = $orig_get->($fields, $row) if $orig_get;
                $row->{$name};
            };
        } elsif ($query_type eq 'db_expr') {
            $field->{db} //= 1;
            $field->{db_expr} //= ['opts' => '->' => \"\$.$name"];
            _setup_get($field, $name);
        } elsif ($query_type eq 'db_generated') {
            $field->{db_generated} //= 1;
            _setup_get($field, $name);
        } else {
            throw
"Unknown opts query type $query_type for field $name from model $model_class. Expected one of: from_hash, db_expr, db_generated";
        }
    }
}

TRUE;
