package QBit::Application::Model::Multistate::DB;

use qbit;

use base qw(QBit::Application::Model::Multistate);

use Exception::Multistate::BadAction;
use Exception::Multistate::NotFound;

__PACKAGE__->abstract_methods(
    qw(
      _multistate_db_table
      )
);

sub check_action {
    my ($self, $object, $action, %opts) = @_;

    $object = $self->_get_object_fields($object, ['multistate']);

    throw Exception::Multistate::NotFound gettext('No object for action "%s"', $action) unless defined($object);

    return FALSE unless exists($object->{'multistate'});
    return FALSE unless $self->check_multistate_action($object->{'multistate'}, $action);

    my $can_action_sub_name = "can_action_$action";
    return FALSE if $self->can($can_action_sub_name) && !$self->$can_action_sub_name($object, %opts);

    return TRUE;
}

sub do_action {
    #shift->_do_action(0, @_);
    # fool caller
    unshift(@_, shift, 0);
    goto &_do_action;
}

sub do_action_with_result {
    #shift->_do_action(3, @_);
    # fool caller
    unshift(@_, shift, 3);
    goto &_do_action;
}

sub force_do_action {
    #
    # This variant may only be useful to eliminate can_action() checks.
    # If an action you're trying to do is not reachable in the state graph from the current state then it will fail.
    #

    #shift->_do_action(2, @_);
    # fool caller
    unshift(@_, shift, 2);
    goto &_do_action;
}

sub get_action_log_entries {
    my ($self, $id_elem, %opts) = @_;

    my $fields = [map {"elem_$_"} @{$self->_action_log_db_table->{'elem_table_pk'}}];

    my $id = {};
    if (ref($id_elem) ne 'HASH' and @$fields > 1) {
        throw gettext('Bad argument. Need hash.');
    } elsif (ref($id_elem) ne 'HASH' and @$fields == 1) {
        $id->{$fields->[0]} = $id_elem;
    } elsif (ref($id_elem) eq 'HASH') {
        $id->{"elem_$_"} = $id_elem->{$_} foreach keys %$id_elem;
        throw gettext(
            'Cannot find fields. Need (%s), got (%s).',
            join(', ', @{$self->_action_log_db_table->{'elem_table_pk'}}),
            join(', ', keys(%$id_elem))
        ) if grep {!exists($id->{$_})} @$fields;
    }

    my $filter = $self->_action_log_db_table->db->filter();

    $filter->and([$_ => '=' => \$id->{$_}]) foreach @$fields;

    if (grep {$opts{$_}} qw(fd td)) {
        $filter->and([dt => '>=' => \$opts{'fd'}]) if $opts{'fd'};
        $filter->and([dt => '<=' => \$opts{'td'}]) if $opts{'td'};
    }

    my $res = $self->_action_log_db_table()->get_all(
        filter   => $filter,
        order_by => [qw(dt id)]
    );

    if (grep {$opts{$_}} qw(explain_actions explain_multistates expand_opts)) {
        foreach (@$res) {
            if ($opts{'explain_actions'}) {
                $_->{'action_name'} = $self->get_action_name($_->{'action'});
            }
            if ($opts{'explain_multistates'}) {
                $_->{'old_multistate_name'} = $self->get_multistate_name($_->{'old_multistate'});
                $_->{'new_multistate_name'} = $self->get_multistate_name($_->{'new_multistate'});
            }
            if ($opts{'expand_opts'}) {
                $_->{'opts'} = from_json($_->{'opts'});
            }
        }
    }

    return $res;
}

sub get_actions {
    my ($self, $object) = @_;

    $object = $self->_get_object_fields($object, ['multistate']);

    return {
        map {$_ => $self->get_action_name($_)}
          grep {$self->check_action($object, $_)}
          keys(%{$self->get_multistates()->{$object->{'multistate'}} || {}})
    };
}

sub maybe_do_action {
    #shift->_do_action(1, @_);
    # fool caller
    unshift(@_, shift, 1);
    goto &_do_action;
}

sub _action_log_db_table { }

sub _do_action {
    my ($self, $mode, $object, $action, %opts) = @_;

    my $pk =
      ref($object) eq 'HASH'
      ? {map {$_ => $object->{$_}} @{$self->_multistate_db_table->primary_key}}
      : $object;

    my $new_multistate;
    my $result;

    $self->_multistate_db_table->db->transaction(
        sub {
            $object = $self->_get_object_fields(
                $pk,
                [
                    @{$self->_multistate_db_table->primary_key}, 'multistate',
                    (ref($object) eq 'HASH' ? keys(%$object) : ())
                ],
                for_update => TRUE
            );
            throw Exception::Multistate::NotFound gettext('No object for action "%s"', $action) unless defined($object);

            unless ($self->check_action($object, $action, %opts)) {
                if ($mode == 0 || $mode == 3) {
                    my $multistate = $object ? $object->{'multistate'} // 'undef' : 'undef';

                    my $action_exists =
                      exists($self->get_multistates()->{$multistate}{$action}) ? "exists" : "not exist";

                    my $right = $self->get_registered_actions_rights->{$action};
                    my $right_given = defined($right) ? ($self->check_rights($right) ? ' given' : ' not given') : '';

                    $self->throw_error_by_action($object, $action) if $self->can('throw_error_by_action');

                    throw Exception::Multistate::BadAction gettext(
'Cannot do action "%s" (login "%s", multistate %s, pk %s, model "%s", action %s, right "%s"%s).',
                        $action,
                        $self->get_option('cur_user', {})->{'login'} || 'unknown',
                        $multistate,
                        to_json($pk, canonical => TRUE),
                        $self->accessor(),
                        $action_exists,
                        $right // 'not defined',
                        $right_given
                      ),
                      sentry => {
                        fingerprint => ['Exception::Multistate::BadAction', $self->accessor(), $action],
                        extra       => {
                            accessor => $self->accessor(),
                            action   => $action,
                        },
                      };
                } elsif ($mode == 1) {
                    return;
                }
            }

            $new_multistate = $self->get_multistates()->{$object->{'multistate'}}{$action};
            $self->_multistate_db_table()->edit($pk, {multistate => $new_multistate});

            my $storage = defined(blessed($self)) ? $self : package_stash($self);
            my $ignore_actions = $storage->{'__DONT_LOG_ACTIONS__'};
            if ($self->_action_log_db_table() && !$ignore_actions->{$action}) {
                my @opts;
                if ($self->_action_log_db_table()->have_fields('opts')) {
                    my %tmp_opts = %opts;
                    $tmp_opts{password} = '?' if exists $tmp_opts{password};
                    @opts = (opts => to_json(\%tmp_opts));
                }
                $self->_action_log_db_table()->add(
                    {
                        user_id => $self->get_option('cur_user', {})->{'id'},
                        (map {("elem_$_" => $object->{$_})} @{$self->_multistate_db_table->primary_key}),
                        old_multistate => $object->{'multistate'},
                        action         => $action,
                        new_multistate => $new_multistate,
                        dt             => curdate(oformat => 'db_time'),
                        comment        => $self->_get_comment($object, $action, \%opts),
                        @opts,
                    }
                );
            }

            my $on_action_name = "on_action_$action";
            if ($self->can($on_action_name)) {
                $result = $self->$on_action_name($object, %opts);
            }
        }
    );

    return $mode == 3 ? $result : $new_multistate;
}

sub _get {
    my ($self, $object, %opts) = @_;

    return $self->_multistate_db_table->get($object, %opts);
}

sub _get_object_fields {
    my ($self, $object, $fields, %opts) = @_;

    if (ref($object) eq 'HASH') {
        return $object if !$opts{'for_update'} && scalar(grep {exists($object->{$_})} @$fields) == @$fields;

        throw gettext(
            'Cannot find PK fields. Need (%s), got (%s).',
            join(', ', @{$self->_multistate_db_table->primary_key}),
            join(', ', keys(%$object))
        ) if grep {!exists($object->{$_})} @{$self->_multistate_db_table->primary_key};
    }

    push(@$fields, @{$self->_multistate_db_table->primary_key});

    return $self->_get(
        $object,
        for_update => $opts{'for_update'},
        fields     => array_uniq(@$fields)
    );
}

sub _get_comment {
    return '';
}

TRUE;
