package QBit::Application::Model::RBAC::DB;

use qbit;

use base qw(QBit::Application::Model::RBAC);

use Exception::Denied;
use Exception::Validation::BadArguments;

use RoleRights qw(get_role_right_object_by_stage);

__PACKAGE__->model_accessors(db => 'QBit::Application::Model::DB::RBAC',);

sub cur_user_is_internal {
    my ($self) = @_;

    my $is_internal = grep {$_->{is_internal}} values %{$self->get_cur_user_roles()};
    return $is_internal;
}

sub get_cur_user_roles {
    my ($self) = @_;

    my $cur_user = $self->get_option('cur_user');
    return undef unless defined($cur_user);

    return $self->get_roles_by_user_id($cur_user->{'id'});
}

sub get_roles_by_rights {
    my ($self, $rights) = @_;

    throw Exception::Validation::BadArguments(gettext("'rights' must be defined"))
      unless defined($rights);

    if (ref($rights) eq '') {
        $rights = [$rights];
    } elsif (ref($rights) ne 'ARRAY') {
        throw Exception::Validation::BadArguments(gettext("Expected that 'rights' is a ref to an ARRAY or string"));
    }

    my $db          = get_role_right_object_by_stage($self->get_option('stage'));
    my $role_rights = $db->get_rights();

    my %roles = ();
    foreach my $right (@$rights) {
        $roles{$_} = TRUE foreach keys %{$role_rights->{$right}};
    }

    return [sort {$a <=> $b} keys(%roles)];
}

sub get_roles_by_user_id {
    my ($self, $user_id) = @_;

    my $roles = $self->db->user_role->get_all(
        fields   => [qw(role_id)],
        filter   => {user_id => $user_id},
        distinct => 1,
    );

    $roles = $self->get_roles(ids => [map {$_->{'role_id'}} @$roles]);

    my %hash_roles = map {$_->{id} => $_} @$roles;

    # PI-25944 Костыль, чтобы сохранить типизацию, на которую фронт завязан (можно убрать после выпиливания rosetta)
    foreach my $role (values %hash_roles) {
        $role->{id} .= "" if (defined($role->{id}));
        $role->{is_internal} += 0 if (defined($role->{is_internal}));
    }

    return \%hash_roles;
}

sub get_rights_by_user_id {
    my ($self, $user_id, $right_names) = @_;

    my $user_roles = $self->get_roles_by_user_id($user_id);

    my $role_ids = [sort {$a <=> $b} keys %$user_roles];

    my $db          = get_role_right_object_by_stage($self->get_option('stage'));
    my $role_rights = $db->get_rights_by_role_ids($role_ids);

    my $rights = {map {$_ => $role_rights->{$_} ? TRUE : FALSE} ($right_names ? @$right_names : keys %$role_rights)};

    return $rights;
}

sub get_roles_rights {
    my ($self, %opts) = @_;

    $opts{'fields'} = [qw(right role_id)] unless exists($opts{'fields'});

    my $db = get_role_right_object_by_stage($self->get_option('stage'));

    return $db->get_roles_rights($opts{role_id}, $opts{fields});
}

sub revoke_roles {
    my ($self, $user_id, $roles) = @_;

    throw Exception::Denied unless $self->check_rights('rbac_revoke_roles');

    throw Exception::Validation::BadArguments(gettext("'user_id' must be defined"))
      unless defined($user_id);
    throw Exception::Validation::BadArguments(gettext("'role_id' must be defined"))
      unless defined($roles);

    $self->db->user_role->delete({user_id => $user_id, role_id => $roles});
}

sub role_add {
    my ($self, %role) = @_;

    throw Exception::Denied unless $self->check_rights('rbac_role_add');

    return $self->db->roles->add(\%role);
}

sub set_user_role {
    my ($self, $user_id, $roles) = @_;

    throw Exception::Denied unless $self->check_rights('rbac_user_role_set');

    throw Exception::Validation::BadArguments(gettext("'user_id' must be defined"))
      unless defined($user_id);
    throw Exception::Validation::BadArguments(gettext("'role_id' must be defined"))
      unless defined($roles);

    $self->db->user_role->replace({user_id => $user_id, role_id => $_}) foreach @$roles;
}

TRUE;
