package QBit::Application::Part;

use qbit;

use base qw(QBit::Class);

__PACKAGE__->mk_accessors(qw(app));

sub init {
    my ($self) = @_;

    $self->SUPER::init();

    throw gettext('Required opt "app" must be QBit::Application descendant')
      unless $self->{'app'} && $self->{'app'}->isa('QBit::Application');

    weaken($self->{'app'});

    my $rights = $self->get_structure_rights_to_register();
    $self->register_rights($rights) if @$rights;

    if ($self->isa('QBit::Application::Model::DBManager')) {
        my $fields = $self->get_structure_model_fields();
        $self->model_fields($fields) if %$fields;

        my $filter = $self->get_structure_model_filter();
        $self->model_filter($filter) if %$filter;

    }
    if ($self->isa('QBit::Application::Model::Multistate')) {
        my $multistates_graph = $self->get_structure_multistates_graph();
        $self->multistates_graph($multistates_graph) if %$multistates_graph;
    }

}

__PACKAGE__->mk_ro_self_or_stash_accessors(
    {
        get_model_accessors       => '__MODEL_ACCESSORS__',
        get_register_right_groups => '__RIGHT_GROUPS__',
        get_register_rights       => '__RIGHTS__',
    }
);

sub accessor {
    throw "accessor() is abstract method";
}

=head2 check_short_rights
    my $bool =  $self->check_short_rights('edit');
=cut

sub check_short_rights {
    my ($self, $right) = @_;

    return $self->check_rights($self->get_right($right));
}

sub get_right {
    my ($self, $right) = @_;

    return $self->accessor() . '_' . $right;
}

sub model_accessors {
    my ($class, %accessors) = @_;

    package_stash($class)->{'__MODEL_ACCESSORS__'} = \%accessors;
}

sub register_rights {
    my ($self, $data) = @_;

    # When this sub is called for an object store in the object.
    # When this sub is called for a package store in that package stash.
    #
    my $storage = defined(blessed($self)) ? $self : package_stash($self);

    $storage->{'__RIGHT_GROUPS__'} = {} unless exists($storage->{'__RIGHT_GROUPS__'});
    $storage->{'__RIGHTS__'}       = {} unless exists($storage->{'__RIGHTS__'});

    foreach my $group (@$data) {
        $group->{'name'} = '__UNDEFINED__' unless exists($group->{'name'});
        $storage->{'__RIGHT_GROUPS__'}{$group->{'name'}} = $group->{'description'} if exists($group->{'description'});

        while (my ($right, $right_name) = each(%{$group->{'rights'}})) {
            $storage->{'__RIGHTS__'}{$right} = {name => $right_name, group => $group->{'name'}};
        }
    }

    # When we define for a model object, not a package, we need to copy it to the app package for get_registered_rights() to work.
    # When we define for a package copying to app is done in that package import().
    #
    if (defined(blessed($self))) {
        my $app_pkg_stash = package_stash(blessed($self->{'app'}));

        $app_pkg_stash->{'__RIGHT_GROUPS__'} =
          {%{$app_pkg_stash->{'__RIGHT_GROUPS__'} || {}}, %{$storage->{'__RIGHT_GROUPS__'} || {}}};

        $app_pkg_stash->{'__RIGHTS__'} = {%{$app_pkg_stash->{'__RIGHTS__'} || {}}, %{$storage->{'__RIGHTS__'} || {}}};
    }
}

sub get_structure_rights_to_register {return []}

sub import {
    my ($package, %opts) = @_;

    my $acc_aliases = $opts{'models'} // {};

    no strict 'refs';

    my $i = 0;
    my $app_pkg;
    while ($app_pkg = caller($i)) {
        last if $app_pkg->isa('QBit::Application');
        $i++;
    }
    throw gettext('Use only in QBit::Application descendant')
      unless $app_pkg->isa('QBit::Application');

    my $app_pkg_stash = package_stash($app_pkg);

    # Always initialize accumulator or undefined $res will be auto-vivified inside and you will never see the result
    my $rights = {};
    package_merge_isa_data(
        $package, $rights,
        sub {
            my ($ipackage, $res) = @_;

            my $ipkg_stash = package_stash($ipackage);
            $res->{'__RIGHTS__'} = {%{$res->{'__RIGHTS__'} || {}}, %{$ipkg_stash->{'__RIGHTS__'} || {}}};
            $res->{'__RIGHT_GROUPS__'} =
              {%{$res->{'__RIGHT_GROUPS__'} || {}}, %{$ipkg_stash->{'__RIGHT_GROUPS__'} || {}}};
        },
        __PACKAGE__
    );

    $app_pkg_stash->{'__RIGHT_GROUPS__'} =
      {%{$app_pkg_stash->{'__RIGHT_GROUPS__'} || {}}, %{$rights->{'__RIGHT_GROUPS__'} || {}}};

    $app_pkg_stash->{'__RIGHTS__'} = {%{$app_pkg_stash->{'__RIGHTS__'} || {}}, %{$rights->{'__RIGHTS__'} || {}}};

    my $accessors = {};
    package_merge_isa_data(
        $package,
        $accessors,
        sub {
            my ($ipackage, $res) = @_;

            my $ipkg_stash = package_stash($ipackage);

            $res->{'__MODEL_ACCESSORS__'} = {
                %{$res->{'__MODEL_ACCESSORS__'}        || {}},
                %{$ipkg_stash->{'__MODEL_ACCESSORS__'} || {}},
                %{
                      $ipackage->can('get_structure_model_accessors')
                    ? $ipackage->get_structure_model_accessors() // {}
                    : {}
                  },
            };
        },
        __PACKAGE__
    );

    $accessors = $accessors->{'__MODEL_ACCESSORS__'} // {};
    package_stash($package)->{'__MODEL_ACCESSORS__'} = $accessors;

    foreach my $acc_name (keys %$accessors) {
        my $acc_class = $accessors->{$acc_name};
        my $app_accessor = $acc_aliases->{$acc_name} // $acc_name;

        if ($acc_class) {
            *{"${package}::${acc_name}"} = sub {
                $_[0]->{'__ACCESSORS__'}{$acc_name} //= do {
                    my $model = $_[0]->app->$app_accessor;
                    throw gettext(
'Class "%s" expects application accessor "%s" used through model accessor "%s" to be "%s" descendant but got %s',
                        $package,
                        $app_accessor,
                        $acc_name,
                        $acc_class,
                        ref($model)
                    ) unless ref($model) && $model->isa($acc_class);
                    weaken($model);
                    return $model;
                };
            };
        } else {
            *{"${package}::${acc_name}"} = sub {$_[0]->app->$app_accessor};
        }
    }

    foreach my $method (qw(check_rights)) {
        *{"${package}::${method}"} = sub {shift->app->$method(@_)};
    }
}

TRUE;
