
=head1 Name

QBit::Array - Functions to manipulate arrays.

=cut

package QBit::Array;

use strict;
use warnings;
use utf8;

use base qw(Exporter);

BEGIN {
    our (@EXPORT, @EXPORT_OK);

    @EXPORT = qw(
      in_array arrays_intersection arrays_difference array_uniq
      array_n_min array_min array_n_max array_max
      array_avg rows_expand arrays_eq
      );
    @EXPORT_OK = @EXPORT;
}

=head2 array_avg

B<Arguments:>

=over

=item

B<@array> - array of numbers.

=back

B<Return value:> number, average value.

=cut

sub array_avg(@) {
    my $sum = 0;
    $sum += $_ foreach @_;
    return $sum / @_;
}

=head2 array_max

B<Arguments:>

=over

=item

B<@array> - array of strings.

=back

B<Return value:> string, max value (string comparasion).

=cut

sub array_max(@) {
    my $max = $_[0];
    foreach (@_) {
        $max = $_ if $max lt $_;
    }
    return $max;
}

=head2 array_min

B<Arguments:>

=over

=item

B<@array> - array of strings.

=back

B<Return value:> string, min value (string comparasion).

=cut

sub array_min(@) {
    my $min = $_[0];
    foreach (@_) {
        $min = $_ if $min gt $_;
    }
    return $min;
}

=head2 array_n_max

B<Arguments:>

=over

=item

B<@array> - array of numbers.

=back

B<Return value:> number, max value (numeric comparasion).

=cut

sub array_n_max(@) {
    my $max = $_[0];
    foreach (@_) {
        $max = $_ if $max < $_;
    }
    return $max;
}

=head2 array_n_min

B<Arguments:>

=over

=item

B<@array> - array of numbers.

=back

B<Return value:> number, min value (numeric comparasion).

=cut

sub array_n_min(@) {
    my $min = $_[0];
    foreach (@_) {
        $min = $_ if $min > $_;
    }
    return $min;
}

=head2 array_uniq

B<Arguments:>

=over

=item

B<@array> - each element may be array ref or scalar.

=back

B<Return value:> array ref, unique values from all arrays.

=cut

sub array_uniq(@) {
    my %hs;
    @hs{ref($_) eq 'ARRAY' ? grep {defined($_)} @$_ : ($_)} = () for @_;
    return [keys %hs];
}

=head2 arrays_difference

Вычитает из первого массива второй (массивы не модифицирует).

B<Arguments:>

=over

=item

B<$array1> - array ref, minuend;

=item

B<$array2> - array ref, subtrahend.

=back

B<Return value:> array ref.

=cut

sub arrays_difference($$) {
    my ($array1, $array2) = @_;

    my %hs;
    @hs{@$array2} = ();

    return [grep {!exists($hs{$_})} @$array1];
}

=head2 arrays_intersection

B<Arguments:>

=over

=item

B<$array_ref1>;

=item

B<$array_ref2>;

=item

B<...>;

=item

B<$array_refN>.

=back

B<Return value:> array ref, intersection of all arrays (unique values).

=cut

sub arrays_intersection(@) {
    my %hs = ();
    foreach my $array (map {array_uniq($_)} @_) {
        exists($hs{$_}) ? ($hs{$_}++) : ($hs{$_} = 1) for @$array;
    }

    return [grep {$hs{$_} == @_} keys %hs];
}

=head1 Functions

=head2 in_array

B<Arguments:>

=over

=item

B<$elem> - scalar;

=item

B<$array> - array ref.

=back

B<Return value:> boolean.

=cut

sub in_array($$) {
    my ($elem, $array) = @_;

    my %hs;
    @hs{@$array} = ();

    return exists $hs{$elem};
}

=head2 arrays_eq

B<Arguments:>

=over

=item

B<$array_ref1>;

=item

B<$array_ref2>;

=item

B<...>;

=item

B<$array_refN>.

=back

B<Return value:> boolean
                 1 if arrays have the same elements (include duplicates),
                 '' otherwise.

=cut

sub arrays_eq(@) {
    my ($first_array, @other) = @_;

    return '' unless ref($first_array) eq 'ARRAY';

    my $size = scalar(@$first_array);
    foreach my $array (@other) {
        return '' unless ref($array) eq 'ARRAY';
        return '' if $size != scalar(@$array);
    }

    my @etalon = sort {defined($a) cmp defined($b) || $a cmp $b} @$first_array;
    foreach my $array (@other) {
        my @sort_array = sort {defined($a) cmp defined($b) || $a cmp $b} @$array;
        for (my $i = 0; $i < $size; $i++) {
            return ''
              unless ((!defined($etalon[$i]) && !defined($sort_array[$i]))
                || (defined($etalon[$i]) && defined($sort_array[$i]) && $etalon[$i] eq $sort_array[$i]));
        }
    }

    return 1;
}

=head2 rows_expand

Expand some data in set with model, hash or other set

Usage:

 rows_expand($list, 'user_id' => $users_hash => 'user')

 rows_expand($list, 'user_id' => $app->users_model => 'user')

 rows_expand($list, 'user_id' => $app->users_model => {'user' => ['user_id', 'user_some_field']})

 # explicit PK
 rows_expand($list, 'user_id' => [ 'pk_id' => $list_array ] => 'user')

 rows_expand($list, 'user_id' => [ 'other_pk_id' => $app->users_model ] => 'user')

=cut

sub rows_expand {
    my ($data, $key_fk, $data_new, $key_new) = @_;

    die "rows_expand: wrong params"
      unless $key_fk && !ref($key_fk) && $key_new && (!ref($key_new) || ref($key_new) eq 'HASH') && ref($data_new);

    # fetch ids from source
    my $vals_fk = {};
    $vals_fk->{$key_fk}->{vals_fk}  = {};
    $vals_fk->{$key_fk}->{key_new}  = $key_new;
    $vals_fk->{$key_fk}->{data_new} = $data_new;
    foreach my $row (@$data) {
        $vals_fk->{$key_fk}->{vals_fk}->{$row->{$key_fk}} = undef if defined $row->{$key_fk};
    }

    # fetch vals
    my $d = $vals_fk->{$key_fk}->{data_new};
    if (ref($d) eq 'HASH') {
        foreach my $v (keys(%{$vals_fk->{$key_fk}->{vals_fk}})) {
            $vals_fk->{$key_fk}->{vals_fk}->{$v} = $d->{$v};
        }
    } elsif (ref($d) eq 'ARRAY' && ref($d->[1]) eq 'ARRAY') {
        my $list     = $d->[1];
        my $key_list = $d->[0];
        die "Wrong format of set - ['pk_field' => \$arr]" unless ref($list) eq 'ARRAY' && $key_list && !ref($key_list);

        foreach my $row_list (@$list) {
            my $row_list_id = $row_list->{$key_list};
            if (exists($vals_fk->{$key_fk}->{vals_fk}->{$row_list_id})) {
                $vals_fk->{$key_fk}->{vals_fk}->{$row_list_id} = $row_list;
            }
        }
    } elsif (ref($d) =~ /Model/ && $d->can('get_all')
        || ref($d) eq 'ARRAY' && ref($d->[1]) =~ /Model/ && $d->[1]->can('get_all'))
    {
        my $m;
        my @fields;
        my @fields_pk;
        if (ref($d) eq 'ARRAY') {
            $m         = $d->[1];
            @fields_pk = ($d->[0]);
        } else {
            $m = $d;
        }
        if ($m->can('get_model_fields')) {
            my $model_fields = $m->get_model_fields;
            @fields = grep({$model_fields->{$_}->{db}} keys(%$model_fields));
            @fields_pk = grep({$model_fields->{$_}->{pk}} keys(%$model_fields)) unless @fields_pk;
        } elsif (ref($m) =~ /Table$/) {
            @fields = $m->field_names();
            @fields_pk = @{$m->{primary_key}} unless @fields_pk;
        }

        my $key_new = $vals_fk->{$key_fk}->{key_new};
        if (ref($key_new) eq 'HASH') {
            my ($k_new, $f) = %$key_new;
            die "Fields must be arrayref - {'new_field' => ['subfield1', 'subfield2']}" unless ref($f) eq 'ARRAY';
            @fields = @$f;
        }
        die "Model has wrong PK (0 or >1)" unless @fields_pk == 1;
        my $key_list = $fields_pk[0];
        push @fields, $key_list;

        my $list = $m->get_all(fields => \@fields, filter => {$key_list => [keys(%{$vals_fk->{$key_fk}->{vals_fk}})]});

        foreach my $row_list (@$list) {
            my $row_list_id = $row_list->{$key_list};
            next unless defined $row_list_id;
            $vals_fk->{$key_fk}->{vals_fk}->{$row_list_id} = $row_list;
        }

    }

    # append data
    foreach my $row (@$data) {
        my $key_new = $vals_fk->{$key_fk}->{key_new};
        my ($k_new) = ref($key_new) eq 'HASH' ? %$key_new : $key_new;
        my $v_new = $vals_fk->{$key_fk}->{vals_fk};
        $row->{$k_new} = $v_new->{$row->{$key_fk}} if defined($row->{$key_fk});
        #Для перехода users на opts
        if (defined $row->{$k_new}) {
            if (ref($row->{$k_new}) eq 'HASH') {
                if (exists $row->{$k_new}->{opts}) {
                    while (my ($k, $v) = each %{$row->{$k_new}->{opts}}) {
                        $row->{$k_new}->{$k} = $v;
                    }
                    delete $row->{$k_new}->{opts};
                }
            }
        }
    }

    return $data;
}

1;
