
=head1 Name

QBit::Class - base class for QBit framework.

=head1 Description

All classes must inherit L<QBit::Class>.

=cut

package QBit::Class;

use qbit;

=head2 abstract_methods

B<Arguments:>

=over

=item

B<@metods> - array of strings, names of abstract methods.

=back

 __PACKAGE__->abstract_methods(qw(method1 method2));
 ....
 $self->method1(); # trow exception with text "Abstract method: method1" if descendant has not override it.

=cut

sub abstract_methods {
    my ($package, @metods) = @_;

    {
        no strict 'refs';
        *{"${package}::$_"} = eval('sub {$package->__abstract__(\'' . $_ . '\')}') foreach @metods;
    }
}

=head2 init

B<No arguments.>

Method called from L</new> before return object.

=cut

sub init { }

=head2 mk_accessors

B<Arguments:>

=over

=item

B<@fields> - array of strings, names of accessors, or hashrefs, pairs of accessor name and field name.

=back

It generate read/write accessors.

 __PACKAGE__->mk_accessors(qw(fieldname fieldname2), {subname => 'fieldname'});
 ....
 $self->fieldname(); # return value
 $self->fieldname('newval'); # set value

=cut

sub mk_accessors {
    my ($self, @fields) = @_;

    my $class = blessed($self) // $self;

    no strict 'refs';

    foreach my $field (@fields) {
        if (ref($field) eq 'HASH') {
            while (my ($s_name, $f_name) = each(%$field)) {
                *{"${class}::$s_name"} = $self->_rw_accessor($f_name);
            }
        } else {
            *{"${class}::$field"} = $self->_rw_accessor($field);
        }
    }
}

=head2 mk_ro_accessors

B<Arguments:>

=over

=item

B<@fields> - array of strings, names of accessors, or hashrefs, pairs of accessor name and field name.

=back

It generate only read accessors.

 __PACKAGE__->mk_ro_accessors(qw(fieldname fieldname2), {subname => 'fieldname'});
 ....
 $self->fieldname(); # return value

=cut

sub mk_ro_accessors {
    my ($self, @fields) = @_;

    my $class = blessed($self) // $self;

    no strict 'refs';

    foreach my $field (@fields) {
        if (ref($field) eq 'HASH') {
            while (my ($s_name, $f_name) = each(%$field)) {
                *{"${class}::$s_name"} = $self->_ro_accessor($f_name);
            }
        } else {
            *{"${class}::$field"} = $self->_ro_accessor($field);
        }
    }
}

=head2 mk_ro_self_or_stash_accessors

B<Arguments:>

=over

=item

B<@fields> - array of strings, names of accessors, or hashrefs, pairs of accessor name and field name.

=back

A very special case for models.
The generated accessor tries to get a field from the given object first and then, if not successful, from it's package stash.

 __PACKAGE__->mk_ro_self_or_stash_accessors(qw(fieldname fieldname2), {subname => 'fieldname'});
 ....
 $self->fieldname(); # return value

=cut

sub mk_ro_self_or_stash_accessors {
    my ($self, @fields) = @_;

    my $class = blessed($self) // $self;

    no strict 'refs';

    foreach my $field (@fields) {
        if (ref($field) eq 'HASH') {
            while (my ($s_name, $f_name) = each(%$field)) {
                *{"${class}::$s_name"} = $self->_ro_self_or_stash_accessor($f_name);
            }
        } else {
            *{"${class}::$field"} = $self->_ro_self_or_stash_accessor($field);
        }
    }
}

=head1 Methods

=head2 new

B<Arguments:>

=over

=item

B<%fields> - fields to store in object.

=back

B<Return value:> blessed object.

=cut

sub new {
    my ($class, %fields) = @_;

    $class = blessed($class) // $class;

    my $self = \%fields;
    bless($self, $class);

    $self->init();

    return $self;
}

sub __abstract__ {
    my ($class, $method) = @_;

    package DB;
    use qbit;

    my @caller = caller(1);

    my $package = ref($DB::args[0]) || $DB::args[0];

    throw gettext('Abstract method "%s" must be defined at %s', $method, $package);
}

sub _ro_accessor {
    my ($self, $field) = @_;

    return sub {$_[0]->{$field}};
}

sub _ro_self_or_stash_accessor {
    my ($self, $field) = @_;

    return sub {$_[0]->{$field} // package_stash(blessed($_[0]) // $_[0])->{$field}};
}

sub _rw_accessor {
    my ($class, $field) = @_;

    return sub {@_ > 1 ? $_[0]->{$field} = $_[1] : $_[0]->{$field}};
}

TRUE;
