package QBit::Cron;

use qbit;

use Fcntl qw(:flock);
use Time::HiRes qw(gettimeofday tv_interval);

use Utils::Logger qw(INFO WARN), {logger => 'Cron',};
use Utils::Cron;

use PiConstants qw($SYSTEM_CRON_USER_ID);

my $MAX_MEMORY_SHARE = 50;
# default value used when LEASEMEM() is without one
my $MAX_MEMORY_PER_CRON_GB = 10;

sub check_rights {
    my ($self, $right_name) = @_;
    return !$right_name || $right_name ne 'edit_protected_pages' || $ENV{FORCE_EDIT_PROTECTED} ? 1 : 0;
}

sub do {
    my ($self, $path, $full_method) = @_;
    local $| = 1;

    my %opts = ();
    unless (defined $full_method) {
        ($path, $full_method, my @optargs) = @ARGV;
        %opts = map {split('=', $_, 2)} @optargs;    # name=value=value will split into name => value=value
    }

    throw gettext("Expecting 'path'")   unless defined $path;
    throw gettext("Expecting 'method'") unless defined $full_method;

    # Add cron path and method to log filename #PI-18283
    my $appenders = Log::Log4perl->appenders();
    foreach my $appender_name (keys %$appenders) {
        if ($appender_name =~ /^CronJson/) {
            my $appender = Log::Log4perl->appender_by_name($appender_name);
            if ($appender->isa('Log::Log4perl::Appender::File')) {
                my $new_path = $appender->filename();
                $new_path =~ s|-PATH|-$path|;
                $new_path =~ s|-METHOD|-$full_method|;
                $appender->file_switch($new_path);
            }
        }
    }

    $self->pre_run();

    my $methods = $self->get_cron_methods();

    my ($method, $instance_number);

    if (exists $methods->{$path}->{$full_method}) {
        $method = $full_method;
    } else {
        ($method, $instance_number) = get_method_and_instance_number($full_method);
        if (!exists $methods->{$path}->{$method}) {
            throw sprintf('Method "%s" with path "%s" does not exists', $full_method, $path);
        }
    }

    $self->set_option(cron_path            => $path);
    $self->set_option(cron_method          => $full_method);
    $self->set_option(cron_instance_method => $method);
    $self->set_option(cron_instance_number => $instance_number);

    Log::Log4perl::MDC->put(cron_path   => $path);
    Log::Log4perl::MDC->put(cron_method => $full_method);
    Log::Log4perl::MDC->put(stage_name  => $ENV{DEPLOY_STAGE_ID} // $self->get_option('stage'));
    Log::Log4perl::MDC->put(unit_name   => $ENV{DEPLOY_UNIT_ID} // $ENV{SYSTEM});
    Log::Log4perl::MDC->put(app_name    => $ENV{SYSTEM});

    my $attrs = $methods->{$path}{$method}{'attrs'};

    if ($attrs->{frequency_limit} && !$opts{'--frequency_limit_skip'}) {
        unless ($self->check_frequency_limit($attrs->{frequency_limit})) {
            $self->report(ignored => TRUE);
            return;
        }
    }

    INFO("[Cron $path - $full_method] START");

    # assume internal user (id = 0)
    $self->set_cur_user({id => $SYSTEM_CRON_USER_ID, login => 'system-cron'});

    my $cron = $methods->{$path}{$method}{'package'}->new(app => $self);

    throw "$path $method does not support instance" if !$attrs->{instances} && $instance_number;
    throw "$path $method instance must be defined"  if $attrs->{instances}  && not defined $instance_number;
    throw "$path $method instance incorrect"        if $attrs->{instances}  && $instance_number !~ /^[0-9]+\z/;
    throw "$path $method instance must be > 0"      if $attrs->{instances}  && $instance_number < 1;
    throw "$path $method instance is too big"       if $attrs->{instances}  && $instance_number > $attrs->{instances};

    my $lock_name = "cron__${path}__${full_method}";

    unless ($self->get_lock($lock_name)) {
        INFO(sprintf("Other %s->%s is running now, I'm exiting", $path, $full_method));
        return;
    }

    try {
        $methods->{$path}{$method}{'sub'}(
            $cron, %opts,
            instance_number => $instance_number,
            instances       => $attrs->{instances}
        );
    }
    catch {
        my ($exception) = @_;

        $self->report();
        $self->release_lock($lock_name);

        throw $exception;
    };
    $self->report(ok => TRUE);
    $self->release_lock($lock_name);
    $self->post_run();
}

sub generate_crond {
    my ($self, %opts) = @_;
    my $cron_pkg = ref($self);

    my $methods = $self->get_cron_methods();

    my $perl5lib = join(':', grep {defined $_} map {$opts{$_}} qw( application_path framework_path  lib_path));

    print "SHELL=/bin/bash\n";
    printf(qq[MAILTO="%s"\n], $opts{'mail_to'}) if exists($opts{'mail_to'});
    printf("LOGSPATH=%s\n", $opts{'logs_path'} // '');
    print qq[CONTENT_TYPE="text/plain; charset=utf-8"\n];
    printf(qq[PERL5LIB="%s:\$PERL5LIB"\n], $perl5lib) if $perl5lib;
    print "\n";

    my $cron_cmd = sprintf(q[perl -M%1$s -e '%1$s->new->do'], $cron_pkg);

    my ($cur_user) = getpwuid($<);
    my $user = $opts{'user'} || $cur_user;

    my $stage = $opts{'stage'} ? lc($opts{'stage'}) : $self->get_option('stage', 'production');

    foreach my $path (sort keys(%$methods)) {
        foreach my $method (sort keys(%{$methods->{$path}})) {

            my $attrs = $methods->{$path}{$method}{'attrs'};

            if ($opts{'deploy_environment'}) {
                next unless $attrs->{'deploy'} // TRUE;

                next if ($opts{'frontend_crons'} xor $attrs->{'frontend'});
            }

            my $method_stages = $attrs->{stage} // {production => TRUE,};
            next unless $method_stages->{$stage};

            my $lease_mem =
              exists $attrs->{leasemem}
              ? (sprintf("--lease FQDN_mem='%d:%d'", $attrs->{leasemem} || $MAX_MEMORY_PER_CRON_GB, $MAX_MEMORY_SHARE))
              : '';

            foreach my $i (1 .. ($attrs->{instances} // 1)) {
                my $full_method = $attrs->{instances} ? $method . '_' . $i : $method;

                my $distributed_lock =
                  $attrs->{lock} && ($stage eq 'production' || $stage eq 'test')
                  ? 'sleep $((RANDOM \% 5)) && /usr/bin/switchman --delay 30 --lockname '
                  . "${stage}___${path}___${full_method} $lease_mem -- "
                  : '';

                my $redirect_to_log      = '';
                my $stdout_log_file_name = $opts{'stdout_log_file_name'};
                if ($stdout_log_file_name) {
                    $stdout_log_file_name =~ s|^/||;
                    $stdout_log_file_name =~ s|\{path\}|$path|;
                    $stdout_log_file_name =~ s|\{method\}|$full_method|;
                    (my $stderr_log_file_name = $stdout_log_file_name) =~ s|\.log$|.err|;
                    # NOTE! Хитрое перенаправление нужно чтобы в почту уходил только STDERR, при этом писались свои логи *.log и *.err
                    $redirect_to_log = sprintf ' 1>>$LOGSPATH/%s 2> >(tee -a $LOGSPATH/%s >&2)', $stdout_log_file_name,
                      $stderr_log_file_name;
                }

                print join("\t",
                    $methods->{$path}{$method}{'time'},
                    ($attrs->{'user'} || $user),
                    "$distributed_lock$cron_cmd $path $full_method$redirect_to_log")
                  . "\n\n";
            }
        }
    }
}    # Cron has full privileges

sub get_cron_methods {
    my ($self) = @_;

    my $methods = {};

    package_merge_isa_data(
        ref($self),
        $methods,
        sub {
            my ($package, $res) = @_;

            my $pkg_methods = package_stash($package)->{'__CRON__'} || {};
            foreach my $path (keys(%$pkg_methods)) {
                foreach my $method (keys(%{$pkg_methods->{$path}})) {
                    $methods->{$path}{$method} = $pkg_methods->{$path}{$method};
                }
            }
        },
        __PACKAGE__
    );

    return $methods;
}

sub get_lock {
    my ($self, $name) = @_;

    $self->{'__LOCKS__'}{$name}{'file'} = "/tmp/${>}_${name}.lock";

    open($self->{'__LOCKS__'}{$name}{'fh'}, '>', $self->{'__LOCKS__'}{$name}{'file'})
      || throw gettext('Cannot create lock file "%s"', $self->{'__LOCKS__'}{$name}{'file'});

    return flock($self->{'__LOCKS__'}{$name}{'fh'}, LOCK_EX | LOCK_NB);
}

sub release_lock {
    my ($self, $name) = @_;

    return unless exists($self->{'__LOCKS__'}{$name});

    flock($self->{'__LOCKS__'}{$name}{'fh'}, LOCK_UN);
    close($self->{'__LOCKS__'}{$name}{'fh'});
    unlink($self->{'__LOCKS__'}{$name}{'file'});
    delete($self->{'__LOCKS__'}{$name});

    return TRUE;
}

sub get_report_cron_name {
    my ($self) = @_;

    my $cron = join('::',
        $self->get_option('cron_path'),
        $self->get_option('cron_instance_method'),
        ($self->get_option('cron_instance_number') // ()));

    return $cron;
}

sub get_report_cron_id {
    my ($self, $cron) = @_;

    $cron //= $self->get_report_cron_name();
    my $id = $self->partner_db->crons->get_all(
        fields => ['id'],
        filter => [name => '=' => \$cron],
    )->[0]{id} // $self->partner_db->crons->add({name => $cron,}, ignore => TRUE,);

    return $id;
}

sub report {
    my ($self, %opts) = @_;

    $self->partner_db->crons_raw_stat->add(
        {
            cron_id  => $self->get_report_cron_id(),
            dt       => trdate(sec => db_time => int($self->get_option('time_of_process'))),
            duration => int($self->get_time()),
            ignored  => ($opts{ignored} ? 1 : 0),
            ok       => ($opts{ok} ? 1 : 0),
        }
    );
}

# Берёт запись о последнем успешном выполнении
# и проверяет не было ли его в текущем периоде
sub check_frequency_limit {
    my ($self, $period) = @_;

    my $last = $self->partner_db->crons_raw_stat->get_all(
        fields => ['dt'],
        filter =>
          [AND => [[cron_id => '=' => \$self->get_report_cron_id()], [ignored => '=' => \0], [ok => '=' => \1],]],
        order_by => [[dt => TRUE]],
        limit    => 1,
    )->[0];

    if ($last) {
        my $dt = get_start_of_period($period, oformat => 'db_time',);
        return $last->{dt} ge $dt ? FALSE : TRUE;
    }

    return TRUE;
}

TRUE;
