package QBit::Cron::Methods;

use qbit;

use base qw(QBit::Application::Part);

my %ATTR_TYPES = (
    ARRAY => {},
    HASH  => {stage => TRUE,},
);

sub MODIFY_CODE_ATTRIBUTES {
    my ($package, $sub, @attrs) = @_;

    my @unknown_attrs = ();

    foreach my $attr (@attrs) {
        if ($attr =~ /^CRON\s*\(\s*'\s*([\d \/*,-]+)\s*'\s*\)$/) {
            $package->_register_method($sub, $1);
        } elsif ($attr =~ /^LOCK$/) {
            $package->_set_method_attr($sub, lock => 1);
        } elsif ($attr =~ /^STAGE\s*\(\s*'\s*(\S+)\s*'\s*\)$/) {
            $package->_set_method_attr($sub, stage => lc($1));
        } elsif ($attr =~ /^INSTANCES\s*\(\s*(\S+)\s*\)$/) {
            $package->_set_method_attr($sub, instances => $1);
        } elsif ($attr =~ /^NOLOCK$/) {
            # This attribute marks subs without LOCK attribute
            # Only a matter of convenience, there is no associated functionality
            # So it's recognized just to keep it from getting into @unknown_attrs
        } elsif ($attr =~ /^TTL\s*\(\s*'\s*(\S+)\s*'\s*\)$/) {
            $package->_set_method_attr($sub, ttl => $1);
        } elsif ($attr =~ /^LEASEMEM\s*\(\s*(\S*)\s*\)$/) {
            $package->_set_method_attr($sub, leasemem => $1);
        } elsif ($attr =~ /^USER\s*\(\s*'\s*([a-z_][a-z0-9_-]{1,31})\s*'\s*\)$/) {
            $package->_set_method_attr($sub, user => $1);
        } elsif ($attr =~ /^FRONTEND$/) {
            $package->_set_method_attr($sub, frontend => 1);
        } elsif ($attr =~ /^DEPLOY\S*\(\s*'\s*(\S+)\s*'\)$/) {
            $package->_set_method_attr($sub, deploy => $1 eq 'FALSE' ? 0 : 1);
        } elsif ($attr =~ /^JUGGLER_TAG\s*\(\s*(\S*)\s*\)$/) {
            $package->_set_method_attr($sub, juggler_tag => $1);
        } elsif ($attr =~ /^FREQUENCY_LIMIT\s*\(\s*'(\S*)'\s*\)$/) {
            $package->_set_method_attr($sub, frequency_limit => lc($1));
        } else {
            push(@unknown_attrs, $attr);
        }
    }

    return @unknown_attrs;
}

sub import {
    my ($package, %opts) = @_;

    $package->SUPER::import(%opts);

    $opts{'path'} ||= '';

    my $app_pkg = caller();
    die gettext('Use only in QBit::Cron and QBit::Application descendant')
      unless $app_pkg->isa('QBit::Cron')
          && $app_pkg->isa('QBit::Application');

    my $pkg_stash = package_stash($package);

    my $app_pkg_stash = package_stash($app_pkg);
    $app_pkg_stash->{'__CRON__'} = {}
      unless exists($app_pkg_stash->{'__CRON__'});

    my $pkg_sym_table = package_sym_table($package);

    foreach my $method (@{$pkg_stash->{'__CRON__'} || []}) {
        my ($name) =
          grep {
                !ref($pkg_sym_table->{$_})
              && defined(&{$pkg_sym_table->{$_}})
              && $method->{'sub'} == \&{$pkg_sym_table->{$_}}
          } keys %$pkg_sym_table;

        $method->{'attrs'} = $pkg_stash->{'__CRON_ATTRS__'}{$method->{'package'}, $method->{'sub'}} || {};

        throw gettext("Cron method \"%s\" is exists in package \"%s\"",
            $name, $app_pkg_stash->{'__CRON__'}{$opts{'path'}}{$name}{'package'})
          if exists($app_pkg_stash->{'__CRON__'}{$opts{'path'}}{$name});
        $app_pkg_stash->{'__CRON__'}{$opts{'path'}}{$name} = $method;
    }

    {
        no strict 'refs';
        foreach my $method (qw(get_option)) {
            *{"${package}::${method}"} = sub {shift->app->$method(@_)};
        }

    }
}

sub _register_method {
    my ($package, $sub, $time) = @_;

    my $pkg_stash = package_stash($package);
    $pkg_stash->{'__CRON__'} = [] unless exists($pkg_stash->{'__CRON__'});

    push(
        @{$pkg_stash->{'__CRON__'}},
        {
            sub     => $sub,
            package => $package,
            time    => $time,
        }
    );
}

sub _set_method_attr {
    my ($package, $sub, $name, $value) = @_;

    my $pkg_stash = package_stash($package);
    $pkg_stash->{'__CRON_ATTRS__'} = {} unless exists($pkg_stash->{'__CRON_ATTRS__'});

    my $method_attr = $pkg_stash->{'__CRON_ATTRS__'}{$package, $sub} //= {};
    if ($ATTR_TYPES{HASH}{$name}) {
        my $attr = $method_attr->{$name} //= {};
        $attr->{$value} = TRUE;
    } elsif ($ATTR_TYPES{ARRAY}{$name}) {
        my $attr = $method_attr->{$name} //= [];
        push @$attr, $value;
    } else {
        $method_attr->{$name} = $value;
    }
}

TRUE;
