
=head1 Name

QBit::Exceptions - qbit exceptions

=cut

package QBit::Exceptions;

=head1 Synopsis

Usage:

 package Exception::Sample;
 use base qw(Exception);

 package Sample;
 use qbit;

 sub ttt {
     throw 'Fatal error';

      # or
      # throw Exception::Sample;

      # or
      # throw Exception::Sample 'Some text describing problem';
 };

 1;

One more sample. Here we are not catching proper exception, and the program
stops. Finally blocks are always executed.

 package Exception::Sample;
 use base qw(Exception);

 package Exception::OtherSample;
 use base qw(Exception);

 package Sample;
 use qbit;

 sub ttt {
  my ($self) = @_;

  try {
   print "try\n";
   throw Exception::Sample 'Exception message';
  }
  catch Exception::OtherSample with {
   print "catch\n";
  }
  finally {
   print "finally\n";
  };

  print "end\n";
 }

 1;

And one more code example. Here we have exception hierarchy. We are throwing
a complex exception but we can catch it with it's parents.

 package Exception::Basic;
 use base qw(Exception);

 package Exception::Complex;
 use base qw(Exception::Basic);

 package Sample;
 use qbit;

 sub ttt {
  my ($self) = @_;

  try {
   print "try\n";
   throw Exception::Complex 'Exception message';
  }
  catch Exception::Basic with {
   print "catch\n";
  }
  finally {
   print "finally\n";
  };

  print "end\n";
 }

 1;

In catch and finally blocks you can access $@ that stores exception object.

=cut

use strict;
use warnings;

use base qw(Exporter);

use Exception;
use Exception::Validation;
use Exception::Validation::BadArguments;
use Exception::Denied;
use Exception::SysDie;

use Scalar::Util qw(blessed);

BEGIN {
    our (@EXPORT, @EXPORT_OK);

    @EXPORT    = qw(try catch with finally throw);
    @EXPORT_OK = @EXPORT;
}

sub catch(&;$) {
    return [Exception => @_];
}

sub finally(&;$) {
    if (defined($_[1])) {die("Expected semicolon after finally block (" . join(", ", (caller())[1, 2]) . ")\n");}
    return ['::FINALLY::' => @_];
}

sub throw($) {
    my ($exception) = @_;
    $exception = Exception->new($exception) unless ref($exception);
    die $exception;
}

sub try(&;$) {
    my ($sub, $catch) = @_;

    eval {$sub->()};

    my $cur_catch = $catch;
    my $find_catch = !defined($catch) || $catch->[0] eq '::FINALLY::';

    my $first_exception = '';
    if ($@) {
        $@ = Exception::SysDie->new($@)
          unless ref($@) && $@->isa('Exception');

        $first_exception = $@;

        while (defined($cur_catch)) {
            last if $cur_catch->[0] eq '::FINALLY::';
            if ($find_catch || $@->isa($cur_catch->[0])) {
                $find_catch = 1;
                if (ref($cur_catch->[1]) eq 'CODE') {
                    eval {$@ = $first_exception; $cur_catch->[1]($first_exception)};

                    if ($@) {
                        $find_catch = 0;

                        $@ = Exception::SysDie->new($@)
                          unless ref($@) && $@->isa('Exception');
                    }

                    last;
                } else {
                    $cur_catch = $cur_catch->[1];
                }
            } else {
                $cur_catch = $cur_catch->[ref($cur_catch->[1]) eq 'CODE' ? 2 : 1];
            }
        }
    }

    $cur_catch = $cur_catch->[ref($cur_catch->[1]) eq 'CODE' ? 2 : 1]
      while ref($cur_catch) && defined($cur_catch) && $cur_catch->[0] ne '::FINALLY::';

    die("Expected semicolon after catch block (" . join(", ", (caller())[1, 2]) . ")\n")
      if defined($cur_catch) && ref($cur_catch) ne 'ARRAY';

    $cur_catch->[1]($first_exception) if defined($cur_catch);

    die $@ if $@ && !$find_catch;

    $@ = $first_exception;
}

sub with(&;$) {
    return @_;
}

=head2

Use the following code in your app if you want to install a $SIG{__DIE__} handler
which will collect a stack trace for all exceptions including Perl runtime errors.

    $SIG{__DIE__} = \&QBit::Exceptions::die_handler;

=cut

sub die_handler {
    die @_ unless defined($^S);    # Undefined $^S means Perl parser errors

    my ($exception) = @_;
    $exception = Exception::SysDie->new($exception) unless ref($exception);

    if (!$^S) {
        # We are not inside an eval (or try)

        # !!! Temporary solution
        # TODO: parse call stack regardless of the exception object type
        # or find that we are in Coro by other methods
        #
        if (defined(blessed($exception)) && $exception->isa('Exception')) {
            if (grep {$_->{'package'} eq 'Coro'} @{$exception->{'callstack'} // []}) {
                # We are inside a Coro routine.
                Coro::terminate($exception);
            }
        }
    }

    # otherwise
    die $exception;
}

1;
