
=head1 Name

QBit::File - Functions to manipulate files.

=cut

package QBit::File;

use strict;
use warnings;
use utf8;
use Fcntl qw(:flock SEEK_END);

require QBit::StringUtils;

use base qw(Exporter);

use Data::Dumper;

BEGIN {
    our (@EXPORT, @EXPORT_OK);

    @EXPORT = qw(
      readfile
      writefile
      );
    @EXPORT_OK = @EXPORT;
}

=head1 Functions

=head2 readfile

B<Arguments:>

=over

=item

B<$filename> - string, file name;

=item

B<%opts> - hash, additional arguments:

=over

=item

B<binary> - boolean, binary ? C<binmode($fh)> : C<binmode($fh, ':utf8')>.

=back

=back

B<Return value:> string, file content.

=cut

sub readfile($;%) {
    my ($filename, %opts) = @_;

    open(my $fh, '<', $filename)
      || die "Cannot open file \"" . ($filename // '<undef>') . "\": " . QBit::StringUtils::fix_utf($!);
    $opts{'binary'} ? binmode($fh) : binmode($fh, ':utf8');
    my $data = join('', <$fh>);
    close($fh);

    return $data;
}

=head2 writefile

B<Arguments:>

=over

=item

B<$filename> - string, file name;

=item

B<$data> - string, file content;

=item

B<%opts> - hash, additional arguments:

=over

=item

B<binary> - boolean, binary ? C<binmode($fh)> : C<binmode($fh, ':utf8')>.

B<append> - boolean

=back

=back

=cut

sub writefile($$;%) {
    my ($filename, $data, %opts) = @_;

    my $mode = $opts{'append'} ? '>>' : '>';

    open(my $fh, $mode, $filename)
      || die "Cannot open file \"" . ($filename // '<undef>') . "\" for write: " . QBit::StringUtils::fix_utf($!);
    $opts{'binary'} ? binmode($fh) : binmode($fh, ':utf8');
    my $locked;
    if ($opts{lock} and $opts{'append'}) {
        flock($fh, LOCK_EX);
        seek($fh, 0, SEEK_END);
        $locked = 1;
    }
    print $fh $data;
    if ($locked) {
        flock($fh, LOCK_UN);
    }
    close($fh);
}

1;
