package QBit::HTTPAPI;

use qbit;

use QBit::WebInterface::Response;

use QBit::HTTPAPI::XML;
use YAML::XS;
use Utils::TSV;
use Utils::Stream::Serializer::XML;
use Utils::Stream::Serializer::JSON;
use Utils::Stream::Serializer::TSV;
use Utils::Logger qw(ERROR);

use Exception::Denied;
use Exception::NotFound;
use Exception::Request::UnknownMethod;
use Exception::TooManyRequests;
use Exception::Validation::BadArguments;
use Exception::Validation;

eval {require Exception::Request::UnknownMethod};

sub request {
    my ($self, $request) = @_;
    return defined($request) ? $self->{'__REQUEST__'} = $request : $self->{'__REQUEST__'};
}

sub response {
    my ($self, $response) = @_;
    return defined($response) ? $self->{'__RESPONSE__'} = $response : $self->{'__RESPONSE__'};
}

our %STREAM_SERIALIZERS = (
    xml  => 'Utils::Stream::Serializer::XML',
    json => 'Utils::Stream::Serializer::JSON',
    tsv  => 'Utils::Stream::Serializer::TSV',
);

our %SERIALIZERS = (
    xml => {
        content_type => 'application/xml; charset=UTF-8',
        sub          => \&QBit::HTTPAPI::XML::pl2xml,
    },
    json => {
        content_type => 'application/json; charset=UTF-8',
        sub          => sub {
            return \to_json($_[0], pretty => $_[1]);
        },
    },
    tsv => {
        content_type => 'text/plain; charset=UTF-8',
        sub          => sub {
            my $result = '';
            if ($_[0]->{'result'} eq 'ok') {
                $result =
                  tsv_with_fields($_[0]->{'data'}{'data'}, $_[0]->{'data'}{'fields'}, %{$_[0]->{'data'}{'options'}});
            } elsif ($_[0]->{'result'} eq 'error') {
                $result = tsv_with_fields([$_[0]], [qw(error_type message)], %{$_[0]->{'data'}{'options'}});
            } else {
                throw gettext('Unknown error type "%s"', $_[0]->{'result'});
            }
            return \$result;
        },
    },
);

our %ACCEPT2TYPE = (
    'application/xml'  => 'xml',
    'text/xml'         => 'xml',
    'application/json' => 'json',
    'text/json'        => 'json',
);

sub build_response {
    my ($self) = @_;

    $self->pre_run();

    throw gettext('No request object') unless $self->request;
    $self->response(QBit::WebInterface::Response->new());

    my $methods = $self->get_methods();
    my ($path, $method, $format) = $self->get_method();

    unless (defined($format)) {
        ($format) = map {s/;.+$//; $ACCEPT2TYPE{$_} || ()} split(',', $self->request->http_header('Accept') || '');
    }
    $format = 'xml' unless defined($format);

    if (exists($methods->{$path}{$method}) && exists($SERIALIZERS{$format})) {
        my $attrs = $methods->{$path}{$method}->{'attrs'};

        my $api = $methods->{$path}{$method}->{'package'}->new(
            app    => $self,
            path   => $path,
            attrs  => $attrs,
            format => $format,
        );

        try {
            my %params;
            foreach my $param (sort keys %{$attrs->{'params'} || {}}) {
                my $properties = $attrs->{'params'}{$param};

                my $value =
                    $properties->{'is_array'}
                  ? $self->request->param_array($param)
                  : $self->request->param($param);

                throw Exception::Validation::BadArguments gettext('Missed required parameter "%s"', $param)
                  if $properties->{'is_required'} && (!defined($value) || ($properties->{'is_array'} && !@$value));

                $params{$param} = $value if defined($value);
            }

            $api->pre_run($method, \%params);

            if (exists($attrs->{'formats'}) && !grep {$format eq $_} @{$attrs->{'formats'}}) {
                throw Exception::Validation gettext('Supported only this formats: %s',
                    join(', ', @{$attrs->{'formats'}}));
            }

            my $ref = $methods->{$path}{$method}{'sub'}($api, %params);

            $self->response->content_type($SERIALIZERS{$format}->{'content_type'});

            if ($attrs->{STREAM}) {
                throw 'Not supported format' unless $STREAM_SERIALIZERS{$format};

                # по умолчанию мы ставим end_marker, только если явно не указано обратное
                # предполагается, что в будущем всегда будет end_marker, и атрибут можно будет убрать
                my $end_marker;
                if ($attrs->{NO_END_MARKER}) {
                    $end_marker = '';
                }

                my $pretty = $self->request->param('pretty');
                my $result = 'ok';

                my $need_topdata = $attrs->{STREAM_TOPDATA} || ('tsv' eq $format);
                my $to_serialize = {
                    data => $need_topdata ? $ref->{data} : $ref,
                    result => \$result,
                };

                weaken($self);
                my $callback;
                if ($format eq 'tsv') {
                    $callback = sub {
                        my ($exception) = @_;
                        $self->exception_dumper->dump_as_html_file($exception);
                        $end_marker = "#ERROR";
                    };
                } else {
                    $callback = sub {
                        my ($exception) = @_;
                        $self->exception_dumper->dump_as_html_file($exception);
                        $result = 'error';
                    };
                }

                $self->set_on_error_callback($to_serialize, $callback);

                my $serializer = $STREAM_SERIALIZERS{$format}->new(
                    data => $to_serialize,
                    (
                        $format eq 'tsv'
                        ? (
                            fields     => $ref->{fields},
                            end_marker => \$end_marker,
                            no_headers => $attrs->{NO_HEADERS},
                          )
                        : (pretty => $pretty,)
                    ),
                );

                $ref = $serializer;
            }

            if (ref($ref) eq 'CODE' or blessed($ref)) {
                $self->response->data($ref);
            } else {
                $self->response->data(
                    $SERIALIZERS{$format}->{'sub'}({result => 'ok', data => $ref}, $self->request->param('pretty')));
            }

            $api->post_run($method, \%params, $self->response->data());
        }
        catch Exception::Validation with {
            $self->response->status(400);
        }
        catch Exception::Denied with {
            $self->response->status(403);
        }
        catch Exception::NotFound with {
            $self->response->status(404);
        }
        catch Exception::TooManyRequests with {
            $self->response->status(429);
        }
        catch {
            my ($e) = @_;

            ERROR($e) if $ENV{FORCE_LOGGER_TO_SCREEN};

            $self->exception_dumper->dump_as_html_file($e);
            $self->response->status(500);
        }
        finally {
            my ($e) = @_;

            if (defined(blessed($e))) {
                $api->on_error($method, $e);
                $self->response->content_type($SERIALIZERS{$format}->{'content_type'});

                my ($type, $message);

                # NOTE! SysDie makes no response->status
                my $status = $self->response->status // '500';

                # hide internal errors
                if ($status =~ /^5/) {
                    $type    = 'Internal';
                    $message = 'Internal Error';
                } else {
                    $type    = ref($e);
                    $message = $e->message();
                }

                $self->response->data(
                    $SERIALIZERS{$format}->{'sub'}({result => 'error', message => $message, error_type => $type},
                        $self->request->param('pretty')));
            }
        };
    } else {
        $self->response->status(404);
    }

    $self->post_run();

    return TRUE;
}

sub get_method {
    my ($self) = @_;

    my $location = $self->get_option('api_location', '/');
    $location = "/$location" unless $location =~ /^\//;
    $location .= '/' unless $location =~ /\/$/;

    return $self->request->uri =~ /^\Q$location\E([^?\/#]+)\/([^?\/#\.]+)(?:\.([a-z]+))?/ ? ($1, $2, $3) : ('', '');
}

sub get_methods {
    my ($self) = @_;

    my $methods = {};

    package_merge_isa_data(
        ref($self) || $self,
        $methods,
        sub {
            my ($package, $res) = @_;

            my $pkg_methods = package_stash($package)->{'__API_METHODS__'} || {};
            foreach my $path (keys(%$pkg_methods)) {
                foreach my $method (keys(%{$pkg_methods->{$path}})) {
                    $methods->{$path}{$method} = $pkg_methods->{$path}{$method};
                }
            }
        },
        __PACKAGE__
    );

    return $methods;
}

sub set_on_error_callback {
    my ($class, $data, $callback) = @_;

    if (blessed($data) and $data->isa('Utils::Stream::DataSource')) {
        $data->on_error($callback);
    } elsif (ref($data) eq 'ARRAR') {
        $class->set_on_error_callback($_, $callback) for (@$data);
    } elsif (ref($data) eq 'HASH') {
        $class->set_on_error_callback($_, $callback) for (values %$data);
    }

    return;
}

TRUE;
