package QBit::HTTPAPI::Method;

use qbit;
use Utils::Safe qw(fail_connect_on_production);

use B qw(svref_2object);

use base qw(QBit::Application::Part);

sub MODIFY_CODE_ATTRIBUTES {
    my ($package, $sub, @attrs) = @_;

    my @unknown_attrs = ();

    foreach my $attr (@attrs) {
        if ($attr =~ /^METHOD$/) {
            $package->_register_method($sub);
        } elsif ($attr =~ /^TITLE(?:\s*\(\s*'\s*([\w\d\s-]+)\s*'\s*\))?$/) {
            $package->_set_method_attr($sub, title => $1);
        } elsif ($attr =~ /^PARAMS\s*\((.+)\)\s*$/) {
            my @vars = split(/\s*,\s*/, $1);
            my %vars;
            foreach (@vars) {
                die "Invalid mehtod param '$_'" unless /^(\!)?([_\w\d]+)(\[\])?/;
                $vars{$2} = {
                    is_required => !!$1,
                    is_array    => !!$3,
                };
            }
            $package->_set_method_attr($sub, params => \%vars);
        } elsif ($attr =~ /^FORMATS\s*\((.+)\)\s*$/) {
            $package->_set_method_attr($sub, formats => [split(/\s*,\s*/, $1)]);
        } elsif ($attr =~ /^STREAM\s*(?:\(\s*([^\)]+)\s*\))?$/) {
            $package->_set_method_attr($sub, STREAM => TRUE);
            if ($1) {
                for my $p (split /\s*,\s*/, $1) {
                    $p =~ s/\W+//g;
                    $package->_set_method_attr($sub, 'STREAM_' . uc($p) => TRUE);
                }
            }
        } else {
            push(@unknown_attrs, $attr);
        }
    }

    return @unknown_attrs;
}

sub import {
    my ($package, %opts) = @_;

    $package->SUPER::import(%opts);

    $opts{'path'} ||= '';

    my $app_pkg = caller();
    throw gettext('Use only in QBit::HTTPAPI and QBit::Application descendant')
      unless $app_pkg->isa('QBit::HTTPAPI')
          && $app_pkg->isa('QBit::Application');

    my $pkg_stash = package_stash($package);

    my $app_pkg_stash = package_stash($app_pkg);
    $app_pkg_stash->{'__API_METHODS__'} = {}
      unless exists($app_pkg_stash->{'__API_METHODS__'});

    my $pkg_sym_table = package_sym_table($package);

    foreach my $method (@{$pkg_stash->{'__API_METHODS__'} || []}) {
        my $name = $method->{'name'};
        $method->{'attrs'} = $pkg_stash->{'__API_METHODS_ATTRS__'}{$method->{'package'}, $method->{'sub'}} || {};

        throw gettext("HTTPAPI method \"%s\" is exists in package \"%s\"",
            $name, $app_pkg_stash->{'__API_METHODS__'}{$opts{'path'}}{$name}{'package'})
          if exists($app_pkg_stash->{'__API_METHODS__'}{$opts{'path'}}{$name});
        $app_pkg_stash->{'__API_METHODS__'}{$opts{'path'}}{$name} = $method;
    }

    {
        no strict 'refs';
        foreach my $method (qw(get_option)) {
            *{"${package}::${method}"} = sub {shift->app->$method(@_)};
        }
    }
}
sub on_error { }
sub post_run { }

sub pre_run {
    my ($self) = @_;

    fail_connect_on_production();
}

sub _register_method {
    my ($package, $sub) = @_;
    my $pkg_stash = package_stash($package);
    $pkg_stash->{'__API_METHODS__'} = [] unless exists($pkg_stash->{'__API_METHODS__'});

    my $cv = svref_2object($sub);
    my $gv = $cv->GV;

    push(
        @{$pkg_stash->{'__API_METHODS__'}},
        {
            sub     => $sub,
            package => $package,
            name    => $gv->NAME,
        }
    );
}

sub _set_method_attr {
    my ($package, $sub, $name, $value) = @_;

    my $pkg_stash = package_stash($package);
    $pkg_stash->{'__API_METHODS_ATTRS__'} = {} unless exists($pkg_stash->{'__API_METHODS_ATTRS__'});

    $pkg_stash->{'__API_METHODS_ATTRS__'}{$package, $sub}{$name} = $value;
}

TRUE;
