
=head1 Name

QBit::Log - Functions to logging

=cut

package QBit::Log;

use strict;
use warnings;
use utf8;

use base qw(Exporter);
use Data::Dumper;

require QBit::Date;
require QBit::StringUtils;

BEGIN {
    our (@EXPORT, @EXPORT_OK);

    @EXPORT = qw(
      l ldump wtf logstr
      );
    @EXPORT_OK = @EXPORT;
}

=head1 Functions

=head2 l

B<Arguments:>

=over

=item

B<@args> - array of strings, messages;

=back

Print joined messsages to STDERR with new line at end.

=cut

sub l {
    print STDERR join(' ', @_) . "\n";
}

=head2 ldump

B<Arguments:>

=over

=item

B<@args> - array, variables;

=back

Print variables dumps with Data::Dumper::Dumper to STDERR. Unicode sequenses will be converted to readable text.

=cut

sub ldump(@) {
    print STDERR _get_pretty_dump(@_);
}

=head2 wtf

    wtf( $some_variable_to_inspect );

Саба для простого дебага вебприложения. В том стучае если приложение
перехватывает STDOUT и STDERR нет возможности использовать print, warn, l и
ldump.

В этом случае можно использовать эту сабу. Саба берет аргумент, и дампит
его в файл ../nginx/wtf.log. Можно сделать `tail -f` на этот файл и
смотреть все то что было передно в wtf.

В том случе если есть файл $ENV{HOME}/.dataprinter и на машине установлен
Data::Printer, то для дампа будет использован он, иначе будет использован
Data::Dumper.

=cut

sub wtf {
    my @data = @_;

    my $lib_path = $INC{'qbit.pm'} =~ /(.+?)qbit\.pm$/ ? $1 : './';

    my $file_name = $lib_path . "../nginx/wtf.log";

    my $dump = _get_pretty_dump(@data);

    open(my $fh, '>>', $file_name) or die "Could not open file '$file_name' $!";
    binmode($fh, ':utf8');
    print $fh $dump;
    close $fh;
}

sub _get_pretty_dump {
    my @data = @_;

    local $Data::Dumper::Varname  = '';
    local $Data::Dumper::Sortkeys = 1;

    my $dump = '';

    # Пример: ldump([ [ $foo, $bar->{baz} ], [ 'foo', 'bar->{baz}' ] ]);
    if (   scalar(@data) == 1
        && ref($data[0]) eq 'ARRAY'
        && scalar(@{$data[0]}) == 2
        && ref($data[0]->[0]) eq 'ARRAY'
        && ref($data[0]->[1]) eq 'ARRAY')
    {
        my ($vals, $names) = @{$data[0]};
        if (@$names > 1) {
            $vals = [{map {$names->[$_] => $vals->[$_]} (0 .. $#$vals)}];
            $names = ['dump'];
        }

        local $Data::Dumper::Indent = 1;
        $dump = Data::Dumper->Dump($vals, $names);

        $dump =~ s/;\s+$//;
        $dump .= sprintf(' at %s line %s', (caller(1))[1, 2]) . "\n\n";
    } else {
        local $Data::Dumper::Indent = 2;
        $dump = Dumper(@data);
    }
    $dump =~ s/\\x\{([a-f0-9]{2,})\}/chr(hex($1))/ge;
    return $dump;
}

sub logstr {

    return
      QBit::Date::curdate(oformat => 'db_time') . ' - '
      . join("\t", map {ref($_) ? QBit::StringUtils::to_json($_, canonical => 1) : $_} @_) . "\n";
}

1;
