
=head1 Name

QBit::StringUtils - Functions to manipulate strings.

=cut

package QBit::StringUtils;

use strict;
use warnings;
use utf8;

use QBit::GetText;
use QBit::Exceptions;

use base qw(Exporter);

use HTML::Entities;
use Template;
use URI::Escape qw(uri_escape_utf8);
use Net::LibIDN qw(idn_to_unicode idn_to_ascii);
use JSON::XS ();
use JSON::PP ();
use POSIX qw(locale_h);

use Exception::Validation::BadArguments;
use Exception::Validation::BadArguments::InvalidJSON;
use Exception::Validation::BadArguments::InvalidPhone;

BEGIN {
    our (@EXPORT, @EXPORT_OK);

    @EXPORT = qw(
      is_place_id_542
      html_encode
      html_decode
      uri_escape
      check_email
      idn_to_unicode
      get_domain_or_error
      get_domain
      get_domain_or_throw
      get_bundle_id
      get_normalized_phone_or_error
      get_normalized_phone
      get_normalized_phone_or_throw
      get_parent_domains
      to_json
      from_json
      from_jsonl
      format_number
      fix_utf
      unset_utf
      sprinttmpl
      trim
      fix_login
      get_url_or_throw);
    @EXPORT_OK = @EXPORT;
}

my $TMPL;

my $RFC822PAT = <<'EOF';
[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\
xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xf
f\n\015()]*)*\)[\040\t]*)*(?:(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\x
ff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015
"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\
xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80
-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*
)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\
\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\
x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x8
0-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n
\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x
80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^
\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040
\t]*)*)*@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([
^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\
\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\
x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-
\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()
]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\
x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\04
0\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\
n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\
015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?!
[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\
]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\
x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\01
5()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*|(?:[^(\040)<>@,;:".
\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]
)|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[^
()<>@,;:".\\\[\]\x80-\xff\000-\010\012-\037]*(?:(?:\([^\\\x80-\xff\n\0
15()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][
^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)|"[^\\\x80-\xff\
n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[^()<>@,;:".\\\[\]\
x80-\xff\000-\010\012-\037]*)*<[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?
:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-
\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:@[\040\t]*
(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015
()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()
]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\0
40)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\
[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\
xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*
)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80
-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x
80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t
]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\
\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])
*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x
80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80
-\xff\n\015()]*)*\)[\040\t]*)*)*(?:,[\040\t]*(?:\([^\\\x80-\xff\n\015(
)]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\
\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*@[\040\t
]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\0
15()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015
()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(
\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|
\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80
-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()
]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x
80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^
\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040
\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".
\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff
])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\
\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x
80-\xff\n\015()]*)*\)[\040\t]*)*)*)*:[\040\t]*(?:\([^\\\x80-\xff\n\015
()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\
\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)?(?:[^
(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-
\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\
n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|
\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))
[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff
\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\x
ff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(
?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\
000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\
xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\x
ff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)
*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*@[\040\t]*(?:\([^\\\x80-\x
ff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-
\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)
*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\
]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]
)[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-
\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\x
ff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(
?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80
-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<
>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x8
0-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:
\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]
*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)
*\)[\040\t]*)*)*>)
EOF
$RFC822PAT =~ s/\n//g;

our $DOMAIN_PART_RE = '[^:\s\/\.!@#$%^&*()\[\]\{\}\?\+;\'"`\\\|<>,]+';

=head2 check_email

B<Arguments:>

=over

=item

B<$email> - string, E-Mail.

=back

B<Return value:> boolean, TRUE if email is valid.

=cut

sub check_email($) {
    my ($email) = @_;

    return 0 unless (0 < index([split('@', ($email // ''))]->[1] // '', '.'));
    return $email =~ /^$RFC822PAT$/;
}

=head2 unset_utf

B<Arguments:>

=over

=item

B<$string> - string.

=back

Convert $string from perl utf8 string if it is with utf8 flag;

B<Return value:> string with utf8 flag.

=cut

sub unset_utf {
    my ($string) = @_;

    utf8::encode($string) if utf8::is_utf8($string);

    return $string;
}

=head2 fix_utf

B<Arguments:>

=over

=item

B<$string> - string.

=back

Convert $string to perl utf8 string if it is without utf8 flag;

B<Return value:> string with utf8 flag.

=cut

sub fix_utf {
    my ($string) = @_;

    utf8::decode($string) unless utf8::is_utf8($string);

    return $string;
}

=head2 format_number

B<Arguments:>

=over

=item

B<$number> - number;

=item

B<%args> - hash, additional arguments:

=over

=item

B<precision>: number, needed precision, if missed then frac will return as is;

B<thousands_sep>: string, thousands separator, default gets from locale;

B<decimal_point>: string, decimal point, default gets from locale.

=back

=back

B<Return value:> string, formatted number.

=cut

sub format_number($%) {
    my ($number, %opts) = @_;

    my $fmt_precision = ($opts{'precision'} || 0) + 1;
    $number = sprintf("%.${fmt_precision}f", $number)
      if $number =~ /^(-?[\d.]+)e([+-]\d+)$/;    # Convert exponent notation

    my $old_locale;
    if (defined($ENV{'LC_ALL'})) {
        $old_locale = setlocale(LC_NUMERIC);
        setlocale(LC_NUMERIC, "$ENV{'LC_ALL'}.utf8");
    }

    my $localeconv = localeconv();
    my $half       = 0.50000000000008;

    foreach my $opt (qw(thousands_sep decimal_point)) {
        unless (defined($opts{$opt})) {
            $opts{$opt} = $localeconv->{$opt};
            utf8::decode($opts{$opt});
        }
    }

    setlocale(LC_NUMERIC, $old_locale) if $old_locale;

    my ($minus, $int, $frac_zero, $frac) =
      $number =~ /^(-?)(\d+)(?:[^\d](0*)(\d*))?$/
      ? ($1, int($2), $3, int($4 || 0))
      : throw Exception::Validation::BadArguments gettext('Invalid number "%s"', $number);

    $frac_zero = '' unless defined($frac_zero);

    if (defined($opts{'precision'})) {
        if ($opts{'precision'} == 0) {
            ++$int if substr($frac, 0, 1) >= 5;
            $frac = '';
        } else {
            $frac = int("0.$frac_zero$frac" * (10**$opts{'precision'}) + $half);
            $frac = substr("$frac_zero$frac", 0, $opts{'precision'});
            $frac = "$opts{'decimal_point'}$frac" . ('0' x ($opts{'precision'} - length($frac)));
        }
    } else {
        $frac = $frac == 0 ? '' : "$opts{'decimal_point'}$frac_zero$frac";
    }

    if (length($int) > 3) {
        $int = reverse($int);
        $int =~ s/(\d\d\d)(?!$)/$1$opts{'thousands_sep'}/g;
        $int = reverse($int);
    }

    return "$minus$int$frac";
}

=head2 from_json

B<Arguments:>

=over

=item

B<$text> - string, JSON.

=back

B<Return value:> scalar, perl structure from JSON.

=cut

sub from_json($;%) {
    my ($text, %opts) = @_;

    my $original_text = $text;
    my $json_object = $opts{'use_pp'} ? 'JSON::PP' : 'JSON::XS';

    utf8::encode($text) if defined($text) and not $opts{keep_utf};
    my $result;
    eval {$result = $json_object->new->utf8->allow_nonref->decode($text);};

    if (!$@) {
        return $result;
    } else {
        my ($error) = ($@ =~ m'(.+) at /');
        $error //= $@;
        throw Exception::Validation::BadArguments::InvalidJSON gettext(
            "Error in from_json: %s\n" . "Input:\n" . "'%s'\n", $error, $original_text // 'undef');
    }
}

sub from_jsonl($) {
    my ($text) = @_;

    my $original_text = $text;

    utf8::encode($text) if defined($text);
    my @result;
    eval {@result = JSON::XS->new->utf8->allow_nonref->incr_parse($text);};

    if (!$@) {
        return \@result;
    } else {
        my ($error) = ($@ =~ m'(.+) at /');
        $error //= $@;
        throw Exception::Validation::BadArguments::InvalidJSON gettext(
            "Error in from_json: %s\n" . "Input:\n" . "'%s'\n", $error, $original_text // 'undef');
    }
}

=head2 get_domain_or_error

B<Arguments:>

=over

=item

B<$url> - string, URL;

=item

B<%opts> - additional arguments:

=over

=item

B<ascii> - boolean, convert unicode chars to ascii;

=item

B<www> - boolean, save 'www.'.

B<with_port> - boolean, save host port

=back

=back

B<Return value:> hash { ok => 'some.domain' } if domain is valid, else hash { error => 'message' }

=cut

sub get_domain_or_error {
    my ($url, %opts) = @_;
    my $original_url = $url;
    $url //= '';

    my $www          = $opts{'www'}    ? ''  : '(?:www\.)?';
    my $https        = $opts{'widget'} ? ''  : '(?:https?:\/\/)?';
    my $domain_quant = $opts{'widget'} ? '+' : '*';
    my $params       = $opts{'widget'} ? '$' : '\.?(:\d{2,5})?($|\/|\?)';

    $url = lc($url);
    $url =~ s/(^\s+)|(\s+$)//g;

    if ($url =~ /^$https$www((?:$DOMAIN_PART_RE\.)$domain_quant$DOMAIN_PART_RE)$params/) {
        my $res = $opts{'ascii'} ? idn_to_ascii($1, 'utf-8') : idn_to_unicode($1, 'utf-8');
        if (defined($res)) {
            $res .= $2 if $opts{'with_port'} && $2;
            utf8::decode($res);
        }
        return {ok => $res};
    } else {
        return {error => gettext('Invalid domain: "%s"', $original_url // 'UNDEF')};
    }
}

sub get_domain {
    my ($url, %opts) = @_;
    return get_domain_or_error($url, %opts)->{ok};
}

=head2 get_domain_or_throw

То же самое что и get_domain, но бросает исключение Exception::Validation::BadArguments
если домен не является валидным.

=cut

sub get_domain_or_throw {
    my ($url, %opts) = @_;

    my $domain = get_domain_or_error($url, %opts);

    if (exists $domain->{ok}) {
        return $domain->{ok};
    } else {
        throw Exception::Validation::BadArguments $domain->{error};
    }
}

=head2 get_bundle_id

B<Arguments:>

=over

=item

B<$bundle_id> - string

=item

B<$type> - type of application

=over

=item

B<1> - Android;

=item

B<2> - iOS.

=back

=back

B<Return value:> bundle_id if it is valid, undef otherwise

=cut

sub get_bundle_id {
    my ($name, $type) = @_;

    my @parts = split(/\./, $name, -1);
    my (@wrong_parts, $name_reserved);

    # Основное описание алгоритма проверки обсуждалось здесь
    # https://st.yandex-team.ru/PI-6550#57063778e4b00f9e870bcc76
    #
    # Итоговые проверки должны быть такие:
    # Android
    # — может содержать латинские буквы (a-z, A-Z), цифры (0-9), нижнее подчеркивание (_), точку (.)
    # — в начале быть буквы (проверяется каждая часть, отделенная точкой)
    # — bundle не должен содержать  com.example
    # iOS
    # — может содержать латинские буквы (a-z, A-Z), цифры (0-9), дефис (-), точку (.)
    # — в начале быть буквы (проверяется каждая часть, отделенная точкой)
    # — в конце буквы или цифры (проверяется каждая часть, отделенная точкой)
    # — максимум 63 символа
    # Проверку на домен в обратном порядке не делаем. Это рекомендация от Apple и Google, но не обязательное условие. Теоретически может быть  bundle ID: lamoda .

    # После выявления ошибки валидации для допустимого iOS bundle_id com.200labs.Lenta
    # https://st.yandex-team.ru/PI-7088#1468241075000
    # Добавилось условие: Может содержать цифры в средней части и перед буквами, и после

    # Дополнительно: https://st.yandex-team.ru/PI-16843 https://st.yandex-team.ru/PI-17149
    # Для iOS:
    #   * первая часть не должна состоять только из цифр или попадать под регулярку id\d+

    # NOTE! см примеры в тесте t/lib/qbit/StringUtils/get_bundle_id.t
    if ($type == 1) {
        # Android.
        # A part starts with a letter, contains letters, digits and underscores

        my $first_part = shift @parts;

        @wrong_parts = grep {$_ !~ /^[a-z][a-z0-9_]*\z/i} $first_part, @parts;

        push(@wrong_parts, $name) if !@wrong_parts && !@parts;

        $name_reserved = $name =~ /^com\.example(?:\.|\z)/;
    } elsif ($type == 2) {
        # iOS.
        # A part is like in DNS (RFC 1035)
        # Starts with a letter, ends with a letter or a digit; with letters, digits and hyphens in between
        # Плюс дополнительное условие 'Может содержать цифры в средней части и перед буквами, и после' - https://st.yandex-team.ru/PI-7088#1468241075000

        my $first_part = shift @parts;

        @wrong_parts = (
            (grep {$_ !~ /^[a-z0-9][a-z0-9-]*\z/i || /^(?:id)?\d+\z/i} $first_part // ()),
            (grep {$_ !~ /^[a-z0-9-]+\z/i} @parts),
        );

        $name_reserved = $name =~ /^(?:public|dyn)(?:\.|\z)/;
    } else {
        throw 'Incorrect usage';
    }

    return ($name =~ /^\s*\z/ || @wrong_parts || $name_reserved) ? undef : $name;
}

=head2 get_normalized_phone_or_error

B<Arguments:>

=over

=item

B<$phone> - string, phone;

=back

B<Return value:> normalized phone if it is valid, exception otherwise.

=cut

sub get_normalized_phone_or_error {
    my ($phone) = @_;

    return {error => gettext("Phone is not defined")} unless defined($phone);

    for ($phone) {
        s/^\s+//;
        s/\s+$//;
    }

    return {error => gettext('Invalid phone: "%s"', $phone)}
      unless $phone =~ /^[0-9() +-]+$/;

    for ($phone) {
        s/\s*-\s*/-/g;
        s/\s*\(/ (/g;
        s/\)\s*/) /g;
    }

    return {ok => $phone};
}

sub get_normalized_phone {
    my ($phone) = @_;
    return get_normalized_phone_or_error($phone)->{ok};
}

sub get_normalized_phone_or_throw {
    my ($phone) = @_;

    my $res = get_normalized_phone_or_error($phone);

    if (exists $res->{ok}) {
        return $res->{ok};
    } else {
        throw Exception::Validation::BadArguments $res->{error};
    }
}

=head2 html_decode

B<Arguments:>

=over

=item

B<$str> - string.

=back

B<Return value:> string with decoded HTML entities.

=cut

sub html_decode($) {
    return defined($_[0]) ? decode_entities($_[0]) : '';
}

=head1 Functions

=head2 html_encode

B<Arguments:>

=over

=item

B<$str> - string.

=back

B<Return value:> string with encoded HTML entities.

=cut

sub html_encode($) {
    return defined($_[0]) ? encode_entities($_[0]) : '';
}

sub sprinttmpl {
    my ($tmpl, %data) = @_;

    my $out = '';

    $TMPL //= Template->new(
        INCLUDE_PATH => [],
        COMPILE_EXT  => '.ttc',
        COMPILE_DIR  => '/tmp/tt_cache-' . $<,
        EVAL_PERL    => 1,
        MINIMIZE     => 1,
        PRE_CHOMP    => 2,
        POST_CHOMP   => 2,
        TRIM         => 2,
        RECURSION    => 1,
    ) || throw $Template::ERROR;

    $TMPL->process(\$tmpl, \%data, \$out) || throw $TMPL->error();

    return $out;
}

=head2 to_json

B<Arguments:>

=over

=item

B<$data> - scalar.

=back

B<Return value:> string, C<$data> as JSON.

=cut

sub to_json($;%) {
    my ($data, %opts) = @_;

    my $res;

    my $json_object = $opts{'use_pp'} ? 'JSON::PP' : 'JSON::XS';

    my $json = $json_object->new->utf8->allow_nonref;
    if ($opts{'canonical'}) {
        $json->canonical;
    } elsif ($opts{'pretty'}) {
        $json->pretty->canonical;
    }

    if ($opts{'blessed'}) {
        $json->convert_blessed;
    }

    $res = $json->encode($data);

    utf8::decode($res);

    return $res;
}

=head2 trim

B<Arguments:>

=over

=item

B<$string> - string.

=back

trim spaces from begin and end string

B<Return value:> string.

=cut

sub trim($) {
    my ($str) = @_;

    $str =~ s/^\s*|\s*$//g;

    return $str;
}

=head2 uri_escape

B<Arguments:>

=over

=item

B<$str> - string.

=back

B<Return value:> string with escaped URI entities.

=cut

sub uri_escape($) {
    return uri_escape_utf8($_[0]);
}

=head2 fix_login

Саба "нормализует" логин. Т.е. переводит логин в нижний регистр и
заменяет все точки на минусы (с точки зрения паспорта минус и точка - это одно и то же).

    my $login = fix_login('yndx.SOME-USER'); # yndx-some-user

=cut

sub fix_login($) {
    my ($login) = @_;

    $login =~ s/\./-/g;

    return lc($login);
}

=head2 is_place_id_542

=cut

sub is_place_id_542 {
    my ($place_id) = @_;

    # должно быть так, но ломается компиляция
    # use PiConstants qw($DIRECT_PLACE_ID $DIRECT_PLACE_ID_E);
    # return $place_id == $DIRECT_PLACE_ID || $place_id == $DIRECT_PLACE_ID_E;

    # поэтому явно
    return $place_id == 542 || $place_id == 1542;
}

sub get_url_or_error {
    my ($url) = @_;

    my $original_url = $url;
    $url //= '';
    $url =~ s/(^\s+)|(\s+$)//g;

    my ($scheme, $authority, $path, $query, $fragment) =
      $url =~ m|(?:([^:/?#]+):)?(?://([^/?#]*))?([^?#]*)(?:\?([^#]*))?(?:#(.*))?|;

    if ($scheme && $authority && $path) {
        return {ok => $url};
    } else {
        return {error => gettext('Invalid URL: "%s"', $original_url // 'UNDEF')};
    }
}

sub get_url_or_throw {
    my ($url) = @_;

    my $r = get_url_or_error($url);

    if (exists $r->{ok}) {
        return $r->{ok};
    } else {
        throw Exception::Validation::BadArguments $r->{error};
    }
}

sub get_parent_domains {
    my ($domain) = @_;

    my @levels = split '\.', $domain;
    return [$domain] if scalar @levels == 2;

    my @output = ();

    while (@levels >= 2) {
        push @output, join('.', @levels);
        shift @levels;
    }
    return \@output;
}

1;
