package QBit::Validator;

use qbit;

use base qw(QBit::Class);

use base qw(Exporter);

use Utils::Logger qw/FATAL/;

BEGIN {
    our (@EXPORT, @EXPORT_OK);

    @EXPORT = qw(
      SKIP
      OPT
      EXTRA
      SCALAR
      HASH
      ARRAY
      );
    @EXPORT_OK = @EXPORT;
}

use constant SKIP   => (skip     => TRUE);
use constant OPT    => (optional => TRUE);
use constant EXTRA  => (extra    => TRUE);
use constant SCALAR => (type     => 'scalar');
use constant HASH   => (type     => 'hash');
use constant ARRAY  => (type     => 'array');

use Exception::Validator;
use Exception::Validator::Fields;
use Exception::Validator::Errors;
use Exception::SysDie;

__PACKAGE__->mk_ro_accessors(qw(data app));

__PACKAGE__->mk_accessors(qw(template));

my %available_fields = map {$_ => TRUE} qw(data template app throw throw_internal_error pre_run stash);

sub checked {
    my ($self, $field) = @_;

    return exists($self->{'__CHECK_FIELDS__'}{$self->_get_key($field)});
}

sub get_all_errors {
    my ($self) = @_;

    my $error = '';

    $error .= join("\n", map {@{$_->{'msgs'}}} $self->get_fields_with_error());

    return $error;
}

sub get_error {
    my ($self, $field) = @_;

    my $key = $self->_get_key($field);

    my $error = '';
    foreach ($self->get_fields_with_error()) {
        $error = join("\n", @{$_->{'msgs'}}) if $key eq $self->_get_key($_->{'path'});
    }

    return $error;
}

sub get_fields_with_error {
    my ($self) = @_;

    return map  {$self->{'__CHECK_FIELDS__'}{$_}{'error'}}
      sort grep {$self->{'__CHECK_FIELDS__'}{$_}{'error'}} keys(%{$self->{'__CHECK_FIELDS__'}});
}

sub get_stash {
    my ($self, $path) = @_;

    return $self->{'__STASH__'}{$self->_get_key($path)};
}

sub has_error {
    my ($self, $field, %opts) = @_;

    if ($opts{'in_depth'}) {
        my $key = $self->_get_key($field);

        return !!grep {$_ =~ /^$key/ && $self->{'__CHECK_FIELDS__'}{$_}{'error'}} keys(%{$self->{'__CHECK_FIELDS__'}});
    } else {
        return exists($self->{'__CHECK_FIELDS__'}{$self->_get_key($field)}{'error'});
    }
}

sub has_errors {
    my ($self) = @_;

    return !!$self->get_fields_with_error();
}

sub init {
    my ($self) = @_;

    foreach (qw(data template)) {
        throw Exception::Validator gettext('Expected "%s"', $_) unless exists($self->{$_});
    }

    my @bad_fields = grep {!$available_fields{$_}} keys(%{$self});
    throw Exception::Validator gettext('Unknown options: %s', join(', ', @bad_fields))
      if @bad_fields;

    $self->{'__STASH__'} = $self->{'stash'} // {};

    if (exists($self->{'pre_run'})) {
        throw Exception::Validator gettext('Option "pre_run" must be code')
          if !defined($self->{'pre_run'}) || ref($self->{'pre_run'}) ne 'CODE';

        $self->{'pre_run'}($self);
    }

    $self->{'__CHECK_FIELDS__'} = {};

    my $data     = $self->data;
    my $template = $self->template;

    $self->_validation($data, $template);

    if ($self->has_errors) {
        $self->throw_internal_exception() if $self->{'throw_internal_error'};

        $self->throw_exception() if $self->{'throw'};
    }
}

sub set_stash {
    my ($self, $path, $value) = @_;

    $self->{'__STASH__'}{$self->_get_key($path)} = $value;
}

sub super_check {
    my ($self, $type_name, @params) = @_;

    my %types = map {$_ => TRUE} $self->_get_all_types($params[2]->{'type'});

    throw Exception::Validator gettext('You can not use sub "check" of type "%s" for this template', $type_name)
      unless $types{$type_name};

    my $type = $self->_get_type_by_name($type_name);

    throw Exception::Validator gettext('Do not exists sub "check" for type "%s"', $type_name)
      unless $type->can('get_template');

    my $type_template = $type->get_template();

    throw Exception::Validator gettext('Do not exists sub "check" for type "%s"', $type_name)
      unless exists($type_template->{'check'});

    throw Exception::Validator gettext('Option "check" must be code')
      if !defined($type_template->{'check'}) || ref($type_template->{'check'}) ne 'CODE';

    $type_template->{'check'}(@params);
}

sub throw_exception {
    my ($self) = @_;

    throw Exception::Validator::Errors [map {{name => $_->{'path'}, messages => $_->{'msgs'}}}
          $self->get_fields_with_error()];
}

sub throw_internal_exception {
    my ($self) = @_;

    throw Exception::Validator join("\n",
        map {(@{$_->{'path'}} ? '[' . join(', ', @{$_->{'path'}}) . ']: ' : '') . join(', ', @{$_->{'msgs'}})}
          $self->get_fields_with_error());
}

sub _add_error {
    my ($self, $template, $error, $field, %opts) = @_;

    my $key = $self->_get_key($field);

    if ($opts{'check_error'}) {
        $self->{'__CHECK_FIELDS__'}{$key}{'error'} = {
            msgs => [$error],
            path => $field // []
        };
    } elsif ($self->has_error($field)) {
        push(@{$self->{'__CHECK_FIELDS__'}{$key}{'error'}{'msgs'}}, $error)
          unless exists($template->{'msg'});
    } else {
        my $msg = exists($template->{'msg'}) ? $template->{'msg'} : $error;

        $self->{'__CHECK_FIELDS__'}{$key}{'error'} = {
            msgs => [ref($msg) eq 'CODE' ? $msg->() : $msg],
            path => $field // []
        };
    }

    delete($self->{'__CHECK_FIELDS__'}{$key}{'ok'}) if exists($self->{'__CHECK_FIELDS__'}{$key}{'ok'});
}

sub _add_ok {
    my ($self, $field) = @_;

    return if $self->checked($field) && $self->has_error($field);

    $self->{'__CHECK_FIELDS__'}{$self->_get_key($field)}{'ok'} = TRUE;
}

sub _get_all_options_by_types {
    my ($self, $types) = @_;

    my @types_name = $self->_get_all_types($types);

    my %uniq_options = ();

    foreach my $type_name (@types_name) {
        my $type = $self->_get_type_by_name($type_name);

        $uniq_options{$_} = TRUE foreach $type->get_all_options_name();
    }

    return [sort keys(%uniq_options)];
}

sub _get_all_types {
    my ($self, $types) = @_;

    $types //= ['scalar'];
    $types = [$types] unless ref($types) eq 'ARRAY';

    my %uniq_types = map {$_ => TRUE} @$types;

    foreach my $type_name (@$types) {
        my $type = $self->_get_type_by_name($type_name);

        if ($type->can('get_template')) {
            my $type_template = $type->get_template();

            $uniq_types{$_} = TRUE foreach $self->_get_all_types($type_template->{'type'});
        }
    }

    return sort(keys(%uniq_types));
}

sub _get_key {
    my ($self, $path_field) = @_;

    $path_field //= [];

    $path_field = [$path_field] unless ref($path_field) eq 'ARRAY';

    return join(' => ', @$path_field);
}

sub _get_type_by_name {
    my ($self, $type_name) = @_;

    unless (exists($self->{'__TYPES__'}{$type_name})) {
        my $type_class = 'QBit::Validator::Type::' . $type_name;
        my $type_fn    = "$type_class.pm";
        $type_fn =~ s/::/\//g;

        try {
            require $type_fn;
        }
        catch {
            FATAL shift->message();
            throw Exception::Validator gettext('Unknown type "%s"', $type_name);
        };

        $self->{'__TYPES__'}{$type_name} = $type_class->new();
    }

    return $self->{'__TYPES__'}{$type_name};
}

sub _validation {
    my ($self, $data, $template, $no_check_options, @path_field) = @_;

    throw Exception::Validator gettext('Key "template" must be HASH')
      if !defined($template) || ref($template) ne 'HASH';

    $template->{'type'} //= ['scalar'];

    $template->{'type'} = [$template->{'type'}] unless ref($template->{'type'}) eq 'ARRAY';

    my $already_check;
    foreach my $type_name (@{$template->{'type'}}) {
        my $type = $self->_get_type_by_name($type_name);

        if ($type->can('get_template')) {
            my $type_template = $type->get_template();

            my $new_template = {
                (
                    map {$_ => $type_template->{$_}}
                    grep {$_ eq 'type' || !exists($template->{$_})} keys(%$type_template)
                ),
                map {$_ => $template->{$_}} grep {$_ ne 'type' && $_ ne 'check'} keys(%$template)
            };

            $self->_validation($data, $new_template, TRUE, @path_field);
        }

        unless ($self->has_error(\@path_field)) {
            $type->check_options($self, $data, $template, @path_field);
            last if $self->has_error(\@path_field);
        } else {
            last;
        }

        if (exists($template->{'check'}) && !$already_check && !$self->has_error(\@path_field, in_depth => TRUE)) {
            $already_check = TRUE;

            throw Exception::Validator gettext('Option "check" must be code')
              if !defined($template->{'check'}) || ref($template->{'check'}) ne 'CODE';

            next if !defined($data) && $template->{'optional'};

            my $error;
            my $error_msg;
            try {
                $template->{'check'}($self, $data, $template, @path_field);
            }
            catch Exception::Validator catch Exception::Validator::Fields catch Exception::Validation::BadArguments
              with {
                $error     = TRUE;
                $error_msg = shift->message;
            }
            catch Exception::SysDie with {
                FATAL shift->message;
                throw 'Internal error';
            }
            catch {
                my ($exception) = @_;
                FATAL $exception;
                $error     = TRUE;
                $error_msg = gettext('Internal error');
            };

            if ($error) {
                $self->_add_error($template, $error_msg, \@path_field, check_error => TRUE);

                last;
            }
        }
    }

    unless ($no_check_options) {
        my $all_options = $self->_get_all_options_by_types($template->{'type'});

        my $diff = arrays_difference([sort keys(%$template)], $all_options);

        throw Exception::Validator gettext('Unknown options: %s', join(', ', @$diff)) if @$diff;
    }
}

sub add_errors_from_other_qv {
    my ($self, $path, $errors) = @_;

    foreach my $error (@$errors) {
        my @path = (@$path, @{$error->{'path'}});

        foreach (@{$error->{'msgs'}}) {
            $self->_add_error(undef, $_, \@path);
        }
    }
}

TRUE;
