package QBit::Validator::Type::bk_data::block;

use qbit;

use Exception::Validator;
use Exception::Validator::Fields;

use PiConstants qw/
  $DSP_MEDIA_TYPE_ID
  $DSP_VIDEO_TYPE_ID
  $DSP_MOBILE_TYPE_ID
  $VIDEO_BLOCK_TYPES
  $IGNORE_EMPTY_SIZES_FOR_INTERSTITIAL
  /;

my $_chi_kv;
my $_chi_key = 'bk_data_block_validator';

my @fingerprint_prefix = ('Validator', 'bk_data', 'block');

sub validate {
    my ($qv, $bk_data) = @_;

    my $page_id   = $qv->data->{$qv->app->get_page_id_field_name};
    my $public_id = $qv->app->public_id($qv->data);

    my $params = [$qv, $bk_data, $page_id, $public_id];

    _check_rtb_has_designs(@$params);
    _check_all_design_exists(@$params);
    _check_rtb_video(@$params);
    _check_dsp_types(@$params);
    _check_video_dsp_types(@$params);
    _check_media_size(@$params);
    _check_interstitial_adtypesets(@$params);
    _check_interstitial_sizes(@$params);
    _check_direct_limit(@$params);
}

sub _check_rtb_has_designs {
    my ($qv, $bk_data) = @_;

    if (   $qv->app->DOES('Application::Model::Role::Has::Block::DesignSettings')
        || $qv->app->accessor =~ /(?:direct)|(?:premium)/)
    {
        my $design_option_name = $qv->app->design_field();

        if (not exists $bk_data->{$design_option_name}) {
            throw Exception::Validator::Fields gettext('Missing key "%s"', $design_option_name),
              sentry => {fingerprint => [@fingerprint_prefix, 'Missing key']};
        }
    }

    return TRUE;
}

sub _check_all_design_exists {
    my ($qv, $bk_data) = @_;

    # Проверка на то, что в кастомном транспорте нет id дизайнов, не принадлежащих
    # текущему блоку. Возможно срабатывание проверки в случае, когда у блока
    # удаляют дизайн и не дождавшись обновления bk_data переводят в годмод.
    if (   $qv->app->DOES('Application::Model::Role::Has::Block::DesignTemplates')
        && ref($bk_data) eq 'HASH'
        && $qv->data->{is_custom_bk_data})
    {
        my @bk_design_ids = $qv->app->get_bk_data_design_ids($bk_data);

        my $valid_design_ids =
          exists $qv->data->{design_templates}
          ? {map {$_->{id} => $_} grep {defined $_->{id}} @{$qv->data->{design_templates}}}
          : {};
        my @bad_designs = grep {!$valid_design_ids->{$_}} @bk_design_ids;

        if (scalar @bad_designs) {
            throw Exception::Validator::Fields gettext("Bad design id's: %s", join(',', sort @bad_designs)),
              sentry => {fingerprint => [@fingerprint_prefix, 'Bad design id\'s']};
        }
    }

    return TRUE;
}

sub _check_rtb_video {
    my ($qv, $bk_data) = @_;

    my $data = clone($bk_data);

    if (
        exists $data->{RtbVideo}
        && (ref($data->{RtbVideo}) ne 'HASH'
            || !%{$data->{RtbVideo}})
       )
    {
        throw Exception::Validator::Fields gettext("Wrong RtbVideo field"),
          sentry => {fingerprint => [@fingerprint_prefix, 'Wrong RtbVideo field']};
    }

    for my $category_id (keys %{$data->{RtbVideo}->{Categories}}) {
        my $category = $data->{RtbVideo}->{Categories}->{$category_id};
        if (!defined($category->{Archive})
            || $category->{Archive} !~ /^[01]{1}$/)
        {
            throw Exception::Validator::Fields gettext("RtbVideo.Categories.%s.Archive not exists or wrong",
                $category_id),
              sentry => {fingerprint => [@fingerprint_prefix, 'RtbVideo.Categories.Archive not exists or wrong']};
        }

        if (!defined($category->{Name})
            || $category->{Name} eq '')
        {
            throw Exception::Validator::Fields gettext("RtbVideo.Categories.%s.Name not exists or empty", $category_id),
              sentry => {fingerprint => [@fingerprint_prefix, 'RtbVideo.Categories.Name not exists or wrong']};
        }
    }

    return TRUE;
}

sub _check_dsp_types {
    my ($qv, $bk_data, $page_id, $public_id) = @_;

    if (defined $bk_data->{DSPType}) {
        if (defined $bk_data->{DSPInfo}) {
            my %bk_data_dsps = map {$_->{DSPID} => 1} @{$bk_data->{DSPInfo}};
            my $all_dsps = _get_all_dsps($qv, [keys %bk_data_dsps]);

            my @unknown_dsp_ids;
            map {push(@unknown_dsp_ids, $_) unless exists $all_dsps->{$_}} keys %bk_data_dsps;

            if (scalar @unknown_dsp_ids) {
                throw Exception::Validator::Fields gettext("DSP with DSPID = %s doesn't exist in DB",
                    join(',', sort @unknown_dsp_ids)),
                  sentry => {fingerprint => [@fingerprint_prefix, 'DSP doesn\'t exist in DB']};
            }

            for my $dsp_id (keys %bk_data_dsps) {
                unless (grep {$bk_data->{DSPType} & $_} @{$all_dsps->{$dsp_id}->{types}}) {
                    throw Exception::Validator::Fields gettext("Block %s has DSPType = %s, DSP %s has DSPType = %s",
                        $public_id, $bk_data->{DSPType}, $dsp_id, join(',', sort @{$all_dsps->{$dsp_id}->{types}})),
                      sentry => {fingerprint => [@fingerprint_prefix, 'Block\'s DSP has incorrect DSPType']};
                }
            }
        }
    }
    return TRUE;
}

sub _check_video_dsp_types {
    my ($qv, $bk_data, $page_id, $public_id) = @_;

    if (   defined $bk_data->{DSPType}
        && $bk_data->{DSPType} == $DSP_VIDEO_TYPE_ID
        && $bk_data->{Video}
        && !in_array($bk_data->{BlockModel}, ['indoor_block', 'outdoor_block']))
    {
        unless (defined($bk_data->{RtbVideo})) {
            throw Exception::Validator::Fields gettext(
                "Block %s has Video field, but current page doesnt have RtbVideo field", $public_id),
              sentry => {fingerprint =>
                  [@fingerprint_prefix, 'Block has Video field, but current page doesnt have RtbVideo field']};
        }

        unless (grep {$bk_data->{Video}->{Type} eq $VIDEO_BLOCK_TYPES->{$_}->{bk}} keys %$VIDEO_BLOCK_TYPES) {
            throw Exception::Validator::Fields gettext("Wrong block.Video subfield Type");
        }
    }

    return TRUE;
}

sub _check_media_size {
    my ($qv, $bk_data, $page_id, $public_id) = @_;

    if (   defined $bk_data->{DSPType}
        && $bk_data->{DSPType} == $DSP_MEDIA_TYPE_ID
        && !$bk_data->{AdBlockBlock}
        && (!$bk_data->{Width} || !$bk_data->{Height}))
    {
        throw Exception::Validator::Fields gettext("Width and Height of PageID=%s ImpID=%s undefined or null", $page_id,
            $public_id),
          sentry => {fingerprint => [@fingerprint_prefix, 'Width and Height undefined or null']};
    }

    return TRUE;
}

sub _check_interstitial_adtypesets {
    my ($qv, $bk_data, $page_id, $public_id) = @_;

    my @creatives = (qw/text media media-performance video video-non-skippable video-performance video-motion audio/);

    if (   defined $bk_data->{AdTypeSet}
        && !$bk_data->{InterstitialBlock}
        && !grep {defined $bk_data->{AdTypeSet}->{$_}} @creatives)
    {
        throw Exception::Validator::Fields gettext("AdTypeSet %s of PageID=%s ImpID=%s incorrect", $public_id, $page_id,
            $public_id),
          sentry => {fingerprint => [@fingerprint_prefix, 'AdTypeSet incorrect']};
    }

    if (
        $bk_data->{InterstitialBlock}
        && (  !defined($bk_data->{AdTypeSet})
            || ref($bk_data->{AdTypeSet}) ne 'HASH')
       )
    {
        throw Exception::Validator::Fields gettext(
            "AdTypeSet %s of PageID=%s ImpID=%s (interstitial) incorrect or not found",
            $public_id, $page_id, $public_id
          ),
          sentry => {fingerprint => [@fingerprint_prefix, 'AdTypeSet (interstitial) incorrect or not found']};
    }

    for my $creative (@creatives) {
        if ($bk_data->{AdTypeSet}->{$creative} && $bk_data->{AdTypeSet}->{$creative} !~ /^\d+$/) {
            throw Exception::Validator::Fields gettext("Nonsense in AdType %s of PageID=%s ImpID=%s", $public_id,
                $page_id, $public_id),
              sentry => {fingerprint => [@fingerprint_prefix, 'Nonsense in AdType']};
        }
    }

    if (   exists $bk_data->{RtbVideo}
        && exists $bk_data->{RtbVideo}->{VPAIDEnabled}
        && !$bk_data->{RtbVideo}->{VPAIDEnabled}
        && !in_array(1, [values %{$bk_data->{AdTypeSet}}]))
    {
        throw Exception::Validator::Fields gettext(
            "AdTypeSet %s of PageID=%s ImpID=%s (VPAIDEnabled=false and all AdTypeSet values is false)",
            $public_id, $page_id, $public_id
          ),
          sentry => {fingerprint =>
              [@fingerprint_prefix, 'AdTypeSet in AdType (VPAIDEnabled=false and all AdTypeSet values is false)']};
    }

    return TRUE;
}

sub _check_interstitial_sizes {
    my ($qv, $bk_data, $page_id, $public_id) = @_;

    if (
          !$IGNORE_EMPTY_SIZES_FOR_INTERSTITIAL
        && $bk_data->{InterstitialBlock}
        && (  !defined($bk_data->{Sizes})
            || ref($bk_data->{Sizes}) ne 'ARRAY'
            || scalar(@{$bk_data->{Sizes}}) == 0)
       )
    {
        throw Exception::Validator::Fields gettext(
            "Sizes %s of PageID=%s ImpID=%s (interstitial) incorrect or not found",
            $public_id, $page_id, $public_id
          ),
          sentry => {fingerprint => [@fingerprint_prefix, 'Sizes (interstitial) incorrect or not found']};
    }

    return TRUE;
}

sub _check_direct_limit {
    my ($qv, $bk_data, $page_id, $public_id) = @_;

    if (exists $bk_data->{DirectLimit} && !defined($bk_data->{DirectLimit})) {
        throw Exception::Validator::Fields gettext("DirectLimit of PageID=%s ImpID=%s incorrect", $page_id, $public_id),
          sentry => {fingerprint => [@fingerprint_prefix, 'DirectLimit incorrect']};
    }

    return TRUE;
}

sub _get_all_dsps {
    my ($qv, $bk_data_dsp_ids) = @_;

    my $is_cron = $qv->app->app->isa('Cron');
    my $all_dsps;
    my %filter;

    if ($is_cron) {
        $_chi_kv = CHI->new(driver => 'Memory', global => 1) unless $_chi_kv;
        $all_dsps = $_chi_kv->get($_chi_key);
        return $all_dsps if $all_dsps;
    } else {
        $filter{filter} = ['id', 'IN', $bk_data_dsp_ids];
    }

    my $tmp_rights = $qv->app->app->add_tmp_rights(qw(dsp_view_all dsp_view_field__id));
    $all_dsps = $qv->app->app->dsp->get_all(fields => ['id', 'types'], %filter);

    my %result = map {
        $_->{types} = [map {my $type = 0; $type |= 2**$_} @{$_->{types}}];
        $_->{id} => $_
    } @$all_dsps;
    if ($is_cron) {
        $_chi_kv->set($_chi_key, \%result, '1 hour');
    }
    return \%result;
}

TRUE;
