package QBit::Validator::Type::busines_rule_condition;

use qbit;
use base qw(QBit::Validator::Type);

use Exception::Validator;
use Exception::Validator::Fields;

sub get_template {
    my ($self, %opts) = @_;

    # "conditions": {
    #   "browsers": [2,88],
    #   "regions":  [1, 255],
    #   "systems":  [15,33],
    #   "devices": ["desktop", "smarttv"],
    #   "urls": ["http://foo.com", "bar.net"],
    #   "headers": {"content-type": ["text/plain"], "accept": ["application/json"]}
    #   "query_args" : {"puid1": ["foo", "bar"], "puid63": ["baz"], "adb_enabled": [1] },
    # },

    return {
        type => 'hash',

        fields => {
            browsers => {
                type     => 'array',
                optional => TRUE,
                all      => {type => 'int_un',},
                check    => sub {
                    my ($qv, $value) = @_;

                    my $dict = $qv->app->get_defaults_for__browsers();
                    my @extra_keys = grep {!$dict->{$_}} @$value;

                    throw Exception::Validator::Fields gettext('Extra values: %s', join(', ', @extra_keys))
                      if @extra_keys;
                },
            },
            regions => {
                type     => 'array',
                optional => TRUE,
                all      => {type => 'int_un'},
            },
            systems => {
                type     => 'array',
                optional => TRUE,
                all      => {type => 'int_un'},
                check    => sub {
                    my ($qv, $value) = @_;

                    my $dict = $qv->app->get_defaults_for__systems();
                    my @extra_keys = grep {!$dict->{$_}} @$value;

                    throw Exception::Validator::Fields gettext('Extra values: %s', join(', ', @extra_keys))
                      if @extra_keys;
                },
            },
            devices => {
                type     => 'array',
                optional => TRUE,
                all      => {type => 'scalar'},
                check    => sub {
                    my ($qv, $value) = @_;

                    my $dict = $qv->app->get_defaults_for__devices();
                    my @extra_keys = grep {!$dict->{$_}} @$value;

                    throw Exception::Validator::Fields gettext('Extra values: %s', join(', ', @extra_keys))
                      if @extra_keys;
                },
            },
            urls => {
                type     => 'array',
                optional => TRUE,
                all      => {type => 'scalar'},
                check    => sub {
                    my ($qv, $value) = @_;
                    foreach my $url (@$value) {
                        throw Exception::Validator::Fields gettext('One of URL is empty') unless $url;
                        throw Exception::Validator::Fields gettext('Url: "%s" is too long', $url) if length($url) > 256;
                    }
                },
            },
            query_args => {
                type     => 'hash',
                extra    => TRUE,
                optional => TRUE,
                check    => sub {
                    my ($qv, $value) = @_;

                    my @not_array = sort grep {ref($value->{$_}) ne 'ARRAY'} keys %$value;
                    throw Exception::Validator::Fields gettext('Some parameter values are not arrays: "%s"',
                        join('", "', @not_array))
                      if @not_array;

                    my $owner_id = $qv->data->{owner_id};

                    my $dict = $qv->app->get_defaults_for__query_args($owner_id);

                    my @extra_keys = sort grep {!$dict->{$_}} keys %$value;

                    throw Exception::Validator::Fields gettext('Extra values: %s', join(', ', @extra_keys))
                      if @extra_keys;

                    if ($value->{adb_enabled}) {
                        throw Exception::Validator::Fields gettext(
                            '"adb_enabled" is not an array of only one value: ["%s"]',
                            join('", "', map {$_ // 'null'} @{$value->{adb_enabled}})
                        ) if scalar(@{$value->{adb_enabled}}) > 1;

                        throw Exception::Validator::Fields gettext(
                            '"adb_enabled" can not be set to a value over ["1"]: ["%s"]',
                            $value->{adb_enabled}->[0] // 'null')
                          if !$value->{adb_enabled}->[0] || $value->{adb_enabled}->[0] ne "1";
                    }
                },
            },
            headers => {
                type     => 'hash',
                extra    => TRUE,
                optional => TRUE,
                check    => sub {
                    my ($qv, $value) = @_;

                    my $owner_id = $qv->data->{owner_id};
                    my $rights =
                      $qv->app->app->rbac->get_rights_by_user_id($owner_id,
                        ['business_rules_nobody-has-this-cool-feature-yet']);

                    throw Exception::Validator::Fields gettext('Access denied')
                      # NOTE! пока никому не даем, но должно быть 'context_on_site_adblock_view' (#PI-11720)
                      unless $rights->{'business_rules_nobody-has-this-cool-feature-yet'};

                    my @not_array = grep {ref($_) ne 'ARRAY'} values %$value;
                    throw Exception::Validator::Fields gettext('Not all values are arrays: %s',
                        join(', ', map {to_json($_, canonical => 1)} @not_array))
                      if @not_array;
                },
            },
        },
    };
}

sub _get_options {[]}

sub _get_options_name {qw()}

TRUE;
