package QBit::Validator::Type::candidates;

use qbit;

use base qw(QBit::Validator::Type);
use Digest::MD5 qw(md5_hex);

our $CANDIDATES_MAX             = 10;
our $CANDIDATES_LISTS_MAX       = 100;
our $CANDIDATE_LISTS_LENGTH_MAX = 128;

my $FIELDS = {
    tags => [qw(tags tags_blacklist)],
    urls => [qw(url_prefixes url_prefixes_blacklist)],
};

#order is important
my $OPTIONS = [];

sub get_template {
    return {
        all => {
            fields => {
                host => {
                    optional => FALSE,
                    type     => 'domain',
                },
                (
                    map {
                        $_ => {
                            all => {
                                len_min => 1,
                                len_max => $CANDIDATE_LISTS_LENGTH_MAX,
                                # разрешить только латинские буквы, цифры и символы _ и -
                                regexp => qr/^[a-z0-9_-]+$/i,
                                type   => 'scalar',
                                msg    => d_gettext(
'Tags can only contain latin letters, numbers, and characters "_" or "-". And cannot be more than %s symbols',
                                    $CANDIDATE_LISTS_LENGTH_MAX
                                ),
                            },
                            optional => TRUE,
                            type     => 'array',
                          }
                      } @{$FIELDS->{'tags'}}
                ),
                (
                    map {
                        $_ => {
                            all => {
                                len_min => 1,
                                len_max => $CANDIDATE_LISTS_LENGTH_MAX,
                                # Путь должен начинаться с символа / и может состоять из букв, цифр и непробельных символов,
                                # кроме ! * ' ( ) ; : @ + $ , [ ]
                                msg => d_gettext(
'Path must start with the / and may consist of letters, numbers, and characters except ! * \' ( ) ; : @ + $ , [ ] and cannot be more than %s symbols',
                                    $CANDIDATE_LISTS_LENGTH_MAX
                                ),
                                regexp => qr/^\/[^!\*'\(\);:\@\+\$,\[\]\s]+$/i,
                                type   => 'scalar',
                            },
                            optional => TRUE,
                            type     => 'array',
                          }
                      } @{$FIELDS->{'urls'}}
                ),
            },
            must_exists => [qw(host)],
            type        => 'hash',
        },
        optional => FALSE,
        size_max => $CANDIDATES_MAX,
        type     => 'array',
        check    => sub {
            my ($qv, $candidates) = @_;

            my @fields = map {@$_} values(%$FIELDS);

            my %totals = map {$_ => 0} @fields;
            my %duplicated;
            my %md5_candidates;
            foreach my $candidate (@$candidates) {
                foreach (@fields) {
                    if (defined $candidate->{$_}) {
                        $totals{$_} += @{$candidate->{$_}};
                        $candidate->{$_} = [sort @{$candidate->{$_}}];
                    }
                }
                my $hash = md5_hex(unset_utf(to_json($candidate, pretty => 1)));
                $duplicated{$candidate->{host}} = TRUE if $md5_candidates{$hash};
                $md5_candidates{$hash} = TRUE;
            }

            my @fields_with_errors = grep {$totals{$_} > $CANDIDATES_LISTS_MAX} @fields;
            if (@fields_with_errors) {
                throw Exception::Validator::Fields ngettext(
                    'For following field %s exceeded limit %d for items on block',
                    'For following fields: %s exceeded limit %d for items on block',
                    scalar(@fields_with_errors),
                    join(', ', map {"\"$_\""} @fields_with_errors),
                    $CANDIDATES_LISTS_MAX,
                );
            }

            throw Exception::Validator::Fields gettext(
                'Duplicated settings for following domains: %s',
                join(', ', map {"\"$_\""} sort keys %duplicated),
            ) if %duplicated;
          }
    };
}

sub _get_options {return clone($OPTIONS);}

sub _get_options_name {
    return map {$_->{'name'}} @$OPTIONS;
}

TRUE;
