package QBit::Validator::Type::design_template;

use qbit;

use base qw(QBit::Validator::Type);

use JSON::Schema::Fit;
use Struct::Diff qw(diff);

use PiConstants qw($FORMAT_SYSTEM $DEFAULT_DESIGN_TYPE $DESIGN_TYPES);

#order is important
my $OPTIONS = [];

my $MANDATORY_FIELDS = {'adaptive' => [qw(height width)],};

my $VALIDATOR_TEMPLATE = {
    $DESIGN_TYPES->{NATIVE} => {
        type    => 'hash',
        fields  => {map {$_ => {}} qw(css template grid_columns grid_rows)},
        all_set => [[qw(css template grid_columns grid_rows)]],
        check   => sub {
            my ($qv, $data) = @_;

            if (25 < $data->{grid_columns} * $data->{grid_rows}) {
                throw Exception::Validator::Fields gettext('grid_columns * grid_rows should not exceed 25');
            }
          }
    },
    $DESIGN_TYPES->{MEDIA} => {
        type   => 'hash',
        fields => {
            filterSizes          => {type => 'json_boolean'},
            horizontalAlign      => {type => 'json_boolean'},
            interscroller        => {type => 'json_boolean', optional => TRUE},
            interscrollerBgColor => {type => 'color', optional => TRUE},
            fullscreen           => {type => 'json_boolean', optional => TRUE},
            fullscreenDuration   => {type => 'scalar', min => -1, optional => TRUE},
        },
        check => sub {
            my ($qv, $data) = @_;

            if ($data->{interscrollerBgColor} and !$data->{interscroller}) {
                throw Exception::Validator::Fields gettext('Cannot set %s without %s', 'interscrollerBgColor',
                    'interscroller');
            }
            if ($data->{fullscreenDuration} and !$data->{fullscreen}) {
                throw Exception::Validator::Fields gettext('Cannot set %s without %s', 'fullscreenDuration',
                    'fullscreen');
            }
          }
    },
};

sub get_template {
    return {
        type  => 'ref',
        check => sub {
            my ($validator, $data, $template, @path) = @_;

            my %dt = $validator->app->design_templates->get_template(
                fields => [qw(caption type design_settings is_custom_format_direct)]);

            my $qv = QBit::Validator->new(
                data => {hash_transform($data, [sort keys %{$dt{template}->{fields}}])},
                template => {
                    type   => 'hash',
                    fields => $dt{template}->{fields}
                },
                app => $validator->app
            );

            my $block_data = $validator->data;

            if ($qv->has_errors) {
                foreach ($qv->get_fields_with_error) {
                    $validator->_add_error($template, $_->{msgs}->[0], [@path, @{$_->{path}}]);
                }
            }

            my $current = $validator->get_stash('current') // {};
            my $old_design_templates = $current->{'design_templates'};
            my $old_design_template;

            if (defined($old_design_templates) && @$old_design_templates && defined($data->{'id'})) {
                ($old_design_template) = grep {$_->{'id'} == $data->{'id'}} @$old_design_templates;
            }
            if ($old_design_template && $data->{'type'} ne $old_design_template->{'type'}) {
                $validator->_add_error($template, gettext("Can't change type for design template"), [@path, 'type']);
            }

            unless (scalar grep {$data->{'type'} eq $_}
                @{$validator->app->get_available_design_template_types($block_data)})
            {
                $validator->_add_error(
                    $template,
                    gettext(q[Can't use "%s" as type for design template], $data->{'type'}),
                    [@path, 'type']
                );
            }

            if ($data->{type} eq $DEFAULT_DESIGN_TYPE && defined($data->{design_settings})) {
                unless ($validator->app->check_short_rights('can_validate_design_as_manager')) {
                    my $current = $validator->get_stash('current') // {};
                    my $old_design_templates = $current->{'design_templates'};

                    my $changed_fields = {};

                    if (   $old_design_template
                        && $data->{design_settings}{name} eq $old_design_template->{'design_settings'}{name})
                    {
                        my $diff = diff(
                            $data->{design_settings}, $old_design_template->{'design_settings'},
                            noO => 1,
                            noU => 1
                        );

                        $changed_fields = $diff->{'D'} if %$diff;
                    } else {
                        my $hook_stash = $validator->app->hook_stash();

                        $changed_fields = $data->{design_settings}
                          if !$hook_stash->inited()
                              || ($hook_stash->inited() && !$hook_stash->mode('duplicate'));
                    }

                    if (%$changed_fields) {
                        my $format = $validator->app->api_format_system->formats(
                            format => $data->{design_settings}{'name'},
                            role   => 'partner',
                            site   => $block_data->{site_version},
                        );

                        my %available_fields = (name => TRUE);
                        my %deps;
                        my $default_limit;
                        foreach my $settings (@{$format->{'settings'}}) {
                            $available_fields{$settings->{'name'}} = TRUE;

                            if ($settings->{'name'} eq 'limit') {
                                $default_limit = $settings->{'type'}{'value'};
                            }
                            if (    $settings->{type}{isEnabled}
                                and $settings->{type}{isEnabled}[1][0] eq 'settings')
                            {
                                $deps{$settings->{name}} = $settings->{type}{isEnabled}[1][1];
                            }
                        }
                        $default_limit ||= 1;

                        for my $f (keys %$changed_fields) {
                            if (my $d = $deps{$f}) {
                                $changed_fields->{$d} = $data->{design_settings}{$d};
                            }
                        }

                        my @not_available_fields = sort grep {!$available_fields{$_}} keys(%$changed_fields);

                        if (@not_available_fields) {
                            foreach (@not_available_fields) {
                                $validator->_add_error(
                                    $template,
                                    gettext('You can not change this field'),
                                    [@path, 'design_settings', $_]
                                );
                            }

                            return;
                        }

                        #Валидируем дизайн через систему форматов под партнером
                        my $design = $data->{design_settings};
                        my $result = $validator->app->api_format_system->validate(
                            design => {
                                #Проставляем обязательные поля
                                (
                                    exists($changed_fields->{'name'})
                                    ? ()
                                    : (name => $design->{'name'})
                                ),
                                (exists($changed_fields->{'limit'}) ? () : (limit => $default_limit)),
                                hash_transform(
                                    $design, [@{$MANDATORY_FIELDS->{$design->{name}} // []}, keys(%$changed_fields)]
                                )
                            },
                            role => 'partner',
                            site => $block_data->{site_version}
                        );

                        _process_errors($result, $validator, $template, @path) or return;
                    }
                }

                #Валидируем дизайн через систему форматов под менеджером
                my $result = $validator->app->api_format_system->validate(
                    design => $data->{design_settings},
                    role   => 'manager',
                    site   => $block_data->{site_version}
                );

                _process_errors($result, $validator, $template, @path);
            } elsif (exists $VALIDATOR_TEMPLATE->{$data->{type}} && defined $data->{design_settings}) {
                my $qv = QBit::Validator->new(
                    data     => $data->{design_settings},
                    template => $VALIDATOR_TEMPLATE->{$data->{type}},
                    app      => $validator->app
                );

                if ($qv->has_errors) {
                    foreach ($qv->get_fields_with_error) {
                        $validator->_add_error($template, $_->{msgs}->[0], [@path, 'design_settings', @{$_->{path}}]);
                    }
                }
            }
        },
    };
}

sub _get_options {return clone($OPTIONS);}

sub _get_options_name {
    return map {$_->{'name'}} @$OPTIONS;
}

sub _process_errors {
    my ($result, $validator, $template, @path) = @_;

    unless ($result->{valid}) {
        for my $message (@{$result->{messages}}) {
            $validator->_add_error($template, $message->{text}, [@path, 'design_settings']);
        }
        for my $item_name (keys(%{$result->{items}})) {
            my $item = $result->{items}{$item_name};
            for my $message (@{$item->{messages}}) {
                $validator->_add_error($template, $message->{text}, [@path, 'design_settings', $item_name]);
            }
        }
    }

    return $result->{valid};
}

TRUE;
