package QBit::Validator::Type::hash;

use qbit;

use base qw(QBit::Validator::Type);

use Exception::Validator;

#order is important
my $OPTIONS = [
    {name => 'optional', required => TRUE},
    {name => 'all_set'},
    {name => 'deps'},
    {name => 'all'},
    {name => 'fields'},
    {name => 'any_of'},
    {name => 'not_any_of'},
    {name => 'not_any_before_dot'},
    {name => 'one_of'},
    {name => 'extra',    required => TRUE},
    {name => 'must_exists'},
];

sub all {
    my ($self, $qv, $data, $template, $option, @path_field) = @_;

    throw Exception::Validator gettext('Options "all" and "fields" can not be used together')
      if exists($template->{$option}) && exists($template->{'fields'});

    throw Exception::Validator gettext('Option "%s" must be HASH', $option)
      if !defined($template->{$option}) || ref($template->{$option}) ne 'HASH';

    foreach (keys %$data) {
        my @path = (@path_field, $_);

        $qv->_validation($data->{$_}, $template->{$option}, undef, @path);

        return FALSE if $qv->has_error(\@path);
    }

    return TRUE;
}

sub all_set {
    my ($self, $qv, $data, $template, $option, @path_field) = @_;

    throw Exception::Validator gettext('Option "all_set" must be defined')
      unless defined($template->{'all_set'});

    throw Exception::Validator gettext('Option "all_set" must be not empty ARRAY')
      if ref($template->{'all_set'}) ne 'ARRAY' || @{$template->{'all_set'}} == 0;

    my $no_error = TRUE;

    foreach my $fields (@{$template->{'all_set'}}) {
        throw Exception::Validator gettext('Items from option "all_set" must be defined')
          unless defined($fields);

        throw Exception::Validator gettext('Option "all_set" must be ARRAY with size more one')
          if ref($fields) ne 'ARRAY' || @$fields < 2;

        my $unknown_fields = arrays_difference($fields, [sort keys(%{$template->{'fields'}})]);

        throw Exception::Validator gettext('Keys is not valid: %s', join(', ', @$unknown_fields))
          if @$unknown_fields;

        my @defined_fields     = ();
        my @not_defined_fields = ();
        foreach (@$fields) {
            if (defined($data->{$_})) {
                push(@defined_fields, $_);
            } else {
                push(@not_defined_fields, $_);
            }
        }

        if (@defined_fields && @not_defined_fields) {
            $no_error = FALSE;

            foreach (@not_defined_fields) {
                my @path = (@path_field, $_);

                $qv->_add_error($template, gettext('Field "%s" must be defined', $_), \@path);
            }
        }
    }

    return $no_error;
}

sub any_of {
    my ($self, $qv, $data, $template, $option, @path_field) = @_;

    throw Exception::Validator gettext('Key "any_of" must be defined') unless defined($template->{'any_of'});

    $template->{'any_of'} = [$template->{'any_of'}] if ref($template->{'any_of'}) ne 'ARRAY';

    my @exists_keys = ();
    foreach (@{$template->{'any_of'}}) {
        throw Exception::Validator gettext('Key "%s" is not valid', $_) unless exists($template->{'fields'}{$_});

        push(@exists_keys, $_) if exists($data->{$_});
    }

    unless (@exists_keys) {
        $qv->_add_error($template, gettext('Expected any keys from list: %s', join(', ', @{$template->{'any_of'}})),
            \@path_field);

        return FALSE;
    }

    return TRUE;
}

sub not_any_before_dot {
    my ($self, $qv, $data, $template, $option, @path_field) = @_;

    my $data_before_dot = {};
    foreach my $key (sort keys %$data) {
        (my $key_before_dot) = ($key =~ m/^([^.]+)/);
        $data_before_dot->{$key_before_dot // ''} = 1;
    }

    return not_any_of($self, $qv, $data_before_dot, $template, $option, @path_field);
}

sub not_any_of {
    my ($self, $qv, $data, $template, $option, @path_field) = @_;

    my $not_any_of = $template->{$option};

    throw Exception::Validator gettext('Key "%s" must be defined', $option) unless defined $not_any_of;

    $not_any_of = {map {$_ => 1} @$not_any_of} if ref($not_any_of) eq 'ARRAY';
    throw Exception::Validator gettext('Key "%s" must be hash or array "%s"', $option, ref($not_any_of))
      unless ref($not_any_of) eq 'HASH';

    my @extra_fields = grep {exists $not_any_of->{$_}} keys %$data;

    if (@extra_fields) {
        $qv->_add_error($template, gettext('Keys are not expected: %s', join(', ', sort @extra_fields)), \@path_field);
        return FALSE;
    }

    return TRUE;
}

sub deps {
    my ($self, $qv, $data, $template, $option, @path_field) = @_;

    throw Exception::Validator gettext('Option "%s" must be HASH', $option) unless ref($template->{$option}) eq 'HASH';

    my $no_error = TRUE;

    foreach my $field (keys(%{$template->{$option}})) {
        my @path = (@path_field, $field);

        if (exists($data->{$field})) {
            my $deps = $template->{$option}{$field};

            throw Exception::Validator gettext('You must specify the fields on which the field "%s"', $field)
              unless defined($deps);

            $deps = [$deps] unless ref($deps) eq 'ARRAY';

            foreach my $dep_field (@$deps) {
                unless (exists($data->{$dep_field})) {
                    $qv->_add_error($template, gettext('Key "%s" depends from "%s"', $field, $dep_field), \@path);

                    $no_error = FALSE;

                    next;
                }

                my @dep_path = (@path_field, $dep_field);

                $qv->_validation($data->{$dep_field}, $template->{'fields'}{$dep_field}, undef, @dep_path)
                  unless $qv->checked(\@dep_path);

                $no_error = FALSE if $qv->has_error(\@dep_path);
            }
        }
    }

    return $no_error;
}

sub extra {
    my ($self, $qv, $data, $template, $option, @path_field) = @_;

    return TRUE if exists $template->{'all'};

    my @extra_fields =
      exists $template->{'fields'} ? sort grep {!$template->{'fields'}{$_}} keys(%$data) : sort keys(%$data);

    if (@extra_fields && !$template->{$option}) {
        $qv->_add_error($template, gettext('Extra fields: %s', join(', ', @extra_fields)), \@path_field);

        return FALSE;
    }

    return TRUE;
}

sub fields {
    my ($self, $qv, $data, $template, $option, @path_field) = @_;

    my $no_error = TRUE;

    foreach my $field (keys(%{$template->{$option}})) {
        my @path = (@path_field, $field);

        if (!$template->{$option}{$field}{'optional'} && !exists($data->{$field})) {
            $qv->_add_error($template, gettext('Key "%s" required', $field), \@path);
        }

        $qv->_validation($data->{$field}, $template->{$option}{$field}, undef, @path)
          unless $qv->checked(\@path);

        $no_error = FALSE if $qv->has_error(\@path);
    }

    return $no_error;
}

sub one_of {
    my ($self, $qv, $data, $template, $option, @path_field) = @_;

    throw Exception::Validator gettext('Option "%s" must be ARRAY', $option)
      if ref($template->{$option}) ne 'ARRAY';

    my $min_size = 2;

    throw Exception::Validator gettext('Option "%s" have size "%s", but expected size equal or more than "%s"',
        $option, scalar(@{$template->{$option}}), $min_size)
      if @{$template->{$option}} < $min_size;

    my @received_fields = ();
    foreach my $field (@{$template->{$option}}) {
        throw Exception::Validator gettext('Key "%s" do not use in option "fields"', $field)
          unless exists($template->{'fields'}{$field});

        push(@received_fields, $field) if exists($data->{$field});
    }

    unless (@received_fields == 1) {
        $qv->_add_error($template, gettext('Expected one key from: %s', join(', ', @{$template->{$option}})),
            \@path_field);

        return FALSE;
    }

    return TRUE;
}

sub must_exists {
    my ($self, $qv, $data, $template, $option, @path_field) = @_;

    throw Exception::Validator gettext('Option "%s" must be ARRAY', $option)
      if ref($template->{$option}) ne 'ARRAY';

    my $min_size = 1;

    throw Exception::Validator gettext('Option "%s" have size "%s", but expected size equal or more than "%s"',
        $option, scalar(@{$template->{$option}}), $min_size)
      if @{$template->{$option}} < $min_size;

    my @missed_fields = grep {!exists $data->{$_}} @{$template->{$option}};
    throw Exception::Validator gettext('Expected key: %s', join(", ", @missed_fields)) if @missed_fields;

    return TRUE;
}

sub optional {
    my ($self, $qv, $data, $template, $option, @path_field) = @_;

    if ($template->{$option}) {
        if (defined($data)) {
            unless (ref($data) eq 'HASH') {
                $qv->_add_error($template, gettext('Data must be HASH'), \@path_field);

                return FALSE;
            }
        } else {
            $qv->_add_ok(\@path_field);

            return FALSE;
        }
    } else {
        if (!defined($data)) {
            $qv->_add_error($template, gettext('Data must be defined'), \@path_field);

            return FALSE;
        } else {
            unless (ref($data) eq 'HASH') {
                $qv->_add_error($template, gettext('Data must be HASH'), \@path_field);

                return FALSE;
            }
        }
    }

    return TRUE;
}

sub _get_options {
    return clone($OPTIONS);
}

sub _get_options_name {
    return map {$_->{'name'}} @$OPTIONS;
}

TRUE;
