package QBit::WebInterface::FastCGI;

use qbit;

use base qw(QBit::WebInterface);

use QBit::WebInterface::FastCGI::Request;
use Utils::Stream::Writer::OutStream;
use PiConstants qw($STREAMING_API_BUFFER_SIZE $IS_PRECISE);

use URI::Escape qw(uri_escape_utf8);
use Utils::Logger qw(ERROR);

my %RESPONSE_TEXT = (
    200 => 'OK',
    201 => 'CREATED',
    202 => 'Accepted',
    203 => 'Partial Information',
    204 => 'No Response',
    301 => 'Moved',
    302 => 'Found',
    303 => 'Method',
    304 => 'Not Modified',
    400 => 'Bad request',
    401 => 'Unauthorized',
    402 => 'PaymentRequired',
    403 => 'Forbidden',
    404 => 'Not found',
    429 => 'Too Many Requests',
    500 => 'Internal Error',
    501 => 'Not implemented',
    502 => 'Service temporarily overloaded',
    503 => 'Gateway timeout',
);

sub get_cmd {
    my ($self) = @_;

    my ($path, $cmd);
    if ($self->request->uri() =~ /^\/([^?\/]+)(?:\/([^\/?#]+))?/) {
        ($path, $cmd) = ($1, $2);
    } else {
        ($path, $cmd) = $self->default_cmd();
    }

    $path = '' unless defined($path);
    $cmd  = '' unless defined($cmd);

    return ($path, $cmd);
}

sub make_cmd {
    my ($self, $new_cmd, $new_path, @params) = @_;

    my %vars = defined($params[0])
      && ref($params[0]) eq 'HASH' ? %{$params[0]} : @params;

    my $anchor = delete($vars{'#anchor'});

    my ($path, $cmd) = $self->get_cmd();

    $path = uri_escape_utf8($self->_get_new_path($new_path, $path));
    $cmd = uri_escape_utf8($self->_get_new_cmd($new_cmd, $cmd));

    return "/$path/$cmd"
      . (
        %vars ? '?'
          . join(
            $self->get_option('link_param_separator', '&amp;'),
            map {uri_escape_utf8($_) . '=' . uri_escape_utf8($vars{$_})} keys(%vars)
          )
        : ''
      )
      . (
        defined($anchor) ? '#' . uri_escape_utf8($anchor)
        : ''
      );
}

sub run {
    my ($self, $r) = @_;

    $self = $self->new() unless blessed($self);

    $self->request(QBit::WebInterface::FastCGI::Request->new(request => $r));

    $self->build_response();

    my $data_ref = \$self->response->data;

    if (defined($data_ref)) {
        $data_ref = $$data_ref if ref($$data_ref);

        if (ref($data_ref) eq 'SCALAR') {
            if (defined($$data_ref)) {
                utf8::encode($$data_ref) if utf8::is_utf8($$data_ref);
            } else {
                $data_ref = \'';
            }
        }
    }

    binmode(STDOUT);

    my $status = $self->response->status || 200;
    print "Status: $status" . (exists($RESPONSE_TEXT{$status}) ? " $RESPONSE_TEXT{$status}" : '') . "\n";
    print 'Set-Cookie: ' . $_->as_string() . "\n" foreach values(%{$self->response->cookies});

    my $headers = $self->response->headers;
    for my $key (sort keys %$headers) {
        my $value = $headers->{$key};
        unless ($IS_PRECISE) {
            utf8::encode($key)   if utf8::is_utf8($key);
            utf8::encode($value) if utf8::is_utf8($value);
        }
        print "$key: $value\n";
    }

    if ($status == 301 || $status == 302) {
        print 'Location: ' . $self->response->location . "\n\n";
    } else {
        print 'Content-Type: ' . $self->response->content_type . "\n";

        my $filename = $self->response->filename;
        if (defined($filename)) {
            utf8::encode($filename) if utf8::is_utf8($filename);

            print 'Content-Disposition: ' . 'attachment; filename="' . $self->_escape_filename($filename) . "\"\n";
        }

        print "\n";    # no more headers
        try {
            if (ref($data_ref) eq 'CODE') {
                my $recs;
                while (@{$recs = &$data_ref()}) {
                    print unset_utf($_) foreach @$recs;
                }
            } elsif (blessed($data_ref)) {
                my $serializer = $data_ref;
                my $writer = Utils::Stream::Writer::OutStream->new(output => \*STDOUT, encode => TRUE);
                $serializer->set_writer($writer);
                $serializer->serialize_full($STREAMING_API_BUFFER_SIZE);
            } else {
                print $$data_ref;
            }
        }
        catch {
            my ($e) = @_;

            ERROR($e) if $ENV{FORCE_LOGGER_TO_SCREEN};
            $self->exception_dumper->dump_as_html_file($e);
        };
    }
    return 1;
}

TRUE;
