/**
 * This block is working like b-form-select with set size could work.
 * You can see several options at once and there is no popup.
 * Though you can't set size for this block for now.
 **/
(function () {
    'use strict';

    function isEqual(a, b) {
        if (typeof a === typeof b) {
            return a === b;
        }
        if (typeof a === 'number' || typeof b === 'number') {
            return Number(a) === Number(b);
        }
        return false;
    }

    function Select($select, $items) {
        this._$select = $select;
        this._$items = $items;
        this._value = this.getSelectValue();
    }

    /**
     * @param {string} [value]
     **/
    Select.prototype.val = function (value) {
        if (value !== undefined && value !== 'undefined') {
            this._setSelectValue(value);
        } else {
            return this.getSelectValue();
        }
    };

    /**
     * @return {string}
     **/
    Select.prototype.displayVal = function () {
        return this._$items.filter('[class*="_selected_yes"]').text();
    };

    /**
     * @param {jQuery} $item
     * @return {boolean} - if value was changed
     **/
    Select.prototype.selectItem = function ($item) {
        if (!$item || !$item.length) {
            return true;
        }

        var newValue = this._getItemValue($item);
        var oldValue = this.getSelectValue();

        if (!isEqual(newValue, oldValue)) {
            this._setSelectValue(newValue);
            return true;
        }
        return false;
    };

    /**
     * Get item by value
     *
     * @return {string} [value]
     * @return {jQuery}
     **/
    Select.prototype.getItem = function (value) {
        if (value === undefined) {
            value = this.getSelectValue();
        }
        return this._$items.filter('[data-value="' + value + '"]');
    };

    /**
     * @param {jQuery} $item
     * @return {string}
     **/
    Select.prototype._getItemValue = function ($item) {
        return $item.attr('data-value');
    };

    /**
     * @param {string} [value]
     **/
    Select.prototype._setSelectValue = function (value) {
        this._$select.val(value);
    };

    /**
     * @return {string}
     **/
    Select.prototype.getSelectValue = function () {
        return this._$select.val();
    };

    var MultiSelect = $.inherit(Select, {
        __constructor: function () {
            return Select.apply(this, arguments);
        },

        getSelectValue: function () {
            return this._$select.val() || [];
        },

        getSelectedCount: function () {
            return this.getSelectValue().length;
        },

        selectItem: function ($item) {
            var value = this.getSelectValue();
            var item = $item.attr('data-value');
            var index = value.indexOf(item);
            if (index > -1) {
                if (value.length > 1) {
                    value.splice(index, 1);
                }
            } else {
                value.push(item);
            }
            this._$select.val(value);
            return true;
        },

        displayVal: function () {
            var items = this._$items.filter('[class*="_selected_yes"]');
            return items.toArray().reduce(function (result, item, index) {
                result = result + $(item).text();
                if (index !== items.length - 1) {
                    result += ', ';
                }
                return result;
            }, '');
        }
    });

    // block
    BEM.DOM.decl('b-form-list', {
        onSetMod: {
            js: function () {
                this._initBlock();
                this._addListeners();
                this._synchronize();
            }
        },

        scrollToSelected: function () {
            var $item = this._select.getItem(),
                $list = this.elem('list');
            if ($item.length === 0) {
                return;
            }

            var itemTop = $item.position().top,
                listHeight = $list.height(),
                listScrollTop = $list.scrollTop();

            if (itemTop < 0) {
                $list.scrollTop(listScrollTop + itemTop);
            } else if (itemTop > listHeight) {
                $list.scrollTop(itemTop);
            }
        },

        /**
         * Set or get value
         *
         * @param {string} [value]
         * @return {string|undefined} - undefined if something was passed
         **/
        val: function (value) {
            if (typeof value === 'undefined') {
                return this._select.val();
            }
            this._select.val(value);
            this._setSelection(value);
        },

        /**
         * Return name attribute of select
         *
         * @return {string}
         **/
        name: function () {
            return this.elem('select').attr('name');
        },

        /**
         * Return text of the selected option
         *
         * @return {string}
         **/
        displayVal: function () {
            return this._select.displayVal();
        },

        _initBlock: function () {
            this._select =
                this.hasMod('multiple', 'yes') ?
                new MultiSelect(this.elem('select'), this.elem('item')) :
                new Select(this.elem('select'), this.elem('item'));
        },

        _addListeners: function () {
            this.bindTo('item', 'click', this._onItemClick);
        },

        _onItemClick: function (e) {
            if (e) {
                e.preventDefault();
                if (!this.hasMod('disabled', 'yes')) {
                    this._selectItem($(e.currentTarget), e);
                }
            }
        },

        /**
         * @param {jQuery} $item
         * @param {event} e
         **/
        _selectItem: function ($item, e) {
            if ($item && this._select.selectItem($item)) {
                // this won't happen if both items have same values
                if (this.hasMod('multiple', 'yes')) {
                    if (!(this._select.getSelectedCount() === 1 &&
                        $item.attr('data-value') === this._select.getSelectValue()[0])) {
                        this.toggleMod($item, 'selected', 'yes');
                    }
                } else {
                    this.delMod(this.elem('item', 'selected', 'yes'), 'selected');
                    this.setMod($item, 'selected', 'yes');
                }
                this.trigger('change', {initialEvent: e});
            }
        },

        _deselectAll: function () {
            this.delMod(this._select._$items, 'selected');
        },

        _setSelection: function (selection) {
            var _this = this;
            selection = Array.isArray(selection) ? selection : [selection];
            this._deselectAll();
            selection.forEach(function (value) {
                var $item = _this._select.getItem(value);
                _this.setMod($item, 'selected', 'yes');
            });
        },

        _synchronize: function () {
            var $item = this._select.getItem(this._select.val());
            this.setMod($item, 'selected', 'yes');
        }
    });
})();
