package RestApi;

use strict;
use warnings FATAL => 'all';
use utf8;

use Mojo::Util qw(monkey_patch url_unescape split_header);
use Mojo::Base qw(Mojolicious);
use Mojo::Cookie::Request;

use Digest::MD5 qw(md5_hex);
use JSON::XS;

use Application;
use RestApi::Controller;
use RestApi::Errors;

use Utils::Swagger;
use Utils::Logger qw(ERROR);
use QBit::Exceptions qw(throw);
use QBit::GetText qw(gettext);
use QBit::Date qw(round_timestamp);
use QBit::Packages qw(package_stash);
use Exception;

use PiConstants qw($CONTENT_TYPE $CONTENT_TYPE_JSON);

$ENV{SYSTEM} = 'restapi';

my $MAX_REQUESTS_AT_SAME_TIME = 6;
# map headers from code attributes, ex ': CACHECONTROL(3600)' to actual http headers names
my $HTTP_HEADER_MAP = {
    cachecontrol => {
        name  => 'Cache-control',
        value => sub {sprintf('max-age=%s', $_[0])}
    },
};

sub startup {
    my $self = shift;

    my $json_xs = JSON::XS->new->utf8->allow_nonref;
    monkey_patch 'Mojo::JSON', '_decode'       => sub {unshift @_, $json_xs; goto \&_json_xs_decode};
    monkey_patch 'Mojo::JSON', '_encode_value' => sub {unshift @_, $json_xs; goto \&_json_xs_encode};
    monkey_patch 'Mojo::JSON', 'true'  => sub () {JSON::XS::true()};
    monkey_patch 'Mojo::JSON', 'false' => sub () {JSON::XS::false()};
    monkey_patch 'Mojo::Cookie::Request', 'parse' => \&_parse_and_unescape_cookies;

    my $app = Application->new();
    $app->pre_run();

    $self->helper(models => sub {my $models = $app;});

    $self->secrets([$self->models->get_option('salt')]);

    $self->controller_class('RestApi::Controller');

    $self->hook(
        before_dispatch => sub {
            my ($controller) = @_;

            $SIG{__DIE__} = \&QBit::Exceptions::die_handler;

            $controller->models->pre_run();

            $controller->models->set_option(restapi_controller => $controller);

            my $auth = $controller->authorization();

            $auth->{result}  //= '';
            $auth->{message} //= '';
            $auth->{is_authed} = $auth->{'result'} eq 'ok';

            $controller->stash(authorization => $auth);

            if ($auth->{'result'} eq 'ok') {
                if ($auth->{'user'}{'authorization_by'} eq 'cookie') {

                    my $server_csrf_token = md5_hex(round_timestamp(scalar(time())) . '_' . $auth->{'user'}{'id'});

                    $controller->res->headers->header('X-Frontend-Authorization' => 'token ' . $server_csrf_token);

                    if ($controller->req->method ne 'GET') {
                        my $csrf_token = $controller->req->headers->header('X-Frontend-Authorization') // '';
                        $csrf_token =~ s/token //;

                        return $controller->render(
                            $controller->get_error(ERROR__NEED_AUTH, gettext('Incorrect csrf token')))
                          unless $csrf_token eq $server_csrf_token;
                    }
                }

                # Обработка лимитов на количество запросов в минуту
                my $request_limits = $controller->models->get_option('api_configs', {})->{'request_limits'};
                my $limit = get_user_rpm_limit($request_limits, $auth->{'user'}{'login'});

                if ($controller->req->url->path->to_string =~ m|^/v1/api/frontend/rosetta|) {
                    $limit = undef;
                }

                # $limit = null means no limit
                if (defined($limit)) {
                    # $limit = 0 means no access
                    if ($limit <= 0) {
                        return $controller->render(
                            $controller->get_error(ERROR__MANY_REQ, gettext('Allowed %s requests for minute', $limit)));
                        # $limit > 0 means $limit RPM
                    } else {
                        my $request_count =
                          $controller->models->memcached->incr(request_count => $auth->{'user'}{'id'});
                        if (!defined($request_count)) {
                            $controller->models->memcached->set(request_count => $auth->{'user'}{'id'}, 1, 60);
                        } elsif ($request_count > $limit) {
                            return $controller->render(
                                $controller->get_error(ERROR__MANY_REQ,
                                    gettext('Allowed %s requests for minute', $limit)
                                )
                            );
                        }

                        my $requests_at_same_time =
                          $controller->models->memcached->get(requests_at_same_time => $auth->{'user'}{'id'}) // 0 + 1;

                        if ($requests_at_same_time > $MAX_REQUESTS_AT_SAME_TIME) {
                            return $controller->render($controller->get_error(ERROR__MANY_REQ));
                        } else {
                            $controller->models->memcached->set(
                                requests_at_same_time => $auth->{'user'}{'id'},
                                $requests_at_same_time, 5 * 60
                            );
                        }
                    }
                }
            }

            my $accept = $controller->req->headers->header('Accept');
            if (
                (
                       $controller->req->url->path->to_string ne '/v1/swagger.json'
                    && $controller->req->url->path->to_string !~ '^/v1/api/'
                )
                && (!defined($accept) || $accept ne $CONTENT_TYPE)
               )
            {
                return $controller->render($controller->get_error(ERROR__NOT_ACCEPTABLE));
            }

            my $content_type = $controller->req->headers->header('Content-Type');
            if (defined($content_type) && $content_type ne $CONTENT_TYPE) {
                return $controller->render($controller->get_error(ERROR__MEDIA_TYPE));
            }
        },
    );

    $self->hook(
        before_render => sub {
            my ($controller, $args) = @_;

            if (defined($args->{'exception'})) {
                my $exception =
                  ref($args->{'exception'}{'message'})
                  ? $args->{'exception'}{'message'}
                  : Exception->new($args->{'exception'}{'message'});

                $controller->models->exception_dumper->dump_as_html_file($exception);
            }

            if ($controller->stash('namespace') && $controller->stash('namespace') eq 'RestApi::Controller::API') {
                $controller->res->headers->content_type($CONTENT_TYPE_JSON);
            } else {
                $controller->res->headers->content_type($CONTENT_TYPE);
            }
            $controller->res->headers->header('Server' => 'nginx');

            if (delete $args->{json}{nometa}) {
                delete $args->{json}{meta};
            }

            return unless exists($args->{'status'});

            if ($args->{'status'} == 404) {
                $args->{'json'} = {errors => [$controller->get_error_object(ERROR__NOT_FOUND)]};
            } elsif ($args->{'status'} =~ /^5[0-9]{2}\z/) {
                $args->{'json'} = {
                    errors => [
                        $controller->get_error_object(ERROR__INTERNAL,
                            (
                                $args->{'exception'} && $self->models->get_option('stage', '') eq 'dev'
                                ? (message => "$args->{'exception'}")
                                : ()
                            ),
                        )
                    ],
                };
            }
        },
    );

    $self->hook(
        after_dispatch => sub {
            my ($controller) = @_;

            $controller->res->headers->header('X-Content-Type-Options' => 'nosniff')
              if $controller->res->headers->content_type;
            $controller->res->headers->header(
                'X-Yandex-Login' => $controller->models->get_option('cur_user', {})->{'login'} // '');

            # headers from code attributes, ex ': CACHECONTROL(3600)'
            for my $http_header (@{_get_additional_headers($controller, $HTTP_HEADER_MAP)}) {
                $controller->res->headers->header($http_header->{name} => $http_header->{value});
            }

            if (   $controller->stash('authorization')
                && $controller->stash('authorization')->{'is_authed'}
                && $controller->models->get_option('api_configs', {})->{'limits'})
            {
                $controller->models->memcached->decr(
                    requests_at_same_time => $controller->models->get_option('cur_user', {})->{'id'});
            }

            $controller->models->set_option(restapi_controller => undef);

            $controller->models->post_run();
        }
    );

    # Router
    my $r = $self->routes;

    $r->namespaces(['RestApi::Controller']);

    $r->add_condition(
        jsonapi__create => sub {
            my ($route, $controller, $captures) = @_;
            return $controller->jsonapi_validate_create($captures);
        }
    );

    $r->add_condition(
        jsonapi__update => sub {
            my ($route, $controller, $captures) = @_;
            return $controller->jsonapi_validate_update($captures);
        },
    );

    $r->add_condition(
        jsonapi__update_multiple => sub {
            my ($route, $controller, $captures) = @_;
            return $controller->jsonapi_validate_update_multiple($captures);
        },
    );

    $r->add_condition(
        chain => sub {
            my ($route, $controller, $captures, $conditions) = @_;

            my $routes = $route;
            while (not $routes->isa('Mojolicious::Routes')) {
                $routes = $routes->parent;
            }

            for my $c (@$conditions) {
                my $sub = $routes->conditions->{$c};
                return 0 unless $sub->($route, $controller, $captures);
            }

            return 1;
        },
    );

    $r->add_condition(
        simple_model => sub {
            my ($route, $controller, $captures) = @_;

            my $resource = $captures->{'resource'};

            return $controller->is_available_resource($resource, 'GET', 'RestApi::SimpleModel');
        }
    );

    $r->add_condition(
        available__get => sub {
            my ($route, $controller, $captures) = @_;

            return
              if grep {exists($captures->{'public_id'}) && $captures->{'public_id'} eq $_}
                  qw(add_fields depends defaults);

            my $resource = $captures->{'resource'};

            return $controller->is_available_resource($resource, 'GET', 'RestApi::DBModel');
        }
    );

    $r->add_condition(
        available__add => sub {
            my ($route, $controller, $captures) = @_;

            my $resource = $captures->{'resource'};

            return $controller->is_available_resource($resource, 'POST', 'RestApi::DBModel')
              && $self->models->$resource->api_can_add();
        }
    );

    $r->add_condition(
        available__edit => sub {
            my ($route, $controller, $captures) = @_;

            my $resource = $captures->{'resource'};

            return $controller->is_available_resource($resource, 'GET', 'RestApi::DBModel')
              && $self->models->$resource->api_can_edit();
        }
    );

    $r->add_condition(
        available__delete => sub {
            my ($route, $controller, $captures) = @_;

            my ($resource, $public_id, $linked_resource) = @$captures{qw(resource public_id linked_resource)};
            my $is_relationships = defined($linked_resource);

            if ($is_relationships) {
                return
                  unless defined($public_id)
                      && $controller->is_available_resource($resource, 'GET', 'RestApi::DBModel');

                return $controller->is_available_resource($linked_resource, 'DELETE', 'RestApi::DBModel')
                  && $self->models->$linked_resource->api_can_delete();
            } else {
                return $controller->is_available_resource($resource, 'DELETE', 'RestApi::DBModel')
                  && $self->models->$resource->api_can_delete();
            }
        }
    );

    $r->add_condition(
        available__do_action => sub {
            my ($route, $controller, $captures) = @_;

            my $resource = $captures->{'resource'};

            my $is_available_resource =
              $controller->is_available_resource($resource, 'GET', 'RestApi::MultistateModel');

            return $is_available_resource unless $is_available_resource;

            my %action_to_operation = reverse $self->models->$resource->operation_to_action();

            my @available_actions =
              map {$action_to_operation{$_} // $_} $self->models->$resource->api_available_actions();

            return !!grep {$captures->{'operation'} eq $_} @available_actions;
        }
    );

    #
    # Routes with authorization
    #

    my $auth = $r->under(
        '/v1' => sub {
            my ($controller) = @_;

            my $auth = $controller->stash('authorization');
            # Authenticated
            return 1 if $auth->{'is_authed'};

            if ($auth->{'result'} eq 'invalid_apikey' || $auth->{'result'} eq 'need_auth') {
                $controller->render(
                    $controller->get_error(ERROR__NEED_AUTH, $auth->{'message'}, (meta => $auth->{meta})));
                return undef;
            } elsif ($auth->{'result'} eq 'fake_login_user_not_found') {
                $controller->render($controller->get_error(ERROR__PARAMS, $auth->{'message'}, (meta => $auth->{meta})));
                return undef;
            } elsif ($auth->{'result'} eq 'fake_login_unallowed'
                || $auth->{'result'} eq 'fake_login_right_not_found')
            {
                $controller->render(
                    $controller->get_error(ERROR__FORBIDDEN, $auth->{'message'}, (meta => $auth->{meta})));
                return undef;
            } elsif ($auth->{meta}) {
                $controller->render(
                    $controller->get_error(ERROR__NEED_AUTH, $auth->{'message'}, (meta => $auth->{meta})));
                return undef;
            }

            # Not authenticated
            $controller->render($controller->get_error(ERROR__NEED_AUTH));
            return undef;
        }
    );

    #API
    my %api_routes_controllers = (
        'api_adfox' => {
            'controller' => 'ApiAdfox',
            'method'     => {'post' => 1,}
        },
        'block_preset_applier' => {
            'controller' => 'BlockPresetApplier',
            'method'     => {'post' => 1,}
        },
        'brands' => {
            'controller' => 'Brands',
            'method'     => {'post' => 1}
        },
        'common_offer' => {
            'controller' => 'CommonOffer',
            'method'     => {'post' => 1,}
        },
        'context_on_site_rtb' => {
            'controller' => 'ContextOnSiteRtb',
            'method'     => {'post' => 1,}
        },
        'frontend' => {
            'controller' => 'Frontend',
            'method'     => {'get' => 1, 'post' => 1,}
        },
        'inviter' => {
            'controller' => 'Inviter',
            'method'     => {'get' => 1, 'post' => 1,}
        },
        'maps' => {
            'controller' => 'Maps',
            'method'     => {'post' => 1}
        },
        'mobile_app_rtb' => {
            'controller' => 'MobileAppRtb',
            'method'     => {'post' => 1,}
        },
        'partner_acts' => {
            'controller' => 'PartnerActs',
            'method'     => {'post' => 1, 'get' => 1,}
        },
        'payoneer' => {
            'controller' => 'Payoneer',
            'method'     => {'get' => 1, 'post' => 1}
        },
        'requisites' => {
            'controller' => 'Requisites',
            'method'     => {'get' => 1, 'post' => 1}
        },
        'statistics' => {
            'controller' => 'Statistics',
            'method'     => {'get' => 1, 'post' => 1}
        },
        'selfemployed' => {
            'controller' => 'SelfEmployed',
            'method'     => {'post' => 1, 'get' => 1,}
        },
        'game_offer' => {
            'controller' => 'GameOffer',
            'method'     => {'post' => 1, 'get' => 1,}
        },
        'uploader' => {
            'controller' => 'Uploader',
            'method'     => {'post' => 1, 'get' => 1,}
        },
        'validator' => {
            'controller' => 'Validator',
            'method'     => {'post' => 1,}
        },
        'video_an_site_fullscreen' => {
            'controller' => 'VideoAnSite::Fullscreen',
            'method'     => {'post' => 1,}
        },
        'video_an_site_inpage' => {
            'controller' => 'VideoAnSite::InPage',
            'method'     => {'post' => 1,}
        },
        'video_an_site_instream' => {
            'controller' => 'VideoAnSite::InStream',
            'method'     => {'post' => 1,}
        },
        'bk_statistics' => {
            'controller' => 'BkStatistics',
            'method'     => {'get' => 1,}
        }
    );

    foreach my $route (keys %api_routes_controllers) {
        my $controller = $api_routes_controllers{$route}->{controller};

        foreach my $request_method (keys %{$api_routes_controllers{$route}->{method}}) {
            $auth->$request_method("/api/${route}/:method")
              ->to(namespace => 'RestApi::Controller::API', controller => $controller, action => 'api_call')
              ->name("api_call_${route}_method");
        }
    }

    #Simple models (only "get" without filters and sorting)
    $auth->get('/:resource')->over('simple_model')->to(controller => 'SimpleModel', action => 'resource')
      ->name('simple_model__resource');
    $auth->get('/:resource/:public_id')->over('simple_model')->to(controller => 'SimpleModel', action => 'resource_get')
      ->name('simple_model__resource_get');

    # get element(s) of model with multistate
    $auth->get('/:resource')->over('available__get')->to(controller => 'MultistateModel', action => 'resource')
      ->name('multistate_model__resource');

    $auth->get('/:resource/:public_id')->over('available__get')
      ->to(controller => 'MultistateModel', action => 'resource_get')->name('multistate_model__resource_get');

    $auth->get('/:resource/:public_id/relationships/:linked_resource')->over('available__get')
      ->to(controller => 'MultistateModel', action => 'relationships')->name('multistate_model__relationships');

    $auth->delete('/:resource/:public_id/relationships/:linked_resource')->over('available__delete')
      ->to(controller => 'MultistateModel', action => 'delete')->name('multistate_model__delete_relationships');

    $auth->get('/:resource/:public_id/:linked_resource')->over('available__get')
      ->to(controller => 'MultistateModel', action => 'linked_resource')->name('multistate_model__linked_resource');

    # models with multistate directory
    $auth->get('/:resource/add_fields')->over('available__add')
      ->to(controller => 'MultistateModel', action => 'add_fields')->name('multistate_model__add_fields');

    $auth->get('/:resource/depends')->over('available__get')->to(controller => 'MultistateModel', action => 'depends')
      ->name('multistate_model__depends');

    $auth->get('/:resource/defaults')->over('available__get')->to(controller => 'MultistateModel', action => 'defaults')
      ->name('multistate_model__defaults');

    # add a new element to model with multistate
    $auth->post('/:resource')->over(chain => ['available__add', 'jsonapi__create'])
      ->to(controller => 'MultistateModel', action => 'add')->name('multistate_model__add');

    # edit a one element of model with multistate
    $auth->patch('/:resource/:public_id')->over(chain => ['available__edit', 'jsonapi__update'])
      ->to(controller => 'MultistateModel', action => 'edit')->name('multistate_model__edit');

    # delete a one element of model with multistate
    $auth->delete('/:resource/:public_id')->over('available__delete')
      ->to(controller => 'MultistateModel', action => 'delete')->name('multistate_model__delete');

    # edit multiple elements of model with multistate
    $auth->patch('/:resource')->over(chain => ['available__edit', 'jsonapi__update_multiple'])
      ->to(controller => 'MultistateModel', action => 'multi_edit')->name('multistate_model__multi_edit');

    # models with mulstitate actions
    $auth->post('/:resource/:public_id/action/:operation')->over('available__do_action')
      ->to(controller => 'MultistateModel', action => 'do_action')->name('multistate_model__do_action');

    #Action logs
    $auth->get('/<resource>_action_log')->to(controller => 'MultistateModel', action => 'action_log')
      ->name('multistate_model__action_log');

    #удалить после перехода на bionic
    $auth->get('/(resource)_action_log')->to(controller => 'MultistateModel', action => 'action_log')
      ->name('multistate_model__action_log');

    #
    # Routes without authorization
    #
    my $no_auth = $r->under('/v1');

    $no_auth->get('swagger.json')->to(
        cb => sub {
            my ($c) = @_;

            if ($c->models->get_option('stage', 'unknown') eq 'dev') {

                my $json = Utils::Swagger::get_swagger_json();

                $json->{'host'} = $c->req->env->{'HTTP_HOST'} . ':' . _get_beta_https_port();

                return $c->render(json => $json);
            } else {
                return $c->render(json => {});
            }
        }
    );

    $no_auth->get('/api/alive')->to(
        cb => sub {
            my ($c) = @_;
            return $c->render(json => {result => 'ok', nometa => 1});
        }
    );

    $no_auth->get('/api/readiness_probe')->to(
        cb => sub {
            my ($c) = @_;
            return $c->render(json => _is_alive($c));
        }
    );
}

sub _get_beta_https_port {
    my $https_port;

    if ($ENV{PARTNER2_HTTPS_PORT}) {
        # В случае креаторной беты выставляется пременная окружения
        $https_port = $ENV{PARTNER2_HTTPS_PORT};
    } else {
        # В случае беты на dev сервере порт можно выяснить из названия папки
        my $pwd = `pwd`;
        my ($beta_http_port) = $pwd =~ /(\d{4})\n\z/;
        $https_port = $beta_http_port + 400;
    }

    return $https_port;
}

sub _json_xs_decode {
    my ($json_xs, $valueref, $json) = @_;

    eval {
        utf8::encode($json);

        $$valueref = $json_xs->decode($json);
    } ? return undef : chomp $@;

    return $@;
}

sub _json_xs_encode {
    my ($json_xs, $data) = @_;
    my $json = $json_xs->encode($data);

    utf8::decode($json) unless utf8::is_utf8($json);

    return $json;
}

sub _parse_and_unescape_cookies {
    my ($self, $str) = @_;

    my @cookies;
    my @pairs = map {@$_} @{split_header $str // []};
    while (my ($name, $value) = splice @pairs, 0, 2) {
        next if $name =~ /^\$/;
        push @cookies, Mojo::Cookie::Request->new(name => $name, value => $value ? url_unescape($value) : '');
    }

    return \@cookies;
}

=head2
    config example:
    "request_limits" : {
         "default_limit" : 60,
         "per_user_limit" : {
            "yndx-robot-adfox-pibot" : 60
         }
    }
    limit : null == no limits
    limit : 0 == no access
=cut

sub get_user_rpm_limit {
    my ($request_limits, $login) = @_;

    return exists $request_limits->{'per_user_limit'}{$login}
      ? $request_limits->{'per_user_limit'}{$login}
      : $request_limits->{'default_limit'};
}

sub _get_additional_headers {
    my ($controller, $header_map) = @_;

    my $header_list = [];

    if (my $attrs = package_stash(ref $controller)->{__API_ATTRS__}) {
        my $method_attrs = $attrs->{ref $controller, $controller->stash('method')};
        for my $attr_name (keys %{$method_attrs}) {
            if (exists $header_map->{$attr_name}) {
                push @$header_list,
                  {
                    name  => $header_map->{$attr_name}{name},
                    value => $header_map->{$attr_name}{value}->($method_attrs->{$attr_name})
                  };
            }
        }
    }

    return $header_list;
}

sub _is_alive {
    my ($c) = @_;

    my $result = 'error';
    eval {
        my ($data, $msg) = $c->models->check_alive->is_alive(raw => 1);
        $result = $data || $msg;
    };

    return {result => $result, nometa => 1};
}

1;
