package RestApi::Controller;

use qbit;

use Mojo::Base qw(Mojolicious::Controller);

use Exception::BlackBox::NeedAuth;
use Exception::BlackBox::NeedRealLogin;
use Exception::BlackBox::NeedResign;
use Exception::BlackBox::NotSecure;
use Exception::Conflict;
use Exception::IncorrectParams;
use Exception::IncorrectParams::Body;

use RestApi::Errors;
use RestApi::Relationships;

use POSIX;
use URI;

use Data::Rmap;

use Utils::Logger qw/ERROR/;
use Utils::JSON qw(clean_decoded_json_from_boolean_objects_in_place get_boolean_representation);

use PiConstants qw($AUTHORIZATION_USER_FIELDS $BLACKBOX_DBFIELDS $MAX_PAGE_SIZE);

my $MAX_BULK_COUNT = 100;

sub authorization {
    my ($self) = @_;

    my $data;
    my $error;

    my $app = $self->models;
    $app->set_cur_user(undef);

    my $api_key = $self->param('apikey') // $self->req->headers->header('Authorization');

    my $ip = $self->req->env->{'HTTP_X_REAL_IP'} // $ENV{'HTTP_X_REAL_IP'}
      // throw gettext('IP not found');    #'2a02:6b8:0:408:652b:5f7a:f541:b23b' //
    my $accept_language = $self->req->headers->header('Accept-Language') // 'en-US,en;q=0.5';
    my $host = $self->req->env->{'HTTP_HOST'} // $ENV{'HTTP_HOST'} // throw gettext('HOST not found');
    my $my_cookie = $self->cookie('my');

    #fix host
    $host =~ s/^jsonapi\.//;

    my ($cur_user, $authorization_by);

    if (defined($api_key)) {
        $authorization_by = 'token';

        $api_key =~ s/^token\s+//;

        try {
            $data = $app->api_cabinet->check_key(
                user_ip => $ip,
                key     => $api_key,
            );

            if (exists($data->{'result'}) && $data->{'result'} eq 'OK') {
                $cur_user = {
                    blackbox         => FALSE,
                    id               => $data->{'key_info'}{'user'}{'uid'},
                    login            => $data->{'key_info'}{'user'}{'login'},
                    display_login    => $data->{'key_info'}{'user'}{'login'},
                    email            => $data->{'key_info'}{'user'}{'email'},
                    client_id        => $data->{'key_info'}{'user'}{'balance_client_id'},
                    business_unit    => 0,
                    authorization_by => $authorization_by,
                };

                ($cur_user->{'lastname'}, $cur_user->{'name'}, $cur_user->{'midname'}) =
                  (split(' ', $data->{'key_info'}{'user'}{'name'} || ''), '', '', '');
            } else {
                $error = {
                    result  => 'error',
                    message => $data->{'error'} // '',
                };
            }
        }
        catch {
            $error = {result => 'invalid_apikey', message => shift->message()};
        };
    } else {
        $authorization_by = 'cookie';

        try {
            $data = $app->api_blackbox->sessionid(
                session_id     => $self->cookie('Session_id'),
                sessionid2     => $self->cookie('sessionid2'),
                remote_addr    => $ip,
                server_name    => $host,
                fields         => $BLACKBOX_DBFIELDS,
                must_be_secure => TRUE,
            );

            $cur_user = {
                blackbox         => TRUE,
                id               => $data->{'uid'}{'content'},
                login            => $data->{'dbfield'}{'accounts.login.uid'},
                language         => $data->{'dbfield'}{'userinfo.lang.uid'},
                display_login    => $data->{'login'}{'content'},
                email            => $data->{'address-list'}->{'address'}{'content'},
                business_unit    => 0,
                authorization_by => $authorization_by,
            };

            ($cur_user->{'lastname'}, $cur_user->{'name'}, $cur_user->{'midname'}) =
              (split(' ', $data->{'dbfield'}{'account_info.fio.uid'} || ''), '', '', '');
        }
        catch Exception::BlackBox::NeedAuth with {
            # разделено на 2 блока, т.к. в IDEA ломается парсинг
            my ($e) = @_;
            $error->{'result'} = 'need_auth';
            $error->{meta} = {'url' => $e->{'url_auth'}};
        }
        catch Exception::BlackBox::NotSecure with {
            my ($e) = @_;
            $error->{'result'} = 'need_auth';
            $error->{meta} = {'url' => $e->{'url_auth'}};
        }
        catch Exception::BlackBox::NeedResign with {
            my ($e) = @_;
            $error->{'result'} = 'need_resign';
            $error->{meta} = {'url' => $e->{'url_resign'}};
        }
        catch Exception::BlackBox::NeedRealLogin with {
            my ($e) = @_;
            $error->{'result'} = 'need_real_login';
            $error->{meta} = {'url' => $e->{'url_real_login'}};
        }
        catch {
            my ($e) = @_;
            $error = {'result' => 'error', message => $self->safe_exception_message($e)};
        };
    }

    $self->simple_lang_detect(
        ip              => $ip,
        accept_language => $accept_language,
        host            => $host,
        (defined($cur_user->{'language'}) ? (pass_language => $cur_user->{'language'},) : ()),
        (defined($my_cookie) ? (my_cookie => $my_cookie,) : ()),
    );

    if ($error) {
        return $error;
    } else {

        my $tmp_rights = $app->add_tmp_rights(
            qw(
              users_view_all
              users_view_field__multistate
              users_view_field__client_id
              )
        );
        my $user_from_db = $app->users->get_by_uid($cur_user->{'id'}, fields => $AUTHORIZATION_USER_FIELDS);

        if (defined($user_from_db)) {
            $cur_user = {
                %$user_from_db,
                display_login    => $cur_user->{'display_login'},
                authorization_by => $authorization_by,
                (defined($cur_user->{'language'}) ? (language => $cur_user->{'language'},) : ()),
            };
        }

        #удаляем временные поля чтобы нельзя было залогиниться под любым пользователем.
        undef($tmp_rights);

        $app->set_cur_user($cur_user);

        my $fake_login = $self->cookie('fakelogin');

        if (defined($fake_login)) {
            return {result => 'fake_login_unallowed', message => gettext('Fake login unallowed')}
              unless $self->models->get_option('allow_fake_login');
            return {result => 'fake_login_right_not_found', message => gettext("Can't fake login. Access denied")}
              unless $self->models->check_rights('fake_login');

            my $fake_user = $self->models->users->get_by_login($fake_login, fields => $AUTHORIZATION_USER_FIELDS);

            return {result => 'fake_login_user_not_found', message => gettext('Bad fake login')} unless $fake_user;

            $fake_user->{'display_login'}    = $fake_user->{'login'};
            $fake_user->{'authorization_by'} = $authorization_by;
            $fake_user->{'fake'}             = TRUE;
            $fake_user->{'real_user'}        = $cur_user;
            $cur_user                        = $fake_user;

            $app->set_cur_user($cur_user);
        }

        return {
            result => 'ok',
            user   => $cur_user,
        };
    }
}

# Введение этой функции считаю хаком
sub name_prefix_for_get_abs {
    throw 'name_prefix_for_get_abs() is abstract method';
}

sub get_abs_for_sprintf {
    my ($self, $name, @params) = @_;

    my %opts = map {$_ => '__' . uc($_) . '__'} @params;

    my $url = $self->url_for($name, %opts)->to_abs;

    $url->scheme($self->req->env->{'HTTP_X_REAL_SCHEME'});
    $url->port($self->req->env->{'HTTP_X_REAL_PORT'});

    my $str = $url->to_string();

    my $regexp = join('|', values(%opts));

    $str =~ s/$regexp/%s/g;
    $str = $self->_escape_url($str);

    return $str;
}

sub get_abs_url {
    my ($self, $name, %opts) = @_;

    my $url = $self->url_for($name, %opts)->to_abs;

    $url->scheme($self->req->env->{'HTTP_X_REAL_SCHEME'});
    $url->port($self->req->env->{'HTTP_X_REAL_PORT'});

    my $str = $url->to_string();
    $str = $self->_escape_url($str);

    return $str;
}

sub get_abs_url_with_params {
    my ($self, $url, %opts) = @_;

    foreach (sort keys(%opts)) {
        $url->query->merge($_, $opts{$_});
    }

    my $str = $url->to_string();
    $str = $self->_escape_url($str);

    return $str;
}

#TODO: move set scheme and port in hook

sub get_current_url {
    my ($self) = @_;

    my $url = $self->req->url->to_abs;

    $url->scheme($self->req->env->{'HTTP_X_REAL_SCHEME'});
    $url->port($self->req->env->{'HTTP_X_REAL_PORT'});

    my $str = $url->to_string();
    $str = $self->_escape_url($str);

    return $str;
}

sub get_error {
    my ($self, $error_code, $message, %opts) = @_;

    my $error = $self->get_error_object($error_code, ($message ? (detail => $message) : ()), %opts);

    $self->res->code($RestApi::Errors::ERRORS->{$error_code}{'http_code'});

    return json => {errors => [$error],};
}

sub get_error_object {
    my ($self, $error_code, %opts) = @_;

    throw gettext('Error with code "%s" not found', $error_code) unless exists($RestApi::Errors::ERRORS->{$error_code});

    return {
        id    => 0 + $error_code,
        title => $RestApi::Errors::ERRORS->{$error_code}{'title'}->(),
        %opts
    };
}

sub get_links_for_collections {
    my ($self, $limit, $page_number, $count) = @_;

    my $url = $self->req->url->to_abs;

    $url->scheme($self->req->env->{'HTTP_X_REAL_SCHEME'});
    $url->port($self->req->env->{'HTTP_X_REAL_PORT'});

    my %links = (self => $self->_escape_url($url->to_string));

    if (defined($limit)) {
        my $page_count = ceil($count / $limit) || 1;

        my $prev = $page_number - 1;
        my $next = $page_number + 1;

        $links{'first'} =
          $page_number == 1 ? $links{'self'} : $self->get_abs_url_with_params($url, 'page[number]' => 1);
        $links{'last'} =
            $page_number == $page_count
          ? $links{'self'}
          : $self->get_abs_url_with_params($url, 'page[number]' => $page_count);
        $links{'prev'} = $page_number == 1 ? undef : $self->get_abs_url_with_params($url, 'page[number]' => $prev);
        $links{'next'} =
          $page_number == $page_count ? undef : $self->get_abs_url_with_params($url, 'page[number]' => $next);
    } else {
        $links{$_} = $links{'self'} foreach (qw(first last));
        $links{$_} = undef          foreach (qw(prev next));
    }

    return %links;
}

sub safe_exception_message {
    my ($self, $exception) = @_;

    $self->models->exception_dumper->dump_as_html_file($exception);

    return gettext('Internal error');
}

sub set_http_code {
    my ($self, $error_code) = @_;

    $self->res->code($RestApi::Errors::ERRORS->{$error_code}{'http_code'});
}

=encoding UTF-8

=head1 simple_lang_detect

    ->simple_lang_detect(
        ip => '95.220.164.216',
        accept_language => 'en-us,en;q=0.5',
        host => 'dev-partner.yandex.ru',
        my_cookie => 'YycCAAM2AQEA', # необязательный параметр
    );

Возвращает скаляр с языком пользователя, например 'ru' или 'en'.

=cut

sub simple_lang_detect {
    my ($self, %opts) = @_;

    $self->models->validator->check(
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {
                ip              => {},
                accept_language => {},
                host            => {},
                my_cookie       => {optional => TRUE},
                pass_language   => {optional => TRUE},
            }
        },
        throw_internal_error => TRUE
    );

    my $regions = $self->models->api_geobase->get_cached_region_parents(ip => $opts{'ip'});

    my $user_lang = $self->models->lang_detect->get_user_lang(
        regions         => $regions,
        supported       => [sort keys(%{$self->models->get_option('locales')})],
        accept_language => $opts{'accept_language'},
        pass_language   => $opts{'pass_language'},
        host            => $opts{'host'},
        cookie          => $opts{'my_cookie'},
        default         => 'ru',
    );

    $self->models->set_app_locale($user_lang);

    return $user_lang;
}

sub check_params {
    my ($self, @params) = @_;

    my %params = map {$_ => 1} qw(resource apikey all_locales), @params;

    my @bad_params = sort grep {!$params{$_}} keys %{$self->req->params->to_hash()};

    if (@bad_params) {
        throw Exception::IncorrectParams gettext('Incorrect parameters: %s', join(', ', @bad_params));
    }

    if ($params{'public_id'}) {
        my $resource = $self->param('resource');

        throw Exception::Conflict gettext('Incorrect id: %s', $self->param('public_id'))
          unless $self->models->$resource->api_check_public_id($self->param('public_id'));
    }

    return 1;
}

sub _fix_filter {
    my ($filter) = @_;

    if (ref($filter) eq 'HASH' && %$filter) {
        my @and = ();

        foreach my $field_name (sort keys(%$filter)) {
            my $fix_filter = [];

            if ($field_name =~ /\./) {
                my @fields = reverse(split(/\./, $field_name));
                if (ref($filter->{$field_name}) eq 'ARRAY') {
                    $fix_filter = [shift(@fields), @{$filter->{$field_name}}];
                } elsif (ref($filter->{$field_name}) eq '') {
                    $fix_filter = [shift(@fields), "=", $filter->{$field_name}];
                } else {
                    throw gettext('In subfilters use array or a scalar');
                }

                foreach (@fields) {
                    $fix_filter = [$_, 'MATCH', $fix_filter];
                }

            } elsif (
                ref($filter->{$field_name}) eq 'ARRAY' && defined($filter->{$field_name}[0]) && grep {
                    uc($filter->{$field_name}[0]) eq $_
                } (
                    '=', '==', '<>', '!=', '>', '>=', '<', '<=', 'NOT', 'IS', 'IS NOT', 'LIKE', 'NOT LIKE', 'IN',
                    'NOT IN'
                  )
              )
            {
                #{"login":["IN",["vk-team"]]} => ["login","IN",["vk-team"]]
                $fix_filter = [$field_name, @{$filter->{$field_name}}];
            } else {
                #{"login":"vk-team"} => ["login","=","vk-team"]
                $fix_filter = [$field_name, '=', $filter->{$field_name}];
            }

            push(@and, $fix_filter);
        }

        $filter = ['AND', \@and];
    }

    return $filter;
}

sub get_body {
    my ($self, %opts) = @_;

    my $body;

    my $error;
    try {
        my $json_body = $self->req->text;

        $json_body = $opts{'default'} if !$json_body && $opts{'default'};

        utf8::decode($json_body);

        $body = clean_decoded_json_from_boolean_objects_in_place(from_json($json_body));
    }
    catch {
        $error = 1;
    };

    throw Exception::IncorrectParams gettext('Incorrect request body') if $error;

    return $body;
}

sub get_fields {
    my ($self, $resource) = @_;

    $resource //= $self->param('resource');

    my %available_fields = map {$_ => 1} $self->models->$resource->api_available_fields();

    my @fields = split(/,/, $self->param("fields[$resource]") // '');

    my %uniq_fields = ();
    my @bad_fields  = ();
    foreach (@fields) {
        $uniq_fields{$_}++;

        push(@bad_fields, $_) unless $available_fields{$_};
    }

    if (@bad_fields) {
        throw Exception::IncorrectParams gettext('Unexpected fields: %s', join(', ', sort @bad_fields));
    }

    my @dup_fields = sort grep {$uniq_fields{$_} > 1} keys(%uniq_fields);

    if (@dup_fields) {
        throw Exception::IncorrectParams gettext('Following fields duplicated: %s', join(', ', sort @dup_fields));
    }

    return @fields;
}

sub get_filter {
    my ($self) = @_;

    my $param_filter = $self->param('filter');

    my $filter;
    if (defined($param_filter) && length($param_filter) > 0) {
        my $error;
        try {
            $filter = clean_decoded_json_from_boolean_objects_in_place(from_json($param_filter));

            rmap_all {$_ = _fix_filter($_)} $filter;
        }
        catch {
            $error = 1;
        };

        throw Exception::IncorrectParams gettext('Parameter "filter" incorrect') if $error;
    }

    return $filter;
}

sub get_include {
    my ($self, $resource) = @_;

    my @include = split(/,/, $self->param('include') // '');

    if (@include) {
        $resource //= $self->param('resource');

        my $relationships = RestApi::Relationships->relationships($resource);

        if (grep {!$relationships->{$_}} @include) {
            throw Exception::IncorrectParams;
        }
    }

    return @include;
}

sub get_page_number {
    my ($self) = @_;

    my $page_number = $self->param('page[number]') // 1;

    unless ($page_number =~ /^[1-9][0-9]*\z/) {
        throw Exception::IncorrectParams;
    }

    return $page_number;
}

sub get_page_size {
    my ($self) = @_;

    my $page_size = $self->param('page[size]') // 100;

    if (defined($page_size) && $page_size !~ /^[1-9][0-9]*\z/) {
        throw Exception::IncorrectParams;
    }

    throw Exception::IncorrectParams gettext('Parameter page[size] must be <= %s', $MAX_PAGE_SIZE)
      if $page_size > $MAX_PAGE_SIZE;

    return $page_size;
}

sub get_sort {
    my ($self) = @_;

    my $param_sort = $self->param('sort');

    my @sort = ();
    if (defined($param_sort) && length($param_sort) > 0) {
        unless ($param_sort =~ /^[-,a-z_]+\z/) {
            throw Exception::IncorrectParams gettext("Incorrent sort param \"%s\"", $param_sort);
        }

        foreach my $sort_field (split(/,/, $param_sort)) {
            push(@sort, $sort_field =~ s/^-// ? [$sort_field, 1] : [$sort_field, 0]);
        }
    }

    return @sort;
}

sub get_meta {
    my ($self) = @_;

    my %available_meta = (total => 1);

    my %meta           = ();
    my @incorrect_meta = ();

    foreach (split(/,/, $self->param('meta') // '')) {
        push(@incorrect_meta, $_) unless $available_meta{$_};

        $meta{$_}++;
    }

    throw Exception::IncorrectParams gettext('Unavailable params into "meta": %s', join(', ', @incorrect_meta))
      if @incorrect_meta;

    my @duplicate = sort grep {$meta{$_} > 1} keys(%meta);
    throw Exception::IncorrectParams gettext('Duplicate params into "meta": %s', join(', ', @duplicate)) if @duplicate;

    return \%meta;
}

sub get_typed_value {
    my ($self, $MODELS_FIELDS, $resource, $field, $value, $field_name) = @_;

    return $value unless defined($value);

    if (!defined($field)) {
        throw
          sprintf("Failed to produce typed value for value of type '%s'. "
              . "Field '%s' is missing in the corresponding model",
            $resource, $field_name // 'undef');
    }

    my $type = $field->{'type'} // 'undef';

    if ($type eq 'string') {
        return "$value";
    } elsif ($type eq 'number') {
        return $value + 0;
    } elsif ($type eq 'boolean') {
        return get_boolean_representation($value);
    } elsif ($type eq 'complex') {
        return exists($field->{'fix_type'}) ? $field->{'fix_type'}->($self->models->$resource, $value) : $value;
    } elsif ($type eq 'array') {
        return [map {$self->get_typed_value($MODELS_FIELDS, $resource, {type => $field->{'sub_type'}}, $_, $field_name)}
              @$value];
    } elsif ($self->models->can($type)) {
        my $model = $self->models->$type();
        $MODELS_FIELDS->{$type} //= $model->get_model_fields();

        map {
            $value->{$_} =
              $self->get_typed_value($MODELS_FIELDS, $type, $MODELS_FIELDS->{$type}{$_}, $value->{$_}, $_)
          }
          keys(%$value);

        return $value;
    } else {
        throw gettext('Field "%s" has unknown type "%s"', $field_name // 'undef', $type);
    }
}

sub is_available_resource {
    my ($self, $resource, $method, $base) = @_;

    if ($self->stash('authorization') && $self->stash('authorization')->{'is_authed'}) {
        my $resources = $self->models->resources->get_api_resources($resource);

        return
             $resources->{$resource}
          && (grep {$_ eq $method} @{$resources->{$resource}->{'methods'}})
          && $self->models->can($resource)
          && $self->models->$resource->isa($base);
    } else {
        return $self->models->can($resource) && $self->models->$resource->isa($base);
    }
}

sub jsonapi_validate_create {
    my ($self, $captures) = @_;

    my @errors;

    try {
        my $body = $self->get_body();

        if (!exists($body->{'data'}{'type'}) || !defined($body->{'data'}{'type'})) {
            push(
                @errors,
                $self->get_error_object(
                    ERROR__VALIDATION(),
                    source => {pointer => "/data"},
                    detail => gettext('Expected "type"'),
                ),
            );

            return FALSE;
        }

        if ($body->{'data'}{'type'} ne $captures->{'resource'}) {
            push(
                @errors,
                $self->get_error_object(
                    ERROR__CONFLICT(),
                    source => {pointer => "/data/type"},
                    detail => gettext('Type does not match resource')
                )
            );

            return FALSE;
        }

        if (exists($body->{'data'}{'id'})) {
            push(
                @errors,
                $self->get_error_object(
                    ERROR__FORBIDDEN(),
                    source => {pointer => "/data/id"},
                    detail => gettext('Unsupported request to create a resource with a client-generated ID')
                )
            );

            return FALSE;
        }
    }
    catch Exception::IncorrectParams with {
        push(@errors, $self->get_error_object(ERROR__PARAMS, detail => shift->message()));
    }
    catch {
        push(@errors, $self->get_error_object(ERROR__INTERNAL(), detail => $self->safe_exception_message(shift)));
    };

    return 1 unless @errors;

    $self->set_http_code($errors[0]->{'id'});
    $self->render(json => {errors => \@errors});
    return 0;
}

sub jsonapi_validate_update {
    my ($self, $captures) = @_;

    my @errors;

    try {
        my $body = $self->get_body();

        if (!exists($body->{'data'}{'type'}) || !defined($body->{'data'}{'type'})) {
            push(
                @errors,
                $self->get_error_object(
                    ERROR__VALIDATION(),
                    source => {pointer => "/data"},
                    detail => gettext('Expected "type"')
                )
            );

            return FALSE;
        }

        if ($body->{'data'}{'type'} ne $captures->{'resource'}) {
            push(
                @errors,
                $self->get_error_object(
                    ERROR__CONFLICT(),
                    source => {pointer => "/data/type"},
                    detail => gettext('Type does not match resource')
                )
            );

            return FALSE;
        }

        if (!exists($body->{'data'}{'id'}) || !defined($body->{'data'}{'id'})) {
            push(
                @errors,
                $self->get_error_object(
                    ERROR__VALIDATION(),
                    source => {pointer => "/data"},
                    detail => gettext('Expected "id"')
                )
            );

            return FALSE;
        }

        if ($body->{'data'}{'id'} ne $captures->{'public_id'}) {
            push(
                @errors,
                $self->get_error_object(
                    ERROR__VALIDATION(),
                    source => {pointer => "/data/id"},
                    detail => gettext('Incorrect id')
                )
            );

            return FALSE;
        }
    }
    catch Exception::IncorrectParams with {
        push(@errors, $self->get_error_object(ERROR__PARAMS, detail => shift->message()));
    }
    catch {
        push(@errors, $self->get_error_object(ERROR__INTERNAL(), detail => $self->safe_exception_message(shift)));
    };

    return 1 unless @errors;

    $self->set_http_code($errors[0]->{'id'});
    $self->render(json => {errors => \@errors});
    return 0;
}

sub jsonapi_validate_update_multiple {
    my ($self, $captures) = @_;

    my @errors;
    try {
        my $body = $self->get_body();

        unless (ref($body->{'data'}) eq 'ARRAY') {
            push(
                @errors,
                $self->get_error_object(
                    ERROR__VALIDATION(),
                    source => {pointer => '/data'},
                    detail => gettext('Expected array')
                )
            );

            return FALSE;
        }

        if (@{$body->{'data'}} > $MAX_BULK_COUNT) {
            push(
                @errors,
                $self->get_error_object(
                    ERROR__VALIDATION(),
                    source => {pointer => '/data'},
                    detail => gettext('Object count must be no more than %s', $MAX_BULK_COUNT),
                )
            );

            return FALSE;
        }

        my $resource = $captures->{'resource'};

        for (my $i = 0; $i < @{$body->{'data'}}; $i++) {
            my $row = $body->{'data'}[$i];

            if (!exists($row->{'type'}) || !defined($row->{'type'})) {
                push(
                    @errors,
                    $self->get_error_object(
                        ERROR__VALIDATION(),
                        source => {pointer => "/data/$i"},
                        detail => gettext('Expected "type"')
                    )
                );

                next;
            }

            if ($row->{'type'} ne $resource) {
                push(
                    @errors,
                    $self->get_error_object(
                        ERROR__CONFLICT(),
                        source => {pointer => "/data/$i"},
                        detail => gettext('Type does not match resource')
                    )
                );

                next;
            }

            if (!exists($row->{'id'}) || !defined($row->{'id'})) {
                push(
                    @errors,
                    $self->get_error_object(
                        ERROR__VALIDATION(),
                        source => {pointer => "/data/$i"},
                        detail => gettext('Expected "id"')
                    )
                );

                next;
            }
        }
    }
    catch Exception::IncorrectParams with {
        push(@errors, $self->get_error_object(ERROR__PARAMS, detail => shift->message()));
    }
    catch {
        push(@errors, $self->get_error_object(ERROR__INTERNAL(), detail => $self->safe_exception_message(shift)));
    };

    return 1 unless @errors;

    $self->set_http_code($errors[0]->{'id'});
    $self->render(json => {errors => \@errors});
    return 0;
}

# Удалить метод после перехода на bionic
sub _escape_url {
    my ($self, $str) = @_;

    my $uri = URI->new($str);

    my $query = $uri->query();
    if (defined($query)) {
        $query =~ s/,/%2C/g;
        $query =~ s/:/%3A/g;

        if (length($query) > 0) {
            $str =~ s/\?.+$/?$query/;
        }
    }

    return $str;
}

TRUE;
