package RestApi::Controller::API;

use Mojo::Base qw(RestApi::Controller);

use RestApi::Errors;

use qbit;

use B qw(svref_2object);

use Exception::Validation;
use Exception::Validation::BadArguments;
use Exception::Validation::BadArguments::InvalidJSON;
use Exception::Denied;
use Exception::NotImplemented;
use Exception::IncorrectParams;

sub check_data_keys {
    my ($self, $data, $valid_keys) = @_;

    my %params_cnt = map {$_ => 0} keys %$valid_keys;

    my @bad_params;
    foreach (keys %$data) {
        if (!exists($params_cnt{$_})) {
            push @bad_params, $_;
        } else {
            $params_cnt{$_}++;
        }
    }

    throw Exception::IncorrectParams gettext('Unknown fields: %s', join(', ', @bad_params))
      if @bad_params;

    my @missing_params = grep {!$params_cnt{$_} && $valid_keys->{$_}} keys %$valid_keys;
    throw Exception::IncorrectParams gettext('Missing fields: %s', join(', ', @missing_params))
      if @missing_params;

    return 1;
}

sub api_call {
    my ($self) = @_;

    my $method = $self->param('method');

    my @data;
    my @errors;
    my $body_params;
    try {
        throw Exception::NotImplemented gettext('Method "%s" is not supported', $method)
          unless $self->can($method);

        if ('get' eq lc($self->req->method)) {
            $body_params = {};
        } else {
            my $body = $self->get_body(default => '{}');
            $self->check_data_keys($body, {data => 1});
            $body_params = $body->{'data'};
        }

        throw Exception::Validation::BadArguments gettext('"data" must be a hash ref')
          unless ref($body_params) eq 'HASH';

        @data = $self->$method(%{$body_params});
    }
    catch Exception::IncorrectParams with {
        push(@errors, $self->get_error_object(ERROR__PARAMS, detail => shift->message()));
    }
    catch Exception::Validation::BadArguments::InvalidJSON with {
        push(
            @errors,
            $self->get_error_object(
                ERROR__VALIDATION, detail => gettext("Invalid json in param '%s': %s", 'data', shift->message())
            )
        );
    }
    catch Exception::Validation with {
        my ($exception) = @_;

        my $validation_errors =
          $exception->isa('Exception::Validator::Errors')
          ? from_json($exception->message())
          : [{name => [], messages => [$exception->message()]}];

        foreach my $err (@$validation_errors) {
            push(
                @errors,
                $self->get_error_object(ERROR__VALIDATION,
                    source => {pointer => "/data/" . join('/', @{$err->{'name'}})},
                    detail => join("\n", @{$err->{'messages'}})
                )
            );
        }
    }
    catch Exception::NotImplemented with {
        push(@errors, $self->get_error_object(ERROR__NOT_IMPLEMENTED, detail => shift->message()));
    }
    catch Exception::Denied with {
        push(@errors, $self->get_error_object(ERROR__FORBIDDEN, detail => shift->message()));
    }
    catch {
        push(@errors, $self->get_error_object(ERROR__INTERNAL, detail => $self->safe_exception_message(shift)));
    };

    if (@errors) {
        $self->set_http_code($errors[0]->{'id'});
        return $self->render(json => {errors => \@errors});
    }

    return $self->render(json => {data => [@data]});
}

sub MODIFY_CODE_ATTRIBUTES {
    my ($package, $sub, @attrs) = @_;

    my @unknown_attrs = ();

    foreach my $attr (@attrs) {
        if ($attr =~ /^CACHECONTROL\s*\(\s*(\S*)\s*\)$/) {
            $package->_set_method_attr($sub, cachecontrol => $1);
        } else {
            push(@unknown_attrs, $attr);
        }
    }

    return @unknown_attrs;
}

sub _set_method_attr {
    my ($package, $sub, $name, $value) = @_;

    my $pkg_stash = package_stash($package);
    $pkg_stash->{'__API_ATTRS__'} = {} unless exists($pkg_stash->{'__API_ATTRS__'});

    my $cv = svref_2object($sub);
    my $gv = $cv->GV;

    $pkg_stash->{'__API_ATTRS__'}{$package, $gv->NAME}{$name} = $value;
}

1;
