package RestApi::Controller::API::BlockPresetApplier;

use Mojo::Base qw(RestApi::Controller::API);

use qbit;

use Exception::Validation::BadArguments;
use Exception::Validator::Errors;

use Partner::BlockPreset;

use Utils::Logger qw(INFOF);

my $MAX_BLOCKS_COUNT = 100;

sub check_blocks {
    my ($self, %opts) = @_;

    $self->check_data_keys(\%opts, {accessor => 1, filter => 1, settings => 1});

    my $accessor = $opts{'accessor'};
    my $filter   = $opts{'filter'};
    my $settings = $opts{'settings'};

    throw Exception::Validation::BadArguments gettext('Accessor "%s" not supported', $accessor)
      if !$self->models->can($accessor) || !$self->models->$accessor->isa('Application::Model::Block');

    my $model = $self->models->$accessor;

    my $preset = Partner::BlockPreset->build_and_validate($self->models, $settings);

    if (my $errors = $preset->get_errors()) {
        throw Exception::Validator::Errors [map {{name => ['settings', @{$_->{'path'}}], messages => $_->{'msgs'}}}
              @$errors];
    }

    my $blocks = $self->models->block_presets->check_blocks(
        model  => $model,
        filter => $filter,
        preset => $preset,
    );

    #TODO: после Машиных изменений возвращать массив?
    return $blocks;
}

sub apply {
    my ($self, %opts) = @_;

    $self->check_data_keys(\%opts, {accessor => 1, public_ids => 1, settings => 1});

    my $accessor = $opts{'accessor'};
    throw Exception::Validation::BadArguments gettext('Accessor "%s" not supported', $accessor)
      if !$self->models->can($accessor) || !$self->models->$accessor->isa('Application::Model::Block');

    my $public_ids = $opts{'public_ids'};
    throw Exception::Validation::BadArguments gettext('"public_ids" must be a non empty list')
      if ref($public_ids) ne 'ARRAY' || !@$public_ids;

    throw Exception::Validation::BadArguments gettext('Max blocks count: %d', $MAX_BLOCKS_COUNT)
      if @$public_ids > $MAX_BLOCKS_COUNT;

    my $operation_id = sprintf('%s_%d_%d', $self->models->get_option('hostname'), $$, time);
    INFOF 'APPLY PRESET TO BLOCKS: %s %s', $operation_id,
      to_json({%opts, user_id => $self->models->get_option('cur_user')->{'id'}});

    my $settings = $opts{'settings'};
    my $preset = Partner::BlockPreset->build_and_validate($self->models, $settings);

    if (my $errors = $preset->get_errors()) {
        throw Exception::Validator::Errors [map {{name => ['settings', @{$_->{'path'}}], messages => $_->{'msgs'}}}
              @$errors];
    }

    my $model = $self->models->$accessor;

    my $result = $self->models->block_presets->apply(
        model      => $model,
        public_ids => $public_ids,
        preset     => $preset,
    );

    INFOF 'OPERATION RESULT: %s %s', $operation_id, to_json($result);

    return $result;
}

TRUE;
