package RestApi::Controller::API::SelfEmployed;

use qbit;

use base qw(RestApi::Controller::API);

use Exception::SelfemployedOeBS;
use Exception::SelfemployedOeBS::Unregistered;
use PiConstants qw(
  $SELFEMPLOYED_COOPERATION_FORM
  $SELFEMPLOYED_STATUS_NOT
  $SELFEMPLOYED_STATUS_AVAILABLE
  $SELFEMPLOYED_STATUS_READY
  $GAME_OFFER_SERVICE
  );

sub model_accessor { }

my $REQUEST_STATUS_INPROGRESS = 'IN_PROGRESS';
my $REQUEST_STATUS_FAILED     = 'FAILED';
my $REQUEST_STATUS_COMPLETED  = 'COMPLETED';

my $MSG_TAXPAYER_ALREADY_BOUND = 'TAXPAYER_ALREADY_BOUND';
my $SPENDABLE_COLLATERAL_TYPE  = 7090;

sub send_self_employed_request {
    my ($self, %opts) = @_;

    my $user = $self->_get_partner_info();
    throw Exception::Validation::BadArguments gettext('Partner already has a request in progress: %s',
        $user->{self_employed_request_id})
      if ($user->{self_employed_request_id} && !$opts{force});

    throw Exception::Validation::BadArguments gettext('Partner cooperation_form restricts self-employed')
      unless ($SELFEMPLOYED_COOPERATION_FORM eq ($user->{cooperation_form} // ''));
    throw Exception::Validation::BadArguments gettext('Partner has no INN') unless $user->{inn};

    my $created_request;
    try {
        $created_request = $self->models->api_selfemployed_oebs->bind_by_inn($user->{inn});
        $self->models->users->do_action($user, 'edit', 'self_employed_request_id' => $created_request->{requestId});
    }
    catch Exception::SelfemployedOeBS::Unregistered with {
        throw Exception::Validation::BadArguments gettext('Partner is not self-employed');
    }
    catch Exception::SelfemployedOeBS with {
        my ($e) = @_;
        if ($e->{text} =~ /TAXPAYER_ALREADY_BOUND/) {
            $self->_complete_self_employed($user, $MSG_TAXPAYER_ALREADY_BOUND);
            $created_request = {requestId => $MSG_TAXPAYER_ALREADY_BOUND,};
        } else {
            throw Exception::Validation::BadArguments $e->{text};
        }
    };

    return $created_request;
}

sub check_self_employed_request {
    my ($self) = @_;

    my $user = $self->_get_partner_info();
    throw Exception::Validation::BadArguments gettext('Partner has no request in progress')
      unless $user->{self_employed_request_id};

    my $request_status;
    try {
        if ($SELFEMPLOYED_STATUS_READY eq ($user->{self_employed} // 0)) {
            $request_status = {resultCode => $REQUEST_STATUS_COMPLETED,};
        } else {
            $request_status = $self->models->api_selfemployed_oebs->bind_status($user->{self_employed_request_id});
            if ($REQUEST_STATUS_COMPLETED eq $request_status->{resultCode}) {
                $self->_complete_self_employed($user);
            } elsif ($REQUEST_STATUS_FAILED eq $request_status->{resultCode}) {
                $self->models->users->do_action(
                    $user, 'edit',
                    'self_employed'            => $SELFEMPLOYED_STATUS_AVAILABLE,
                    'self_employed_request_id' => undef,
                );
            }
        }
    }
    catch Exception::SelfemployedOeBS with {
        my ($e) = @_;
        throw Exception::Validation::BadArguments $e->{text};
    };

    return $request_status;
}

sub check_step {
    my ($self) = @_;

    my $step;

    my $user = $self->_get_partner_info();

    unless ($SELFEMPLOYED_COOPERATION_FORM eq ($user->{cooperation_form} // '')
        && $user->{inn})
    {
        $step = 0;
    } else {
        # checking whether offer is accepted, we cannot create collateral otherwise
        my $offer = $self->models->documents->get_contracts(client_id => $user->{client_id}, need_signed_offer_ru => 1);
        unless ($offer) {
            $step = 0;
        } else {
            my $self_employed = $user->{self_employed} // 0;
            if ($user->{self_employed_request_id}) {
                if ($SELFEMPLOYED_STATUS_READY == $self_employed) {
                    # request was completed
                    $step = 4;
                } else {
                    $step = 3;
                }
            } else {
                # partner was checked and was self_employed in oebs
                if ($SELFEMPLOYED_STATUS_AVAILABLE == $self_employed) {
                    $step = 2;
                    # partner wasnt checked in oebs
                } else {
                    $step = 1;
                }
            }
        }
    }

    return {
        step => $step,
        inn  => $user->{inn},
    };
}

sub _complete_self_employed {
    my ($self, $user, $self_employed_request_id) = @_;

    throw Exception::Validation::BadArguments gettext('Missed required parameter "%s"', 'client_id')
      unless $user->{client_id};

    my $offer = $self->models->documents->get_contracts(client_id => $user->{client_id}, need_signed_offer_ru => 1);

    throw Exception::Validation::BadArguments gettext('Partner has no offer contract')
      unless $offer;

    $self->models->api_balance->create_collateral_selfemployed(
        operator_uid => $user->{id},
        contract_id  => $offer->{Contract}->{contract2_id},
    );

    my @game_offers = $self->models->documents->get_games_contracts($user->{client_id}, live_only => TRUE);
    for my $game_offer (@game_offers) {
        $self->models->api_balance->create_collateral(
            'operator_uid'    => $user->{id},
            'contract_id'     => $game_offer->{Contract}{contract2_id},
            'collateral_type' => $SPENDABLE_COLLATERAL_TYPE,
            'selfemployed'    => 1,
        );
    }

    my %request_id_patch = $self_employed_request_id ? (self_employed_request_id => $self_employed_request_id) : ();

    $self->models->users->do_action($user, 'edit', 'self_employed' => $SELFEMPLOYED_STATUS_READY, %request_id_patch);

    return TRUE;
}

sub _get_partner_info {
    my ($self) = @_;

    my $tmp_rights = $self->models->add_tmp_rights('users_view_field__client_id');
    return $self->models->users->get($self->models->get_option('cur_user'),
        fields => [qw(id client_id inn cooperation_form self_employed self_employed_request_id features)]);
}

1;
