package RestApi::Controller::DBModel;

use Data::Rmap qw(rmap);
use Mojo::Base qw(RestApi::Controller);

use RestApi::Errors;
use RestApi::Relationships;

use Exception::DB::DuplicateEntry;
use Exception::Denied;
use Exception::IncorrectParams;
use Exception::Multistate::BadAction;
use Exception::Multistate::NotFound;
use Exception::NotFound;
use Exception::Validation;

use qbit;

sub name_prefix_for_get_abs {
    return 'dbmodel__';
}

sub add {
    my ($self) = @_;

    my $resource = $self->param('resource');

    my (@errors, $public_id);

    try {
        $self->check_params();

        my $body = $self->get_body(default => '{"data":{"attributes":{}}}');
        $self->check_bad_fields($self->models->$resource, %{$body->{'data'}{'attributes'}});
        $public_id = $self->models->$resource->api_add(%{$body->{'data'}{'attributes'}});
    }
    catch Exception::IncorrectParams with {
        push(@errors, $self->get_error_object(ERROR__PARAMS, detail => shift->message()));
    }
    catch Exception::Validation with {
        my ($exception) = @_;

        my $validation_errors =
          $exception->isa('Exception::Validator::Errors')
          ? from_json($exception->message())
          : [{name => [], messages => [$exception->message()]}];

        foreach my $err (@{$validation_errors}) {
            push(
                @errors,
                $self->get_error_object(
                    ERROR__VALIDATION,
                    source => {pointer => "/data/attributes/" . join('/', @{$err->{'name'}})},
                    detail => join("\n", @{$err->{'messages'}})
                )
            );
        }
    }
    catch Exception::Denied with {
        push(@errors, $self->get_error_object(ERROR__FORBIDDEN, detail => shift->message()));
    }
    catch Exception::DB::DuplicateEntry with {
        push(@errors,
            $self->get_error_object(ERROR__CONFLICT, detail => gettext('Resource with this settings already exists')));
    }
    catch {
        push(@errors, $self->get_error_object(ERROR__INTERNAL, detail => $self->safe_exception_message(shift)));
    };

    if (@errors) {
        $self->set_http_code($errors[0]->{'id'});

        return $self->render(json => {errors => \@errors});
    }

    $self->res->code(201);
    $self->res->headers->header(
        Location => $self->get_abs_url(
            $self->name_prefix_for_get_abs() . '__resource_get',
            resource  => $resource,
            public_id => $public_id
        )
    );

    $self->render(
        json => {data => $self->get_resource_objets($resource, [{public_id => $public_id}], [], {}, one_object => TRUE)}
    );
}

sub edit {
    my ($self) = @_;

    my $resource  = $self->param('resource');
    my $public_id = $self->param('public_id');

    my (@errors, $result);
    try {
        $self->check_params(qw(public_id));

        my $body = $self->get_body(default => '{"data":{"attributes":{}}}');
        my %patch = exists $body->{'data'}{'attributes'} ? %{$body->{'data'}{'attributes'}} : ();
        $self->check_bad_fields($self->models->$resource, %patch);

        $self->models->$resource->api_edit($public_id, %patch);

        my $object = $self->models->$resource->api_get($public_id, fields => ['public_id', keys(%patch)]);

        my $last_fields = [grep {$_ ne 'public_id'} sort keys %{$self->models->$resource->last_fields()}];
        $result = {
            data => $self->get_resource_objets(
                $resource, [$object], $last_fields,
                {},
                one_object      => 1,
                no_object_links => 1
            ),
            links => {
                self => $self->get_abs_url(
                    $self->name_prefix_for_get_abs() . '__resource_get',
                    resource  => $resource,
                    public_id => $public_id,
                ),
            },
            meta => {fields => $last_fields,},
        };
    }
    catch Exception::IncorrectParams with {
        push(@errors, $self->get_error_object(ERROR__PARAMS, detail => shift->message()));
    }
    catch Exception::Conflict with {
        push(@errors, $self->get_error_object(ERROR__CONFLICT, detail => shift->message()));
    }
    catch Exception::Multistate::NotFound with {
        push(@errors, $self->get_error_object(ERROR__CONFLICT, detail => gettext('Not found %s', $public_id)));
    }
    catch Exception::Multistate::BadAction with {
        push(@errors, $self->get_error_object(ERROR__PARAMS, detail => gettext('Cannot do action "%s"', 'edit')));
    }
    catch Exception::Validation with {
        my ($exception) = @_;

        my $validation_errors =
          $exception->isa('Exception::Validator::Errors')
          ? from_json($exception->message())
          : [{name => [], messages => [$exception->message()]}];

        foreach my $err (@{$validation_errors}) {
            push(
                @errors,
                $self->get_error_object(
                    ERROR__VALIDATION,
                    source => {pointer => "/data/attributes/" . join('/', @{$err->{'name'}})},
                    detail => join("\n", @{$err->{'messages'}})
                )
            );
        }
    }
    catch Exception::Denied with {
        push(@errors, $self->get_error_object(ERROR__FORBIDDEN, detail => shift->message()));
    }
    catch {
        push(@errors, $self->get_error_object(ERROR__INTERNAL, detail => $self->safe_exception_message(shift)));
    };

    if (@errors) {
        $self->set_http_code($errors[0]->{'id'});

        return $self->render(json => {errors => \@errors});
    }

    $self->render(json => $result);
}

sub delete {
    my ($self) = @_;

    my $resource         = $self->param('resource');
    my $public_id        = $self->param('public_id');
    my $linked_resource  = $self->param('linked_resource');
    my $is_relationships = defined($linked_resource);

    my @errors;
    try {
        $self->check_params(qw(public_id linked_resource));

        if ($is_relationships) {
            my $relationships = RestApi::Relationships->relationships($resource);

            my %convertor = map {$_->[0] => $_->[1]} @{$relationships->{$linked_resource}};

            my $object = $self->models->$resource->get(
                $public_id,
                fields     => [sort keys(%convertor)],
                query_opts => {assistant_can_edit => 1},
            ) or throw Exception::Conflict gettext('Not found %s', $public_id);

            $self->models->$linked_resource->api_delete({hash_transform($object, [], \%convertor)});
        } else {
            my $object = $self->models->$resource->get(
                $public_id,
                fields     => $self->models->$resource->get_pk_fields(),
                query_opts => {assistant_can_edit => 1},
            ) or throw Exception::Conflict gettext('Not found %s', $public_id);

            $self->models->$resource->api_delete($object);
        }
    }
    catch Exception::IncorrectParams with {
        push(@errors, $self->get_error_object(ERROR__PARAMS, detail => shift->message()));
    }
    catch Exception::Conflict with {
        push(@errors, $self->get_error_object(ERROR__CONFLICT, detail => shift->message()));
    }
    catch Exception::Multistate::NotFound with {
        push(@errors, $self->get_error_object(ERROR__CONFLICT, detail => gettext('Not found %s', $public_id)));
    }
    catch Exception::Multistate::BadAction with {
        push(@errors, $self->get_error_object(ERROR__PARAMS, detail => gettext('Cannot do action "%s"', 'delete')));
    }
    catch Exception::Validation with {
        my ($exception) = @_;

        my $validation_errors =
          $exception->isa('Exception::Validator::Errors')
          ? from_json($exception->message())
          : [{name => [], messages => [$exception->message()]}];

        foreach my $err (@{$validation_errors}) {
            push(
                @errors,
                $self->get_error_object(
                    ERROR__VALIDATION,
                    source => {pointer => "/data/attributes/" . join('/', @{$err->{'name'}})},
                    detail => join("\n", @{$err->{'messages'}})
                )
            );
        }
    }
    catch Exception::Denied with {
        push(@errors, $self->get_error_object(ERROR__FORBIDDEN, detail => shift->message()));
    }
    catch {
        push(@errors, $self->get_error_object(ERROR__INTERNAL, detail => $self->safe_exception_message(shift)));
    };

    if (@errors) {
        $self->set_http_code($errors[0]->{'id'});

        return $self->render(json => {errors => \@errors});
    }

    $self->res->code(204);
    $self->res->headers->header(
        Location => $is_relationships
        ? $self->get_abs_url(
            $self->name_prefix_for_get_abs() . '__delete_relationships',
            resource        => $resource,
            public_id       => $public_id,
            linked_resource => $linked_resource,
          )
        : $self->get_abs_url(
            $self->name_prefix_for_get_abs() . '__delete',
            resource  => $resource,
            public_id => undef,
        )
    );

    return $self->render(text => '');
}

sub add_fields {
    my ($self) = @_;

    my $resource = $self->param('resource');
    my $attributes = $self->param('attributes') // '{}';

    my ($error, $exception, @fields);

    try {
        $self->check_params('attributes');

        my $attr;
        try {
            $attr = from_json($attributes);
        }
        catch {
            throw Exception::IncorrectParams gettext('Value "attributes" must be in JSON format');
        };

        my %no_api_fields = _get_no_api_fields($self->models->$resource);

        @fields = sort grep {!exists $no_api_fields{$_}} keys(%{$self->models->$resource->get_add_fields($attr)});
    }
    catch Exception::IncorrectParams catch Exception::Validation with {
        $error     = ERROR__PARAMS;
        $exception = shift->message();
    }
    catch {
        $error     = ERROR__INTERNAL;
        $exception = $self->safe_exception_message(shift);
    };

    return $self->render($self->get_error($error, $exception))
      if defined($error);

    my $type = $resource . '_add_fields';

    my $resource_meta = $self->models->$resource->api_get_meta() // {};

    $self->render(
        json => {
            data => [map              {{type => $type, id => $_,}} @fields],
            meta => {%$resource_meta, count  => scalar(@fields),},
            links =>
              {self => $self->get_abs_url($self->name_prefix_for_get_abs() . '__add_fields', resource => $resource,),},
        }
    );
}

sub depends {
    my ($self) = @_;

    my $resource = $self->param('resource');

    my ($error, $exception, $fields_depends);

    try {
        $self->check_params();

        $fields_depends = $self->models->$resource->get_fields_depends();
    }
    catch Exception::IncorrectParams with {
        $error     = ERROR__PARAMS;
        $exception = shift->message();
    }
    catch {
        $error     = ERROR__INTERNAL;
        $exception = $self->safe_exception_message(shift);
    };

    return $self->render($self->get_error($error, $exception))
      if defined($error);

    $self->render(
        json => {
            data => {
                type       => $resource . '_depends',
                id         => $resource,
                attributes => $fields_depends
            },
            links =>
              {self => $self->get_abs_url($self->name_prefix_for_get_abs() . '__depends', resource => $resource,),},
        }
    );
}

sub make_params_for_defaults {
    my ($self, $resource) = @_;
    my $params      = $self->models->$resource->get_params_for_defaults;
    my @param_names = keys %$params;
    $self->check_params(@param_names);
    my %opts;
    my %orig;
    for my $name (@param_names) {
        my $param = $params->{$name};
        my $value = $self->param($name);
        $value //= $param->{default};
        $orig{$name} = $value;

        if (defined $value) {
            if ($param->{from_json}) {
                try {
                    $value = from_json($value);
                }
                catch {
                    throw Exception::IncorrectParams gettext('Value "%s" must be in %s format', $name, 'JSON');
                };
            } elsif (my $split = $param->{split_by}) {
                $value = [split $split, $value];
            }
            if (my $ref_type = $param->{ref_type}) {
                throw Exception::IncorrectParams gettext('Parameter "%s" must be %s', $name, $ref_type)
                  if ref($value) ne $ref_type;
            }
        }
        $opts{$name} = $value;
    }
    return (\%opts, \%orig);
}

sub defaults {
    my ($self) = @_;

    my $resource = $self->param('resource');

    my ($error, $exception, $fields_defaults);

    my ($opts, $orig);
    try {
        ($opts, $orig) = $self->make_params_for_defaults($resource);
        $self->models->$resource->fix_params_for_defaults($opts);
        $fields_defaults = $self->models->$resource->get_fields_defaults($opts);
        rmap {$_ = "$_" if defined($_) && !ref($_) && !Internals::SvREADONLY($_)}
        map    {$fields_defaults->{$_}}
          grep {defined $fields_defaults->{$_}}
          qw(
          custom_bk_options
          dsps_available
          dsps_default
          page_id
          strategy
          );
    }
    catch Exception::IncorrectParams catch Exception::Validation with {
        $error     = ERROR__PARAMS;
        $exception = shift->message();
    }
    catch {
        $error     = ERROR__INTERNAL;
        $exception = $self->safe_exception_message(shift);
    };

    return $self->render($self->get_error($error, $exception))
      if defined($error);

    my $url = $self->url_for($self->name_prefix_for_get_abs() . '__defaults', resource => $resource)->to_abs;
    $url->scheme($self->req->env->{'HTTP_X_REAL_SCHEME'});
    $url->port($self->req->env->{'HTTP_X_REAL_PORT'});

    $self->render(
        json => {
            data => {
                type       => $resource . '_defaults',
                id         => $resource,
                attributes => $fields_defaults
            },
            links => {self => $self->get_abs_url_with_params($url, %$orig)}
        }
    );
}

sub do_action {
    my ($self) = @_;

    my $resource  = $self->param('resource');
    my $public_id = $self->param('public_id');
    my $action    = $self->param('operation');
    my (@errors, $result);

    try {
        $self->check_params(qw(public_id operation));

        my $body = $self->get_body(default => '{}');
        if ('edit' eq $action) {
            $self->check_bad_fields($self->models->$resource, %$body);
        }

        $self->models->$resource->do_action($public_id, $action, %$body);

        my $object = $self->models->$resource->api_get($public_id, fields => [qw(public_id actions)]);

        my $last_fields = [sort grep {$_ ne 'public_id'} keys %{$self->models->$resource->last_fields()}];
        $result = {
            data => $self->get_resource_objets(
                $resource, [$object], $last_fields,
                {},
                one_object      => 1,
                no_object_links => 1
            ),
            links => {
                self => $self->get_abs_url(
                    $self->name_prefix_for_get_abs() . '__resource_get',
                    resource  => $resource,
                    public_id => $public_id,
                ),
            },
            meta => {fields => $last_fields},
        };
    }
    catch Exception::IncorrectParams with {
        push(@errors, $self->get_error_object(ERROR__PARAMS, detail => shift->message()));
    }
    catch Exception::Conflict with {
        push(@errors, $self->get_error_object(ERROR__CONFLICT, detail => shift->message()));
    }
    catch Exception::Validation with {
        my ($exception) = @_;

        my $validation_errors =
          $exception->isa('Exception::Validator::Errors')
          ? from_json($exception->message())
          : [{name => [], messages => [$exception->message()]}];

        foreach my $err (@{$validation_errors}) {
            push(
                @errors,
                $self->get_error_object(
                    ERROR__VALIDATION,
                    source => {pointer => "/" . join('/', @{$err->{'name'}})},
                    detail => join("\n", @{$err->{'messages'}})
                )
            );
        }
    }
    catch Exception::Denied with {
        push(@errors, $self->get_error_object(ERROR__FORBIDDEN, detail => shift->message()));
    }
    catch {
        push(@errors, $self->get_error_object(ERROR__INTERNAL, detail => $self->safe_exception_message(shift)));
    };

    if (@errors) {
        $self->set_http_code($errors[0]->{'id'});

        return $self->render(json => {errors => \@errors});
    }

    $self->render(json => $result);
}

sub get_key {
    my ($self, $row, $fields) = @_;

    return join($;, map {$row->{$_} // 'undef'} @$fields);
}

sub get_resource_objets {
    my ($self, $resource, $data, $fields, $relationships_data, %opts) = @_;

    my $MODELS_FIELDS = {};

    $MODELS_FIELDS->{$resource} = $self->models->$resource->get_model_fields();

    my $relationships = RestApi::Relationships->relationships($resource);

    my @result = ();

    my $url_resource_get =
      $self->get_abs_for_sprintf($self->name_prefix_for_get_abs() . '__resource_get', qw(resource public_id));
    my $url_relationships = $self->get_abs_for_sprintf($self->name_prefix_for_get_abs() . '__relationships',
        qw(resource public_id linked_resource));
    my $url_linked_resource = $self->get_abs_for_sprintf($self->name_prefix_for_get_abs() . '__linked_resource',
        qw(resource public_id linked_resource));

    foreach my $model (keys(%$relationships_data)) {
        if (@{$relationships_data->{$model}{'data'}} == 0) {
            next;
        } elsif (@$data == 1) {
            my $key = $self->get_key($data->[0], [map {$_->[0]} @{$relationships->{$model}}]);

            $relationships_data->{$model}{'group_data'}{$key} = $relationships_data->{$model}{'data'};
        } else {
            foreach my $elem (@{$relationships_data->{$model}{'data'}}) {
                my $key = $self->get_key($elem, [map {$_->[1]} @{$relationships->{$model}}]);

                push(@{$relationships_data->{$model}{'group_data'}{$key}}, $elem);
            }
        }
    }

    my $relationships_cache = {};

    foreach my $row (@$data) {
        my $type = exists($row->{'__MODEL_TYPE__'}) ? $row->{'__MODEL_TYPE__'} : $resource;

        $relationships_cache->{$type} //= RestApi::Relationships->relationships($type);

        my $relationships_value = {};

        foreach my $model (keys(%{$relationships_cache->{$type}})) {
            my @related_data = ();
            if (exists($relationships_data->{$model}) && @{$relationships_data->{$model}{'data'}}) {
                my $key = $self->get_key($row, [map {$_->[0]} @{$relationships->{$model}}]);

                push(@related_data, @{$relationships_data->{$model}{'group_data'}{$key}})
                  if exists($relationships_data->{$model}{'group_data'}{$key});
            }

            $relationships_value->{$model} = {
                links => {
                    self    => sprintf($url_relationships,   $type, $row->{'public_id'}, $model),
                    related => sprintf($url_linked_resource, $type, $row->{'public_id'}, $model),
                },
                (
                    @related_data
                    ? (
                        data => [
                            map {
                                {
                                    type => exists($_->{'__MODEL_TYPE__'}) ? $_->{'__MODEL_TYPE__'} : $model,
                                    id => "$_->{'public_id'}"
                                }
                              } @related_data
                        ]
                      )
                    : ()
                ),
            };
        }

        my $attributes = {};

        foreach my $field (@$fields) {
            next unless $row->{'available_fields'}{$field};

            $attributes->{$field} =
              $self->get_typed_value($MODELS_FIELDS, $resource, $MODELS_FIELDS->{$resource}{$field},
                $row->{$field}, $field);
        }

        push(
            @result,
            {
                type       => $type,
                id         => "$row->{'public_id'}",
                attributes => $attributes,
                (
                    $opts{'no_object_links'}
                    ? ()
                    : (links => {self => sprintf($url_resource_get, $type, $row->{'public_id'}),})
                ),
                relationships => $relationships_value,
            }
        );
    }

    return $opts{'one_object'} ? $result[0] : \@result;
}

sub get_response_for_collection {
    my ($self, %opts) = @_;

    my $resource = delete($opts{'resource'});

    my $fields = delete($opts{'fields'});
    my $filter = delete($opts{'filter'});
    my @sort   = @{delete($opts{'sort'})};
    my $meta   = delete($opts{'meta'});

    my $limit       = delete($opts{'limit'});
    my $page_number = delete($opts{'page_number'});

    my @include = @{delete($opts{'include'})};

    my $relationships = RestApi::Relationships->relationships($resource);

    my %specific_fields = (public_id => 1);
    foreach my $model (@include) {
        map {$specific_fields{$_->[0]} = 1} grep {!in_array($_->[0], $fields)} @{$relationships->{$model}};
    }

    my $all = $self->models->$resource->api_get_all(
        fields => [sort keys(%specific_fields), grep {!$specific_fields{$_}} @$fields],
        (defined($limit) ? (limit => $limit, offset => ($page_number - 1) * $limit) : ()),
        (@sort            ? (order_by  => \@sort)  : ()),
        (defined($filter) ? (filter    => $filter) : ()),
        ($meta->{'total'} ? (calc_rows => 1)       : ()),
    );

    my $last_fields = $self->models->$resource->last_fields();

    my $filter_simple_fields = $self->models->$resource->get_db_filter_simple_fields();

    # этот map можно будет выкинуть, когда в АПИ
    # get_db_filter_simple_fields будет плоским списком
    $filter_simple_fields = [
        sort {$a->{'name'} cmp $b->{'name'}}
        map {ref($_) eq 'ARRAY' ? @$_ : $_} @$filter_simple_fields
    ];

    my $filter_fields = $self->models->$resource->get_db_filter_fields();

    foreach my $def (@$filter_simple_fields) {
        my $filter_definition = $self->_get_filter_definition([split(/\./, $def->{'name'})], $filter_fields);

        $def->{'type'} = $filter_definition->{'type'};

        if (exists($filter_definition->{'values'})) {
            if (ref($filter_definition->{'values'}) eq 'HASH') {
                $def->{'values'} = [
                    map {{id => $_ . "", label => $filter_definition->{'values'}{$_}}}
                    sort keys(%{$filter_definition->{'values'}})
                ];
            } else {
                foreach (sort {$a->{'id'} cmp $b->{'id'}} @{$filter_definition->{'values'}}) {
                    delete($_->{'key'});
                    $_->{'id'} .= '';

                    push(@{$def->{'values'}}, $_);
                }
            }
        }
    }

    my $meta_fields = [grep {!$specific_fields{$_}} sort keys(%$last_fields)];

    my $resource_meta = $self->models->$resource->api_get_meta() // {};

    my $result->{'meta'} = {
        count => scalar(@$all),
        ($meta->{'total'} ? (found_rows => $self->models->$resource->found_rows() + 0) : ()),
        fields  => $meta_fields,
        filters => $filter_simple_fields,
        %$resource_meta
    };

    my $relationships_data = {};
    if (@include) {
        my %filter       = ();
        my $uniq_filters = {};

        foreach my $row (@$all) {
            foreach my $model (@include) {
                my $key = $self->get_key($row, [map {$_->[0]} @{$relationships->{$model}}]);

                unless ($uniq_filters->{$model}{$key}) {
                    push(@{$filter{$model}}, {map {$_->[1] => $row->{$_->[0]}} @{$relationships->{$model}}});

                    $uniq_filters->{$model}{$key} = 1;
                }
            }
        }

        foreach my $model (@include) {
            my %related_specific_fields = (public_id => 1);
            map {$related_specific_fields{$_->[1]} = 1} @{$relationships->{$model}};

            $relationships_data->{$model}{'fields'} = [$self->get_fields($model)];

            my $relationships_fields =
              array_uniq(keys(%related_specific_fields), @{$relationships_data->{$model}{'fields'}});

            $relationships_data->{$model}{'data'} =
              $filter{$model}
              ? $self->models->$model->api_get_all(
                fields => $relationships_fields,
                filter => ['OR', $filter{$model}],
              )
              : [];
        }
    }

    $result->{'data'} = $self->get_resource_objets($resource, $all, $meta_fields, $relationships_data);

    $result->{'links'} = {
        (
            $meta->{'total'} ? $self->get_links_for_collections($limit, $page_number, $result->{'meta'}{'found_rows'})
            : ()
        ),
        (
            $self->is_available_resource($resource, 'POST', 'RestApi::DBModel')
            ? (add_fields =>
                  $self->get_abs_url($self->name_prefix_for_get_abs() . '__add_fields', resource => $resource,))
            : ()
        ),
    };

    $result->{'included'} = [
        map {
            @{
                $self->get_resource_objets(
                    $_,
                    $relationships_data->{$_}{'data'},
                    $relationships_data->{$_}{'fields'}, {},
                )
              }
          } @include
      ]
      if @include;

    return $result;
}

sub linked_resource {
    my ($self) = @_;

    my $resource        = $self->param('resource');
    my $linked_resource = $self->param('linked_resource');

    my $relationships = RestApi::Relationships->relationships($resource);

    return $self->render($self->get_error(ERROR__NOT_FOUND))
      unless exists($relationships->{$linked_resource});

    my ($error, $exception, $result);

    try {
        my @include = $self->get_include($linked_resource);

        $self->check_params(
            qw(
              linked_resource
              public_id
              filter
              sort
              page[size]
              page[number]
              meta
              include
              ), map {"fields[$_]"} $linked_resource, @include
        );

        my $public_id = $self->param('public_id');

        my $object =
          $self->models->$resource->get($public_id, fields => [map {$_->[0]} @{$relationships->{$linked_resource}}],);

        throw Exception::NotFound unless defined($object);

        my @filter =
          ({map {$_->[1] => $object->{$_->[0]}} @{$relationships->{$linked_resource}}}, $self->get_filter() // (),);

        $result = $self->get_response_for_collection(
            resource    => $linked_resource,
            fields      => [$self->get_fields($linked_resource)],
            filter      => ['AND', \@filter],
            limit       => $self->get_page_size(),
            page_number => $self->get_page_number(),
            sort        => [$self->get_sort()],
            meta        => $self->get_meta(),
            include     => \@include,
        );
    }
    catch Exception::NotFound with {
        $error = ERROR__NOT_FOUND;
    }
    catch Exception::Conflict with {
        $error     = ERROR__CONFLICT;
        $exception = shift->message();
    }
    catch Exception::IncorrectParams catch Exception::Validation::BadArguments with {
        $error     = ERROR__PARAMS;
        $exception = shift->message();
    }
    catch {
        $error     = ERROR__INTERNAL;
        $exception = $self->safe_exception_message(shift);
    };

    return $self->render($self->get_error($error, $exception))
      if defined($error);

    $self->render(json => $result);
}

sub multi_edit {
    my ($self) = @_;

    my $resource = $self->param('resource');

    my (@errors, @result);

    my $body = $self->get_body(default => '{}');

    $self->models->partner_db->begin();

    for (my $i = 0; $i < @{$body->{'data'}}; $i++) {
        my $row   = $body->{'data'}[$i];
        my %patch = %{$row->{'attributes'}};
        $self->check_bad_fields($self->models->$resource, %patch);

        try {
            $self->models->$resource->do_action($row->{'id'}, 'edit', %patch);

            my $object = $self->models->$resource->api_get($row->{'id'}, fields => ['public_id', keys(%patch)]);

            my $last_fields = [grep {$_ ne 'public_id'} keys %{$self->models->$resource->last_fields()}];

            push(@result, $self->get_resource_objets($resource, [$object], $last_fields, {}, one_object => 1,));
        }
        catch Exception::Validation with {
            my ($exception) = @_;

            my $validation_errors =
              $exception->isa('Exception::Validator::Errors')
              ? from_json($exception->message())
              : [{name => [], messages => [$exception->message()]}];

            foreach my $err (@{$validation_errors}) {
                push(
                    @errors,
                    $self->get_error_object(
                        ERROR__VALIDATION,
                        source => {pointer => "/data/$i/attributes/" . join('/', @{$err->{'name'}})},
                        detail => join("\n", @{$err->{'messages'}})
                    )
                );
            }
        }
        catch Exception::Denied with {
            push(
                @errors,
                $self->get_error_object(
                    ERROR__FORBIDDEN,
                    source => {pointer => "/data/$i"},
                    detail => shift->message()
                )
            );
        }
        catch {
            push(
                @errors,
                $self->get_error_object(
                    ERROR__INTERNAL,
                    source => {pointer => "/data/$i"},
                    detail => $self->safe_exception_message(shift)
                )
            );
        };
    }

    if (@errors) {
        $self->models->partner_db->rollback();
        $self->set_http_code($errors[0]->{'id'});
        return $self->render(json => {errors => \@errors});
    }

    $self->models->partner_db->commit();
    $self->render(json => {data => \@result});
}

sub relationships {
    my ($self) = @_;

    my $resource        = $self->param('resource');
    my $linked_resource = $self->param('linked_resource');

    my $relationships = RestApi::Relationships->relationships($resource);

    return $self->render($self->get_error(ERROR__NOT_FOUND)) unless exists($relationships->{$linked_resource});

    my $public_id = $self->param('public_id');

    my ($error, $exception, $data);

    try {
        $self->check_params(
            qw(
              linked_resource
              public_id
              )
        );

        my $object =
          $self->models->$resource->get($public_id, fields => [map {$_->[0]} @{$relationships->{$linked_resource}}]);

        throw Exception::NotFound unless defined($object);

        $data = $self->models->$linked_resource->api_get_all(
            fields => ['public_id'],
            filter => {map {$_->[1] => $object->{$_->[0]}} @{$relationships->{$linked_resource}}}
        );
    }
    catch Exception::NotFound with {
        $error = ERROR__NOT_FOUND;
    }
    catch Exception::Conflict with {
        $error     = ERROR__CONFLICT;
        $exception = shift->message();
    }
    catch Exception::IncorrectParams catch Exception::Validation::BadArguments with {
        $error     = ERROR__PARAMS;
        $exception = shift->message();
    }
    catch {
        $error     = ERROR__INTERNAL;
        $exception = $self->safe_exception_message(shift);
    };

    return $self->render($self->get_error($error, $exception))
      if defined($error);

    $self->render(
        json => {
            links => {
                self => $self->get_abs_url(
                    $self->name_prefix_for_get_abs() . '__relationships',
                    resource        => $resource,
                    public_id       => $public_id,
                    linked_resource => $linked_resource
                ),
                related => $self->get_abs_url(
                    $self->name_prefix_for_get_abs() . '__linked_resource',
                    resource        => $resource,
                    public_id       => $public_id,
                    linked_resource => $linked_resource
                ),
            },
            data => [
                map {
                    {
                        type => exists($_->{'__MODEL_TYPE__'}) ? $_->{'__MODEL_TYPE__'} : $linked_resource,
                        id => $_->{'public_id'}
                    }
                  } @$data
            ]
        }
    );
}

sub resource {
    my ($self) = @_;

    my $resource = $self->param('resource');

    my ($error, $exception, $result);

    try {
        my @include = $self->get_include($resource);

        $self->check_params(
            qw(
              filter
              sort
              page[size]
              page[number]
              include
              meta
              ), map {"fields[$_]"} $resource, @include
        );

        $result = $self->get_response_for_collection(
            resource    => $resource,
            fields      => [$self->get_fields($resource)],
            filter      => $self->get_filter(),
            limit       => $self->get_page_size(),
            page_number => $self->get_page_number(),
            sort        => [$self->get_sort()],
            meta        => $self->get_meta(),
            include     => \@include,
        );
    }
    catch Exception::IncorrectParams catch Exception::Validation::BadArguments with {
        $error     = ERROR__PARAMS;
        $exception = shift->message();
    }
    catch {
        $error     = ERROR__INTERNAL;
        $exception = $self->safe_exception_message(shift);
    };

    return $self->render($self->get_error($error, $exception))
      if defined($error);

    $self->render(json => $result);
}

sub resource_get {
    my ($self) = @_;

    my $resource = $self->param('resource');

    my ($error, $exception, @include, @fields);

    try {
        @include = $self->get_include($resource);

        $self->check_params(
            qw(
              include
              public_id
              ), map {"fields[$_]"} $resource, @include
        );

        @fields = $self->get_fields($resource);
    }
    catch Exception::Conflict with {
        $error     = ERROR__CONFLICT;
        $exception = shift->message();
    }
    catch Exception::IncorrectParams with {
        $error     = ERROR__PARAMS;
        $exception = shift->message();
    };

    return $self->render($self->get_error($error, $exception))
      if defined($error);

    my $relationships = RestApi::Relationships->relationships($resource);

    my %specific_fields = (public_id => 1);
    foreach my $model (@include) {
        map {$specific_fields{$_->[0]} = 1} grep {!in_array($_->[0], \@fields)} @{$relationships->{$model}};
    }

    my $object;
    try {
        $object = $self->models->$resource->api_get($self->param('public_id'),
            fields => [sort keys(%specific_fields), grep {!$specific_fields{$_}} @fields],);

        throw Exception::NotFound unless defined($object);
    }
    catch Exception::NotFound with {
        $error = ERROR__NOT_FOUND;
    }
    catch Exception::Validation::BadArguments with {
        $error     = ERROR__PARAMS;
        $exception = shift->message();
    }
    catch {
        $error     = ERROR__INTERNAL;
        $exception = $self->safe_exception_message(shift);
    };

    return $self->render($self->get_error($error, $exception))
      if defined($error);

    my $resource_meta = $self->models->$resource->api_get_meta() // {};

    my $result->{'meta'} = {
        fields => [sort grep {!$specific_fields{$_}} keys %{$self->models->$resource->last_fields()}],
        %$resource_meta
    };

    my $relationships_data = {};
    foreach my $model (@include) {
        try {
            my $model_filter = {map {$_->[1] => $object->{$_->[0]}} @{$relationships->{$model}}};

            my @relationships_fields = $self->get_fields($model);

            $relationships_data->{$model}{'fields'} = \@relationships_fields;

            my %related_specific_fields = (public_id => 1);
            map {$related_specific_fields{$_->[1]} = 1} @{$relationships->{$model}};

            $relationships_data->{$model}{'data'} = $self->models->$model->api_get_all(
                fields =>
                  [sort keys(%related_specific_fields), grep {!$related_specific_fields{$_}} @relationships_fields],
                filter => $model_filter,
            );
        }
        catch Exception::IncorrectParams with {
            $error     = ERROR__PARAMS;
            $exception = shift->message();
        }
        catch {
            $error     = ERROR__INTERNAL;
            $exception = $self->safe_exception_message(shift);
        };

        return $self->render($self->get_error($error, $exception))
          if defined($error);
    }

    $result->{'data'} = $self->get_resource_objets(
        $resource, [$object], $result->{'meta'}{'fields'},
        $relationships_data,
        one_object      => 1,
        no_object_links => 1
    );

    $result->{'links'} = {
        self => $self->get_abs_url(
            $self->name_prefix_for_get_abs() . '__resource_get',
            resource  => $resource,
            public_id => $self->param('public_id')
        ),
        (
            $self->models->$resource->api_can_add()
            ? (add_fields =>
                  $self->get_abs_url($self->name_prefix_for_get_abs() . '__add_fields', resource => $resource,))
            : ()
        ),
    };

    $result->{'included'} =
      [map {@{$self->get_resource_objets($_, $relationships_data->{$_}{'data'}, $relationships_data->{$_}{'fields'})}}
          @include]
      if @include;

    $self->render(json => $result);
}

sub _get_filter_definition {
    my ($self, $field, $filter_fields) = @_;

    my $sub_field = shift(@$field);

    my $sub_filter_fields = $filter_fields->{$sub_field};

    return @$field
      ? $self->_get_filter_definition($field, $sub_filter_fields->{'subfields'})
      : $sub_filter_fields;
}

sub action_log {
    my ($self) = @_;

    return $self->render($self->get_error(ERROR__NOT_IMPLEMENTED));
}

sub check_bad_fields {
    my ($self, $model, %patch) = @_;
    my @bad_fields = $self->get_bad_fields($model, %patch);
    if (@bad_fields) {
        throw Exception::IncorrectParams gettext('You can not edit the following fields: %s',
            join(', ', sort @bad_fields));
    }
}

sub get_bad_fields {
    my ($self, $model, %fields) = @_;
    my %fields_no_api_can_edit = _get_no_api_fields($model);
    return grep {exists $fields_no_api_can_edit{$_}} keys %fields;
}

sub _get_no_api_fields {
    my ($model) = @_;
    my %model_fields = %{$model->get_model_fields()};
    return map {$_ => 1}
      grep {exists $model_fields{$_}{api_can_edit} && !$model_fields{$_}{api_can_edit}} keys %model_fields;
}

1;
