package RestApi::Controller::MultistateModel;

use Mojo::Base qw(RestApi::Controller::DBModel);

use RestApi::Errors;

use Exception::Denied;
use Exception::IncorrectParams;
use Exception::Multistate::BadAction;
use Exception::Multistate::NotFound;
use Exception::Validation;

use qbit;

sub name_prefix_for_get_abs {
    return 'multistate_model';
}

sub do_action {
    my ($self) = @_;

    my $resource  = $self->param('resource');
    my $public_id = $self->param('public_id');
    my $operation = $self->param('operation');
    my (@errors, $result);

    try {
        $self->check_params(qw(public_id operation));

        my $model = $self->models->$resource;

        my %operation_to_action = $model->operation_to_action();

        my $action = $operation_to_action{$operation} // $operation;
        my $body = $self->get_body(default => '{}');
        if ('edit' eq $action) {
            $self->check_bad_fields($self->models->$resource, %$body);
        }

        my $object = $model->do_action_with_result($public_id, $action, %$body);
        my $last_fields = [];

        my %actions_with_result = (duplicate => 1, change_contract => 1);
        my %object_as_result = (change_contract => 1);

        unless ($actions_with_result{$action}) {

            # Получаем дефолтные поля которые нужно вернуть для текущего action
            my @default_fields_for_action = ();
            {
                my $model_fields = $model->get_model_fields();
                foreach my $field_name (keys %$model_fields) {
                    my $actions = $model_fields->{$field_name}->{'default_for_actions'} // [];
                    if ($actions && grep {$_ eq $action} @$actions) {
                        push @default_fields_for_action, $field_name;
                    }
                }
            }

            # Получаем дефолтные поля и available_fields, чтобы потом по ним отфильтровать недоступные для текущих условий
            $object = $model->api_get($public_id,
                fields =>
                  [qw(public_id multistate multistate_name actions available_fields), @default_fields_for_action]);

            # Выкидываем available_fields если только они не входят в дефолты для текущего action
            my $exclude_fields = {
                'public_id' => 1,
                (grep {$_ eq 'available_fields'} @default_fields_for_action)
                ? ()
                : ('available_fields' => 1)
            };

            $last_fields = [sort grep {!$exclude_fields->{$_}} keys %{$model->last_fields()}];

        }
        $object = {public_id => $public_id} if (!defined $object);
        $result = {
            data => (
                $object_as_result{$action} ? $object : $self->get_resource_objets(
                    $resource, [$object], $last_fields,
                    {},
                    one_object      => 1,
                    no_object_links => 1
                )
            ),
            links => {
                self => $self->get_abs_url(
                    $self->name_prefix_for_get_abs() . '__resource_get',
                    resource  => $resource,
                    public_id => $object->{'public_id'},
                ),
            },
            meta => {fields => $last_fields,},
        };
    }
    catch Exception::IncorrectParams with {
        push(@errors, $self->get_error_object(ERROR__PARAMS, detail => shift->message()));
    }
    catch Exception::Conflict with {
        push(@errors, $self->get_error_object(ERROR__CONFLICT, detail => shift->message()));
    }
    catch Exception::Multistate::NotFound with {
        push(@errors, $self->get_error_object(ERROR__CONFLICT, detail => gettext('Not found %s', $public_id)));
    }
    catch Exception::Multistate::BadAction with {
        push(@errors, $self->get_error_object(ERROR__PARAMS, detail => gettext('Cannot do action "%s"', $operation)));
    }
    catch Exception::Validation with {
        my ($exception) = @_;

        my $validation_errors =
          $exception->isa('Exception::Validator::Errors')
          ? from_json($exception->message())
          : [{name => [], messages => [$exception->message()]}];

        foreach my $err (@{$validation_errors}) {
            push(
                @errors,
                $self->get_error_object(
                    ERROR__VALIDATION,
                    source => {pointer => "/" . join('/', @{$err->{'name'}})},
                    detail => join("\n", @{$err->{'messages'}})
                )
            );
        }
    }
    catch Exception::Denied with {
        push(@errors, $self->get_error_object(ERROR__FORBIDDEN, detail => shift->message()));
    }
    catch {
        push(@errors, $self->get_error_object(ERROR__INTERNAL, detail => $self->safe_exception_message(shift)));
    };

    if (@errors) {
        $self->set_http_code($errors[0]->{'id'});

        return $self->render(json => {errors => \@errors});
    }

    $self->render(json => $result);
}

1;
