package Rosetta;

=encoding UTF-8
=cut

=head1 DESCRIPTION

=head2 Стандартное использование

В продакшене используется 2 типа запросов:

=over

=item * authorization

=item * call

=back

Сначала пользователь авторизируется с помощью запроса authorization, а потом
идет несколько запросов call (и эти запросы уже видят что польователь
авторизован).

Но для того чтобы облегчить разработку и тестирование были сделаны еще
несколько других типов запросов.

=head2 Использование fake login

Это работает только на серверах разработки и тестовых серверах.

Сначала пользователь авторизируется с помощью запроса authorization,
потом с помощью запроса fake_login пользователь превщащается в другого
пользователя, а уже потом идут call запросы, которые выполняются от
имени пользователя которым зафейклогинились.

Цель fake login - посмотреть как выглядит интерфес под другим пользователям.

=head2 Использовать fake authorization

Это работает только на серверах разработки и тестовых серверах.

Пользователь авторизируется с помощью запроса fake_authorization (запрос
authorization в этом случае не используется), а дальше идет работа
с помощью call запросов.

Цель этого способа работы - возможность написания тестов.

L<https://wiki.yandex-team.ru/partner/w/task-deploy-partner2-frontend>

=cut

use qbit;

use base qw(Application);

use PiConstants qw($AUTHORIZATION_USER_FIELDS $BLACKBOX_DBFIELDS);

use Utils::Logger qw/ERROR/;

use Exception::BlackBox::NeedAuth;
use Exception::BlackBox::NeedRealLogin;
use Exception::BlackBox::NeedResign;
use Exception::BlackBox::NotSecure;
use Exception::Validation::BadArguments;
use Rosetta::Methods qw(call);

$ENV{SYSTEM} = 'rosetta';

=head1 authorization

На входе:

    session_id => '...',
    session_id2 => '...',
    user_ip => '192.168.1.1',
    server_name => 'partner2-test.yandex.ru',

На выходе:

    {
        result        => "ok",
        user_id       => 155209804,
        client_id     => 2901607,
        login         => "yndx-bessarabov",
        display_login => "yndx.bessarabov",
    }

=cut

sub authorization {
    my ($self, %opts) = @_;

    my $bb_res;
    my $need_auth;
    my $error_message;

    # clear cached val
    $self->exception_dumper->set_curr_dump_link('');

    # Сохраняем заголовки на время сессии (чтобы потом вывести в pi_errors)
    $self->set_option(http_headers => $opts{headers});

    try {
        throw "No server_name" unless defined($opts{server_name});

        $bb_res = $self->api_blackbox->sessionid(
            session_id     => $opts{session_id},
            sessionid2     => $opts{session_id2},
            remote_addr    => $opts{user_ip},
            server_name    => $opts{server_name},
            fields         => $BLACKBOX_DBFIELDS,
            must_be_secure => TRUE,
        );
    }
    catch Exception::BlackBox::NeedAuth catch Exception::BlackBox::NotSecure with {
        my ($e) = @_;
        $need_auth = {'result' => 'need_auth', data => [{'url' => $e->{'url_auth'}}]};
    }
    catch Exception::BlackBox::NeedResign with {
        my ($e) = @_;
        $need_auth = {'result' => 'need_resign', data => [{'url' => $e->{'url_resign'}}]};
    }
    catch Exception::BlackBox::NeedRealLogin with {
        my ($e) = @_;
        $need_auth = {'result' => 'need_real_login', data => [{'url' => $e->{'url_real_login'}}]};
    }
    catch {
        my ($e) = @_;
        $error_message = $self->_safe_exception_message($e);
    };

    if ($error_message) {
        return {
            result  => 'error',
            message => $error_message,
            (defined($opts{request_id}) ? (request_id => $opts{request_id}) : ()),
        };
    } elsif ($need_auth) {
        return {
            result => $need_auth->{result},
            ($need_auth->{data}         ? (data       => $need_auth->{data}) : ()),
            (defined($opts{request_id}) ? (request_id => $opts{request_id})  : ()),
        };
    } else {

        my $cur_user;

        {
            my $tmp_rights =
              $self->add_tmp_rights(qw(users_view_all users_view_field__multistate users_view_field__client_id));

            $cur_user = $self->users->get_by_uid($bb_res->{'uid'}{'content'}, fields => $AUTHORIZATION_USER_FIELDS);

            if ($cur_user) {
            } else {
                $cur_user = {
                    blackbox      => TRUE,
                    id            => $bb_res->{'uid'}{'content'},
                    login         => $bb_res->{'dbfield'}{'accounts.login.uid'},
                    email         => $bb_res->{'address-list'}->{'address'}->{content},
                    business_unit => 0,
                };
                ($cur_user->{'lastname'}, $cur_user->{'name'}, $cur_user->{'midname'}) =
                  (split(' ', $bb_res->{'dbfield'}{'account_info.fio.uid'} || ''), '', '', '');
            }
            $cur_user->{'display_login'} = $bb_res->{'login'}{'content'} // '';
        }

        $cur_user->{'language'} = $bb_res->{'dbfield'}{'userinfo.lang.uid'};

        $self->set_cur_user($cur_user);

        my $canonical_login = fix_login($cur_user->{'display_login'});

        return {
            result        => 'ok',
            user_id       => $bb_res->{uid}->{content},
            client_id     => $cur_user->{client_id},
            login         => $canonical_login,
            display_login => $cur_user->{'display_login'},
            (defined($opts{request_id}) ? (request_id => $opts{request_id}) : ()),
        };

    }
}

=head1 fake_authorization

На входе:

    login => "yndx.bessarabov",

На выходе такой же ответ как и у authorization:

    {
        result        => "ok",
        user_id       => 155209804,
        client_id     => 2901607,
        display_login => "yndx.bessarabov",
        login         => "yndx-bessarabov",
    }

=cut

sub fake_authorization {
    my ($self, %opts) = @_;

    my $cur_user;
    my $error_message;

    try {

        throw Exception::Validation::BadArguments gettext('unallowed') unless $self->get_option('allow_fake_login');
        throw Exception::Validation::BadArguments gettext('No login')  unless $opts{login};

        my $tmp_rights =
          $self->add_tmp_rights(qw(users_view_all users_view_field__multistate users_view_field__client_id));

        my $data = $self->users->get_all(
            filter => {login => $opts{login}},
            fields => $AUTHORIZATION_USER_FIELDS,
        );

        throw Exception::Validation::BadArguments gettext('No user with login %s', $opts{login}) if @$data != 1;

        $cur_user = $data->[0];

        $cur_user->{'display_login'} = $cur_user->{'login'};

        $self->set_cur_user($cur_user);

    }
    catch {
        $error_message = $self->_safe_exception_message($_[0]);
    };

    if ($error_message) {
        return {
            result  => 'error',
            message => $error_message,
        };
    } else {

        return {
            result        => 'ok',
            user_id       => $cur_user->{'id'},
            client_id     => $cur_user->{'client_id'},
            login         => $cur_user->{'login'},
            display_login => $cur_user->{'login'},
        };

    }

}

=head1 fake_login

На входе:

    login => "platonov",

На выходе точно такой же ответ как у метода authorization:

    {
        result        => "ok",
        user_id       => 35203,
        client_id     => 2901550,
        login         => "platonov",
        display_login => "platonov",
    }

Или же ошибка вида:

    {
        result  => "error",
        message => "Can't locate object method "frontend2" via package "Rosetta" at /home/bessarabov/beta.8066/lib/Rosetta.pm line 200.
    ",
    }

=cut

sub fake_login {
    my ($self, %opts) = @_;

    my $error_message;

    try {

        throw Exception::Validation::BadArguments gettext('unallowed') unless $self->get_option('allow_fake_login');
        throw Exception::Validation::BadArguments gettext('No login')  unless $opts{login};
        throw Exception::Validation::BadArguments gettext("Can't fake login") unless $self->check_rights('fake_login');

        my $cur_user = $self->get_option('cur_user');

        my $fake_user = $self->users->get_by_login($opts{login}, fields => $AUTHORIZATION_USER_FIELDS)
          // throw gettext('Bad fake login');
        $fake_user->{'display_login'} = $fake_user->{'login'};
        $fake_user->{'fake'}          = TRUE;
        $fake_user->{'real_user'}     = $cur_user;
        $cur_user                     = $fake_user;

        $self->set_cur_user($cur_user);

    }
    catch {
        $error_message = $self->_safe_exception_message($_[0]);
    };

    if ($error_message) {
        return {
            result  => 'error',
            message => $error_message,
        };
    } else {

        my $cur_user = $self->get_option('cur_user');

        return {
            result        => 'ok',
            user_id       => $cur_user->{'id'},
            client_id     => $cur_user->{'client_id'},
            login         => $cur_user->{'login'},
            display_login => $cur_user->{'display_login'},
        };

    }

}

=head1 ping

Метод ожидает что-нибудь на входе (не важно что, просто в протоколе зашито
что обязательно должен быть json с запросом).

На выходе ответ:

    {
        result => 'ok',
    }

или

    {
        result  => 'error',
        message => <Сообщение об ошибке>,
    }

Метод проверяет работоспособность кода и доступность базы данных.
Метод написан для использования в мониторинге.

=cut

sub ping {
    my ($self) = @_;

    my $error_message;

    try {
        my $data = $self->partner_db->_get_all('select 1')->[0]{1};
        throw "MySQL insanity, 'select 1' did not return 1" unless defined($data) && $data eq '1';
    }
    catch {
        $error_message = $self->_safe_exception_message($_[0]);
    };

    if ($error_message) {
        return {
            result  => 'error',
            message => $error_message,
        };
    } else {
        return {result => 'ok'};
    }
}

=head1 request

Store and fetch the request

=cut

sub request {
    my ($self, $request) = @_;
    return defined($request) ? $self->{'__REQUEST__'} = $request : $self->{'__REQUEST__'};
}

=head1 response

Store and fetch the response

=cut

sub response {
    my ($self, $response) = @_;
    return defined($response) ? $self->{'__RESPONSE__'} = $response : $self->{'__RESPONSE__'};
}

sub _safe_exception_message {
    my ($self, $exception) = @_;
    my $message = $exception->message;

    $message =~ s/ at \/.*$//;    # cut off the module name and line, if any

    return $message;
}

TRUE;
