package Rosetta::Methods;

use Exporter;

use qbit;

use Exception::Denied;
use Exception::Rosetta;
use Exception::Validation;

our @ISA       = qw(Exporter);
our @EXPORT    = ();
our @EXPORT_OK = qw(call nacked_call);

my @prohibited_models = qw(
  memcached
  partner_db
  partner_logs_db
  );

my @allow_models = qw(
  common_offer
  resources
  frontend
  rbac
  users
  widgets
  simple_notification
  cur_user
  user_notifications
  );

sub nacked_call {
    my ($self, %opts) = @_;

    local $ENV{SUBSYSTEM} = 'rosetta';

    my $model  = $opts{model};
    my $method = $opts{method};

    throw Exception::Rosetta gettext("Parameter '%s' must be an array.", 'args')
      if (ref($opts{args}) ne 'ARRAY');

    my @args = @{$opts{args}};

    my @data;

    throw Exception::Rosetta gettext('Model "%s" is unavailable', $model)
      if !$self->can($model) || in_array($model, \@prohibited_models);

    my $cur_user = $self->get_option('cur_user', {});
    throw Exception::Denied gettext('Model "%s" is unavailable', $model)
      if defined $cur_user->{common_offer}
          and $cur_user->{common_offer} < 0
          and !in_array($model, \@allow_models);

    throw Exception::Rosetta gettext('Method "%s" is not supported', $method)
      if !$self->$model->can($method);

    @data = $self->$model->$method(@args);

    return \@data;
}

=head1 call

На входе:

    model => 'users',
    method => 'get',
    args => ['155209804', fields => ['login', 'email', 'lastname']],

На выходе:

    {
        data   => [
            {
                email    => "bessarabov@yandex-team.ru",
                lastname => "Бессарабов",
                login    => "yndx-bessarabov",
            },
        ],
        result => "ok",
    }

Или же ошибка вида:

    {
        description           "Внутренняя ошибка",
        private_description   "Модель "ussr" недоступна",
        result                "error",
        type                  "internal"
    }

    {
        description           "Внутренняя ошибка",
        private_description   "Метод "gets" не поддерживается",
        result                "error",
        type                  "internal"
    }

=cut

sub call {
    my ($self, %opts) = @_;

    my $request_id = $opts{request_id};

    throw Exception::Rosetta gettext("Parameter '%s' must be an array.", 'args')
      if (ref($opts{args}) ne 'ARRAY');

    my $data;
    try {

        $data = {
            result => 'ok',
            data   => nacked_call($self, %opts),
            (defined($request_id) ? (request_id => $request_id) : ()),
        };
    }
    catch Exception::Validation with {
        my ($exception) = @_;

        $data = {
            result => 'error',
            type   => 'validation',
            fields => $exception->isa('Exception::Validator::Errors')
            ? from_json($exception->message())
            : [{name => [], messages => [$exception->message()]}],
            (defined($request_id) ? (request_id => $request_id) : ()),
        };
    }
    catch Exception::Denied with {
        my ($exception) = @_;

        $data = {
            result      => 'error',
            type        => 'denied',
            description => $exception->message(),
            (defined($request_id) ? (request_id => $request_id) : ()),
        };
    }
    catch {
        my ($exception) = @_;

        $self->exception_dumper->dump_as_html_file($exception);

        my $private_description = ref($exception) ? $exception->message() : '';
        $private_description =~ s/\s+/ /g;

        $data = {
            result              => 'error',
            type                => 'internal',
            description         => gettext("Internal Error"),
            private_description => $private_description,
            (defined($request_id) ? (request_id => $request_id) : ()),
        };
    };

    return $data;
}

TRUE;
