package RosettaProtocol;

use strict;
use warnings FATAL => 'all';

use qbit;

use Rosetta;
use Utils::Logger qw(INFO ERROR), {logger => 'RosettaServer',};

use Encode;
use Time::HiRes qw(
  gettimeofday
  tv_interval
  );

use Exception::Rosetta;
use Exception::Rosetta::SocketError;
use Exception::Rosetta::PartialIO;

our @ISA       = qw(Exporter);
our @EXPORT_OK = qw(
  read_bytes
  write_bytes
  read_request
  read_response
  write_response
  write_request
  interact
  );
our @EXPORT = @EXPORT_OK;

sub read_bytes {
    my ($socket, $length) = @_;

    my $n      = 0;
    my $result = '';
    my $part_n;

    while ($n < $length && ($part_n = sysread($socket, $result, $length - $n, $n))) {
        $n += $part_n;
    }

    throw Exception::Rosetta::SocketError "Socket error during sysread: $!"
      unless defined $part_n;

    return $result;
}

sub write_bytes {
    my ($socket, $scalar) = @_;

    my $n      = 0;
    my $length = length($scalar);
    my $part_n;

    while ($n < $length && defined($part_n = syswrite($socket, $scalar, $length - $n, $n))) {
        $n += $part_n;
    }

    throw Exception::Rosetta::SocketError "Socket error during syswrite: $!"
      unless defined $part_n;

    return $n;
}

sub read_rosetta_message {
    my ($socket) = @_;

    my $length = '';
    my $size   = '';

    while (TRUE) {
        my $char = read_bytes($socket, 1);
        if ($char eq "\n") {
            last;
        } elsif ($char eq "") {
            if (length($size)) {
                throw Exception::Rosetta::PartialIO "Connection was closed by client before sending a complete message";
            } else {
                return undef;
            }
        } else {
            $size .= $char;
        }
    }

    unless ($size =~ /^\d+$/) {
        my $ip;
        $ip = $socket->sockhost if $socket->can('sockhost');
        throw Exception::Rosetta sprintf("Incorrect rosetta request \"%s\" from \"%s\"", $size, $ip || 'unknown');
    }

    my $data = read_bytes($socket, $size);
    throw Exception::Rosetta::PartialIO "Connection was closed by client before sending a complete message"
      if length($data) < $size;
    $data = decode_utf8($data);

    return $data;
}

sub write_rosetta_message {
    my ($socket, $data) = @_;

    $data = encode_utf8($data);
    $data = length($data) . "\n" . $data;
    my $written = write_bytes($socket, $data);
}

sub read_response {
    my ($socket) = @_;

    return from_json(read_rosetta_message($socket) // return undef);
}

sub read_request {
    my ($socket) = @_;

    my ($type, $args) = split /\n/, (read_rosetta_message($socket) // return undef), 2;
    return [$type, from_json($args)];
}

sub write_response {
    my ($socket, $data) = @_;

    write_rosetta_message($socket, to_json($data));
}

sub write_request {
    my ($socket, $type, $data) = @_;

    write_rosetta_message($socket, $type . "\n" . to_json($data));
}

sub log_message {
    my (%args) = @_;

    my $sep        = "\t";
    my $quoted_sep = '\t';
    my @message    = ();

    my $method = $args{request}[0];
    my $params = $args{request}[1];
    push @message, $method, $args{login};

    return join($sep, @message) if grep {$_ eq $method} qw(fake_login fake_authorization ping);

    push @message, ($params->{nginx_request_id} // 'unknown_nginx_request_id');
    push @message, ($params->{request_id}       // 'unknown_request_id');
    if ($method eq 'call') {
        push @message, ($params->{model}  // 'unknown_model');
        push @message, ($params->{method} // 'unknown_method');
    }

    if (exists($args{result})) {
        push @message, ($args{result}  // 'unknown_result');
        push @message, ($args{elapsed} // 'unknown_elapsed_time');

        if ($args{result} eq 'error') {
            push @message, ($args{response}{type}                // 'type');
            push @message, ($args{response}{description}         // 'no_description');
            push @message, ($args{response}{private_description} // 'no_private_description');
        }
    } else {
        my $json = to_json($params);
        $json =~ s/\n/\\n/g;
        $json =~ s/$sep/$quoted_sep/g;
        push @message, $json;
    }

    return join($sep, @message);
}

sub process_request {
    my ($connection, $app, $methods) = @_;
    $app->set_time();

    my $request = read_request($connection) // return 0;

    $app->request($request);

    my $method = $request->[0];
    if (!grep {$_ eq $method} @$methods) {
        write_response($connection, {result => 'error', message => "Unexpected request type: $method"});
        return 0;
    }

    my $params   = $request->[1];
    my $cur_user = $app->get_option('cur_user');
    my $login    = $cur_user->{display_login} // 'unknown';

    INFO log_message(
        login   => $login,
        request => $request,
    );
    my $t0       = [gettimeofday()];
    my $response = $app->$method(%$params);
    my $elapsed  = tv_interval($t0, [gettimeofday()]) * 1000;
    INFO log_message(
        login    => $login,
        request  => $request,
        result   => $response->{result},
        response => $response,
        elapsed  => $elapsed,
    );

    write_response($connection, $response);

    return 0 if $response->{result} eq 'error';
    return 1;
}

sub interact {
    my ($connection, $app) = @_;

    try {
        {
            $app->pre_run();
            # Authorization
            process_request($connection, $app, [qw(authorization fake_authorization ping)])
              or last;

            # Calls
            while (process_request($connection, $app, [qw(call fake_login)])) { }

            $app->post_run();
        }
    }
    catch {
        my ($exception) = @_;

        $app->exception_dumper->dump_as_html_file($exception);
    };
}
