package Utils::CSV;

use qbit;

use Text::CSV;

our @ISA    = qw(Exporter);
our @EXPORT = qw(
  combine_csv_indoor_blocks
  csv_with_fields_and_names
  parse_csv
  parse_csv_indoor_blocks
  );

my $SEP_FIELD    = ';';
my $SEP_IN_FIELD = ',';

my @indoor_block_headers =
  qw( nn page_id block_id page_caption page_gps page_address page_oname facility_type_name facility_type
  block_caption zone_category_name zone_category block_reso_width block_reso_height block_aspect_ratio block_min_duration block_max_duration
  block_sound block_touch block_mincpm block_photos block_comment);

=head2 parse_csv

  На входе:
  "a,b,c
   1,2,3
   4,5,6"

  на выходе:
  [
    {
      a => 1,
      b => 2,
      c => 3,
    },
    {
      a => 4,
      b => 5,
      c => 6,
    },
  ]
=cut

sub parse_csv {
    my ($original_csv, %opts) = @_;

    my $result = [];
    my $csv    = Text::CSV->new(\%opts);
    my $data   = [split /\n+/, $original_csv];
    my $line   = shift @$data;
    $csv->parse($line);
    my @fields = map {trim($_)} $csv->fields();

    while (my $line = shift @$data) {
        $csv->parse($line);
        my $row = {};
        @$row{@fields} = $csv->fields;

        push @$result, $row;
    }
    return $result;
}

sub parse_csv_indoor_blocks {
    my ($csv) = @_;

    my $parser = Text::CSV->new({sep_char => $SEP_FIELD, binary => 1, eol => $/});
    open my $io, "<", \$csv;

    $parser->column_names(qw( _ login ));
    my $login = $parser->getline_hr_all($io, 0, 1)->[0]{login};

    my @thematics = @{$parser->getline_all($io, 1, 1)->[0] // []};
    @thematics = 1 < scalar @thematics ? grep {trim($_) ne ''} splice(@thematics, 1) : ();
    my @brands = @{$parser->getline_all($io, 1, 1)->[0] // []};
    @brands = 1 < scalar @brands ? grep {trim($_) ne ''} splice(@brands, 1) : ();

    my $headers = $parser->getline_all($io, 1, 1)->[0];

    $parser->column_names(@indoor_block_headers);
    my $blocks = $parser->getline_hr_all($io, 0);

    throw $parser->error_input() if $parser->error_input();

    $blocks = [
        map {
            for my $bool_field (qw(block_sound block_touch))
            {
                $_->{$bool_field} = 'Y' eq uc($_->{$bool_field} // '') ? 1 : 0;
            }
            $_->{block_photos} = [split(',', $_->{block_photos})];
            $_;
          } @$blocks
    ];

    close($io);

    return {
        login    => $login,
        blocks   => $blocks,
        articles => \@thematics,
        brands   => \@brands,
        headers  => $headers,
    };
}

sub combine_csv_indoor_blocks {
    my ($parsed) = @_;

    my $parser = Text::CSV->new({sep_char => $SEP_FIELD, binary => 1, eol => $/});

    my @lines;
    push @lines, combine_csv($parser, [['login',    $parsed->{login}]]);
    push @lines, combine_csv($parser, [[]]);
    push @lines, combine_csv($parser, [['articles', @{$parsed->{articles}}]]);
    push @lines,
      combine_csv($parser,
        [@{$parsed->{articles_not_found}} ? ['articles not found', @{$parsed->{articles_not_found}}] : []]);
    push @lines, combine_csv($parser, [['brands', @{$parsed->{brands}}]]);
    push @lines,
      combine_csv($parser,
        [@{$parsed->{brands_not_found}} ? ['brands not found', @{$parsed->{brands_not_found}}] : []]);
    # its either linebreaks in fields OR utf8 in Excel, not both
    push @lines, combine_csv($parser, [[map {(my $h = $_) =~ s/\n/ /g; $h} @{$parsed->{headers}}]]);

    my @b = map {
        my %t = %{$_};
        $t{block_photos} = join($SEP_IN_FIELD, @{$_->{block_photos}});
        $t{block_sound} = $_->{block_sound} ? 'Y' : 'N';
        $t{block_touch} = $_->{block_touch} ? 'Y' : 'N';
        \%t;
    } @{$parsed->{blocks}};
    push @lines, combine_csv($parser, \@b, \@indoor_block_headers, $parsed->{errors});

    return join('', @lines);
}

sub combine_csv {
    my ($parser, $rows, $headers, $additions, $default) = @_;

    $headers   //= [];
    $additions //= [];

    my @lines;
    for (my $i = 0; $i < @$rows; $i++) {
        my @t = @$headers ? @{$rows->[$i]}{@$headers} : @{$rows->[$i]};
        @t = map {$_ // $default} @t if defined $default;
        my $status = $parser->combine(@t, ($additions->[$i] // ''));
        throw $parser->error_input() unless $status;
        push @lines, $parser->string();

    }
    return @lines;
}

sub csv_with_fields_and_names {
    my ($data, $fields, %opts) = @_;

    throw "'field' should be arrayref" if ref $fields ne 'ARRAY';
    throw "'field' should have values" if @{$fields} == 0;

    my $default = $opts{default};
    my $sep_char = $opts{sep_char} // $SEP_FIELD;

    foreach (@{$fields}) {
        throw "'field' element should be hashref" if ref($_) ne 'HASH';
        throw "'field' element should have 'field'" unless defined $_->{field};
        throw "'field' element should have 'name'"  unless defined $_->{name};

        $_->{name} =~ s/&nbsp;/ /g;
    }

    my $parser = Text::CSV->new({sep_char => $sep_char, binary => 1, eol => $/});
    my @worksheet;

    # header
    push @worksheet, combine_csv($parser, [[map {$_->{name}} @$fields]], undef, undef, $default);

    # content
    push @worksheet, combine_csv($parser, $data, [map {$_->{field}} @$fields], undef, $default);

    # footer / disclaimer
    my $disclaimer = gettext('payment_disclaimer');
    $disclaimer =~ s/\.\s+/.\n/g;
    chomp $disclaimer;
    push @worksheet, combine_csv($parser, [[''], [''], [$disclaimer]]);

    return join '', @worksheet;
}

TRUE;
