package Utils::DB;

=encoding UTF-8

=cut

use qbit;
use Exporter;
use lib::abs qw();

use PiConstants qw( $DB_CONFIG_FILE_PATH );
use PiSecrets qw(get_secret);
use Utils::PublicID;

our @ISA       = qw(Exporter);
our @EXPORT_OK = qw(
  get_db_connection_header
  get_db_settings
  fields_to_complex_filter
  fields_to_filter
  model_to_prefix
  );
our @EXPORT = @EXPORT_OK;

=head2 get_db_settings();

    my $settings = get_db_settings();

Читает файл DatabaseConfig.json текущей системы и возвращает HASHREF вида:

    {
        banks_db => {
            database => 'kladr',
            host => 'dev-partner2.ppc.yandex.ru',
            password => 'secret',
            port => 3478,
            user => 'pi2_dev',
        },
        kladr => {
            database => 'kladr',
            host => 'dev-partner2.ppc.yandex.ru',
            password => 'secret',
            port => 3478,
            user => 'pi2_dev',
        },
        partner_db => {
            database => 'partner',
            host => 'dev-partner2.ppc.yandex.ru',
            password => 'secret',
            port => 3478,
            user => 'pi2_dev',
        },
        partner_logs_db => {
            database => 'partner_logs',
            host => 'dev-partner2.ppc.yandex.ru',
            password => 'secret',
            port => 3478,
            user => 'pi2_dev',
        },
    }


=cut

sub get_db_settings {

    my $content = readfile(_get_database_config_file_name());

    my $config;
    try {
        $config = from_json($content);
    }
    catch {
        throw 'DatabaseConfig.json is not a valid json';
    };

    $config->{'partnerdb2'}{'password'} //= get_secret('partner2-mysql-password');

    my $settings = _convert_config_to_settings($config);

    return $settings;
}

=begin comment _convert_config_to_settings

=end comment

=cut

sub _convert_config_to_settings {
    my ($config) = @_;

    my %db_names = (
        partner_db      => 'partner',
        partner_logs_db => 'partner_logs',
        kladr           => 'kladr',
        banks_db        => 'kladr',
    );

    my %options;
    my ($db_data, $parent);

    foreach my $db (qw(partner_db partner_logs_db kladr banks_db)) {
        $db_data = $config->{$db};
        if ($db_data) {
            # one level of parent configuration is possible
            $options{$db} = {};
            $parent = delete($db_data->{'parent'});
            %{$options{$db}} = (($parent ? %{$config->{$parent} // {}} : ()), %$db_data);
        } else {
            # compatibility with old configs
            $options{$db} = {
                database => $db_names{$db},
                %{$config->{'partnerdb2'}}
            };
        }
    }

    return \%options;
}

=begin comment _get_database_config_file_name

    my $file_name = _get_database_config_file_name();

Возвращает полный путь с файлу с DatabaseConfig.json конфигом базы данных.

Падает если не может найти файл с конфигом.

=end comment

=cut

sub _get_database_config_file_name {

    my $file_name;

    my $db_dev_config_file_path = '../../configs/dev/DatabaseConfig.json';

    if (-e $DB_CONFIG_FILE_PATH) {
        $file_name = $DB_CONFIG_FILE_PATH;
    } elsif (-e lib::abs::path($db_dev_config_file_path)) {
        warn 'Using DatabaseConfig.json from "configs/dev/" directory. '
          . 'If you are not in the build release phase then this is an error.' . "\n";
        $file_name = lib::abs::path($db_dev_config_file_path);
    } else {
        throw sprintf('Can\'t find "%s"', $DB_CONFIG_FILE_PATH);
    }

    return $file_name;
}

sub fields_to_filter {
    my ($pk_fields, $entity_list, %opts) = @_;

    unless (@$entity_list) {
        return $opts{'for_db'} ? ['AND', [\undef]] : [undef];
    }

    my $group_by_first_field = {};

    foreach my $row (@$entity_list) {
        my ($val, $val2) = @$row{@$pk_fields};

        if (!defined($val) || !defined($val2)) {
            throw Exception::Validation::BadArguments(
                '[' . join(',', @$pk_fields) . '] should exist and be defined in every hash of 2nd arg');
        }

        $group_by_first_field->{$val}{$val2} = TRUE;
    }

    my $mutate = $opts{'for_db'} ? sub {\$_[0]} : sub {$_[0]};

    # TODO: make it (f1 = 1 and (f2 in (1,2,3))) instead of (f1 = 1 and f2 = 1) or (f1 = 1 and f2 = 2) or ...
    return [
        'OR',
        [
            map {
                [
                    'AND',
                    [
                        [$pk_fields->[0], '=',  $mutate->($_)],
                        [$pk_fields->[1], 'IN', $mutate->([sort keys(%{$group_by_first_field->{$_}})])]
                    ]
                ]
              } sort keys(%$group_by_first_field)
        ]
    ];
}

sub fields_to_complex_filter {
    my ($pks, %opts) = @_;

    unless (@$pks) {
        return $opts{'for_db'} ? ['AND', [\undef]] : [undef];
    }

    my $mutate = $opts{'for_db'} ? sub {\$_[0]} : sub {$_[0]};

    my @filter = ();

    foreach my $pk (@$pks) {
        my @filter_part = ();
        foreach my $field (sort keys(%$pk)) {
            push(@filter_part, [$field, '=', $mutate->($pk->{$field})]);
        }

        push(@filter, ['AND', \@filter_part]);
    }

    return ['OR', \@filter];
}

sub get_db_connection_header {
    my ($app) = @_;

    return () unless $ENV{TAP_VERSION};

    my %map = (
        database => 'schema',
        host     => 'mysql_host',
        port     => 'mysql_port',
        user     => 'username',
        password => 'password'
    );
    my $conf = to_json({map {$map{$_} => $app->partner_db->get_option($_)} keys %map});
    return ('Mysql-Configuration' => $conf);
}

sub model_to_prefix {
    my ($app, $field) = @_;

    my $hash = Utils::PublicID::get_block_accessors_by_public_id_prefixes($app);
    my $next = \'BAD-MODEL-';
    for my $prefix (keys %$hash) {
        $next = {if => [[$field, 'IN', \$hash->{$prefix}], \"$prefix", $next]};
    }

    return $next;
}

TRUE;
