package Utils::Deploy;

use Exporter;
use HTTP::Request;
use LWP::UserAgent;
use Net::INET6Glue::INET_is_INET6;

use qbit;

use Utils::Logger qw( INFO  INFOF );
use Carp 'croak';

use YAML::PP::Common qw(YAML_DOUBLE_QUOTED_SCALAR_STYLE);

our @ISA       = qw(Exporter);
our @EXPORT_OK = qw(
  get_cur_spec
  get_cur_yaml
  get_box_spec
  get_node_by_id
  get_node_by_tag
  get_per_cluster_stages
  get_revision_from_status
  set_replica_set_path
  get_stages_storage_class
  get_stage_status
  patch_yaml_pp
  resolve_db_endpoint
  wait_deploy_status_ready

  $STAGE_DEFAULTS
  $AUTOTEST_STAG
  $FUNC_TESTS_STAGE
  $FUNC_TESTS_TESTING_STAGE
  $DATABASE_UNIT_NAME
  );

our @EXPORT = @EXPORT_OK;

my $STATUS_READY = 'Ready';

my $PROD_STAGE = 'prod';
our $TEST_STAGE               = 'test';
our $PREPROD_STAGE            = 'preprod';
our $RELEASE_TEST_STAGE       = 'release-test';
our $FUNC_TESTS_STAGE         = 'func-tests';
our $FUNC_TESTS_TESTING_STAGE = 'func-tests-testing';
our $PERL_IMAGE               = 'partners/perl-backend';
our $NODE_IMAGE               = 'partners/frontend-node';
our $MOCK_IMAGE               = 'partners/mock-service';
our $DATABASE_UNIT_NAME       = 'Database';
our @COMMON_UPDATES           = qw(
  adfox
  frontend
  java_jsonapi
  java_intapi
  perl
  secrets
  );
our @CRON_UPDATES = qw(
  cron
  secrets
  );
our @DB_UPDATES = qw(
  mysql
  redeploy_db
  );

our $STAGE_DEFAULTS = {
    $PROD_STAGE => {
        project    => 'partner',
        stage_tag  => 'production',
        stage_name => 'partner-production-stage',
        config     => 'configs/production/deploy-stage.json',

        perl_image => $PERL_IMAGE,
        node_image => $NODE_IMAGE,
        adfox_host => 'api.ad-fox.yandex.ru',

        force_db_redeploy => 0,

        units => {
            'Backend'   => {unit_updates => [@COMMON_UPDATES],},
            'Crons'     => {unit_updates => \@CRON_UPDATES,},
            'Hourglass' => {unit_updates => [qw(hourglass haproxy secrets)],},
        },
    },
    $PREPROD_STAGE => {
        project    => 'partner',
        stage_name => 'partner-preprod-stage',
        config     => 'configs/preprod/deploy-stage.json',

        perl_image => $PERL_IMAGE,
        node_image => $NODE_IMAGE,
        adfox_host => 'crowdtest.adfox.yandex.ru',

        force_db_redeploy => 1,

        units => {
            'Backend'           => {unit_updates => [@COMMON_UPDATES]},
            'Crons'             => {unit_updates => \@CRON_UPDATES,},
            $DATABASE_UNIT_NAME => {unit_updates => \@DB_UPDATES,},
            'Hourglass'         => {unit_updates => [qw(hourglass haproxy secrets)],},
        },
    },
    $TEST_STAGE => {
        project    => 'partner',
        stage_name => 'partner-test-stage',
        config     => 'configs/test/deploy-stage.json',

        perl_image => $PERL_IMAGE,
        node_image => $NODE_IMAGE,
        adfox_host => 'test.ui-dev.adfox.yandex.ru',

        force_db_redeploy => 0,

        units => {
            'Backend'   => {unit_updates => [@COMMON_UPDATES],},
            'Crons'     => {unit_updates => \@CRON_UPDATES,},
            'Hourglass' => {unit_updates => [qw(hourglass haproxy secrets)],},
        },
    },
    $RELEASE_TEST_STAGE => {
        project    => 'partner',
        stage_name => 'partner-release-integration-test-2',
        config     => 'configs/release-test/deploy-stage.json',

        perl_image => $PERL_IMAGE,
        node_image => $NODE_IMAGE,
        adfox_host => 'test.ui-dev.adfox.yandex.ru',

        force_db_redeploy => 0,

        units => {
            'Backend'   => {unit_updates => [@COMMON_UPDATES]},
            'Crons'     => {unit_updates => \@CRON_UPDATES,},
            'Hourglass' => {unit_updates => [qw(hourglass haproxy secrets)],},
        },
    },
    $FUNC_TESTS_STAGE => {
        project    => 'partner',
        stage_name => 'partner_func-tests_env',
        config     => 'configs/func-tests/deploy-stage.json',

        units => {'teamcity' => {unit_updates => ['secrets'],},},
    },
    $FUNC_TESTS_TESTING_STAGE => {
        project    => 'partner',
        stage_name => 'partner_func-tests-testing',
        config     => 'configs/func-tests-testing/deploy-stage.json',

        units => {'teamcity' => {unit_updates => ['secrets'],},},
    },
};

my $REPLICA_SET_PATH = {
    'multi-cluster' => [qw(multi_cluster_replica_set replica_set)],
    'per-cluster'   => [qw(replica_set replica_set_template)],
};

my $YAML_ALREADY_LOADED = {};

sub get_cur_spec {
    my ($stage_name, $YPP) = @_;
    my $cur_yaml_spec_filename = get_cur_yaml($stage_name);
    my $cur_spec_data          = $YPP->load_file($cur_yaml_spec_filename);
    return $cur_spec_data;
}

sub get_cur_yaml {
    my ($stage_name) = @_;

    my $cur_yaml_spec_filename = sprintf 'deploy-%s-spec-cur.yaml', $stage_name;
    unless ($YAML_ALREADY_LOADED->{$stage_name}) {
        run_shell(sprintf('ya tool dctl get stage %s  > %s', $stage_name, $cur_yaml_spec_filename), silent => 1);
        $YAML_ALREADY_LOADED->{$stage_name} = 1;
    }
    return $cur_yaml_spec_filename;
}

sub get_box_spec {
    my ($spec, $unit_name, $box_name) = @_;
    return $spec->{spec}{deploy_units}{$unit_name}{images_for_boxes}{$box_name};
}

sub get_multi_cluster_stages {
    my $result =
      run_shell("ya tool dctl list multi_cluster_replica_set | tail -n +4 | head -n -1 | awk '{ print \$4 }'",
        silent => 1);
    chomp($result);

    return {map {$_ => 1} split "\n", $result};
}

sub get_per_cluster_stages {
    my $list =
      run_shell("ya tool dctl list replica_set | tail -n +4 | head -n -1 | awk '{ print \$2, \$4 }'", silent => 1);
    chomp($list);

    my %result = ();

    for my $line (split "\n", $list) {
        my ($cluster, $id)          = split ' ',  $line;
        my ($stage,   $deploy_unit) = split '\.', $id;

        $result{$cluster}{$stage} = 1;
    }
    for (keys %result) {
        $result{$_} = [keys %{$result{$_}}];
    }
    return \%result;
}

sub get_stages_storage_class {
    my ($project, $yp_token) = @_;
    my $list = run_shell(
        "YP_TOKEN=$yp_token ya tool yp select stage "
          . " --address xdc "
          . "--selector /spec/deploy_units/Backend/replica_set/replica_set_template/pod_template_spec/spec/disk_volume_requests/0/storage_class "
          . "--selector /meta/id "
          . "--filter \"[/annotations/project]='$project'\" "
          . " | tail -n +4 | head -n -1 | awk '{ print \$2, \$4 }'",
        silent => 1
    );
    chomp($list);

    my %result = ();

    for my $line (split "\n", $list) {
        $line =~ s/"//g;
        my ($storage, $stage) = split ' ', $line;
        next unless $storage;
        if (exists $result{$storage}) {
            push @{$result{$storage}}, $stage;
        } else {
            $result{$storage} = [$stage];
        }
    }
    return \%result;
}

sub get_node_by_id {
    my ($items, $id) = @_;
    return get_node_by_tag($items, 'id', $id);
}

sub get_node_by_tag {
    my ($items, $tag_name, $tag_value) = @_;

    foreach my $item (@$items) {
        if ($item->{$tag_name} eq $tag_value) {
            return $item;
        }
    }

    croak(sprintf('Can not find a node with %s "%s"', $tag_name, $tag_value));
}

sub get_revision_from_status {
    my ($deploy_units) = @_;

    my $first_unit_revision;
    for my $unit (@$deploy_units) {
        unless (defined $first_unit_revision) {
            $first_unit_revision //= $unit->{SpecRev};
        } elsif ($unit->{SpecRev} != $first_unit_revision) {
            die sprintf('Deployment is in progress. Different revisions: %s and %s',
                $first_unit_revision, $unit->{SpecRev});
        }
    }

    return $first_unit_revision;
}

sub set_replica_set_path {
    my $mcrs = get_multi_cluster_stages();

    for my $stage (keys %$STAGE_DEFAULTS) {
        for my $unit (keys %{$STAGE_DEFAULTS->{$stage}->{units}}) {
            my $id = sprintf("%s.%s", $STAGE_DEFAULTS->{$stage}->{stage_name}, $unit);
            my $type = $mcrs->{$id} ? 'multi-cluster' : 'per-cluster';

            $STAGE_DEFAULTS->{$stage}->{units}->{$unit}->{replica_set_path} = $REPLICA_SET_PATH->{$type};
        }
    }
}

sub get_stage_status {
    my ($ya_dir, $stage) = @_;
    my $result = run_shell($ya_dir . "ya tool dctl status stage $stage", silent => TRUE);

    my @deploy_units_raw_data = map {[split(/\s*\|\s*/)]} grep {s/^\|\s*//} split(/\n/, $result);

    my @deploy_units;
    foreach my $i (1 .. $#deploy_units_raw_data) {
        my %hash;
        @hash{@{$deploy_units_raw_data[0]}} = map {$_ // ''} @{$deploy_units_raw_data[$i]};
        push @deploy_units, \%hash;
    }

    return \@deploy_units;
}

sub patch_yaml_pp {
    no strict 'refs';
    no warnings 'redefine';

    my $rb = \&{'YAML::PP::Schema::JSON::represent_bool'};
    *{'YAML::PP::Schema::JSON::represent_bool'} = sub {
        $rb->(@_);
        delete($_[1]->{anchor});
        return 1;
    };

    my $rl = \&{'YAML::PP::Schema::JSON::represent_literal'};
    *{'YAML::PP::Schema::JSON::represent_literal'} = sub {
        $rl->(@_);
        $_[1]->{style} = YAML_DOUBLE_QUOTED_SCALAR_STYLE;
        return 1;
    };

    my $dn = \&{'YAML::PP::Dumper::dump_node'};
    *{'YAML::PP::Dumper::dump_node'} = sub {
        # Хак, чтобы не вставлять анкоры при повторных использованиях объектов (например, JSON::PP::true)
        # С этим хаком дамп выглядит так:
        #   field1: true
        #   field2: true
        # Без хака:
        #   field1: &1 true
        #   field2: &2 true
        if (ref $_[1]) {
            my $refaddr = refaddr($_[1]);
            delete($_[0]->{anchors}{$refaddr});
            $_[0]->{seen}{$refaddr} = 1 if $_[0]->{seen}{$refaddr} && $_[0]->{seen}{$refaddr} > 1;
        }
        $dn->(@_);
    };
}

sub resolve_db_endpoint {
    my ($cluster, $stage_name) = @_;

    my $ua = LWP::UserAgent->new(timeout => 180,);
    my $res = $ua->request(
        HTTP::Request->new(
            'POST',
            'http://sd.yandex.net:8080/resolve_endpoints/json',
            undef,
            to_json(
                {
                    cluster_name    => $cluster,
                    endpoint_set_id => "${stage_name}.Database",
                    client_name     => 'robot-partner',
                }
            )
        )
    );
    my $data = from_json($res->decoded_content, use_pp => TRUE)->{endpoint_set}{endpoints}[0];

    return wantarray ? ($data->{fqdn}, $data->{port}) : $data->{fqdn};
}

sub wait_deploy_status_ready {
    my ($ya_dir, $stage, $wait_sec, $prev_revision) = @_;

    die 'Revision expected' unless $prev_revision;
    die "Should be waiting at least two minutes" if $wait_sec < 120;

    INFOF 'Start waiting next deploy revision, current %s - https://deploy.yandex-team.ru/stages/%s', $prev_revision,
      $stage;

    my $current_revision = $prev_revision;
    my $current_status   = '';
    my $wait_time        = time + $wait_sec;

    my $is_ok         = 0;
    my $is_workaround = 0;
    while (time < $wait_time && !$is_ok) {
        sleep(30);

        my $deploy_units = get_stage_status($ya_dir, $stage);

        $is_ok = 1;
        foreach my $unit (@$deploy_units) {
            $current_status = $unit->{DeployStatus};
            INFOF 'Unit "%s": status %s, revision %s', $unit->{DeployUnitID}, $current_status, $unit->{SpecRev};
            $current_revision = $unit->{SpecRev};
            if ($current_status ne $STATUS_READY || $current_revision <= $prev_revision) {
                $is_ok = 0;
            }
        }

        if ($is_ok) {
            unless ($is_workaround) {
                $is_ok         = 0;
                $is_workaround = 1;
                INFO 'Start wait workaround (DEPLOY-3845)';
                sleep(30);    # временное решение пока не пофиксят DEPLOY-3845
            } else {
                last;
            }
        }
    }

    if ($is_ok && $current_revision >= $prev_revision) {
        INFOF 'The stage "%s" goes to status "%s" with revision %s', $stage, $current_status, $current_revision;
    } else {
        die sprintf('%d seconds have passed but status is "%s", revision %s',
            $wait_sec, $current_status, $current_revision);
    }
}

TRUE;
