
=encoding UTF-8

=head1 DESCRIPTION

Модуль для парсинга логов.

=cut

package Utils::LogParser;

use qbit;

use Utils::Logger qw(INFO WARN ERROR INFOF);

use Storable;
use Fcntl qw(SEEK_SET SEEK_END);

=head2 process_logs

Пример входных параметров:

my $logs = {
    'log_key' => {
        name => 'log_name',
        path => '/var/log/super.log',
    },
};
my $checks = {
    'check-name' => {
        apply_to     => 'log-name' | qr/partial-name/ | sub { is_interesting($log_name) },
        regex        => qr/FATAL ERROR/,
        check_sub    => sub {
            my ($log, $check, $temp_data, $line) = @_;
            aggregate($log, $check, $line, $temp_data);
        },
        final_sub    => sub {
            my ($check, $logs, $temp_data, $results) = @_;

            my $result = finalize($check, $logs);
            $results->{check_key} = $result;
        },
    },
};

=cut

sub process_logs {
    my ($logs, $checks, $positions, %opts) = @_;

    $logs->{$_}{name}   //= $_ for keys %$logs;
    $checks->{$_}{name} //= $_ for keys %$checks;
    $logs->{$_}{preparse} //= sub {TRUE}
      for keys %$logs;

    my %checks_temp_data = map {$_ => {}} keys %$checks;

    my $limit = $opts{limit};
    my %stat_path = map {$_ => {}} split /\s*,\s*/, $opts{stat_path} // '';

    for my $log_key (keys %$logs) {
        my $log = $logs->{$log_key};

        my $fh = _seek_to_start($log, $positions) or next;

        my @applicable_checks = (grep {_check($checks->{$_}{apply_to}, $log_key)} keys %$checks);

        unless (@applicable_checks) {
            WARN "Skip processing $log_key: no checks are applicable";
            next;
        }

        my $cnt = 0;
        while (my $line = readline($fh)) {
            my $preparse = $log->{preparse}($line, $log, \%stat_path);
            next unless $preparse;
            for my $check_key (@applicable_checks) {
                my $check = $checks->{$check_key};
                if ($line =~ $check->{regex}) {
                    $check->{check_sub}->($log, $check, $checks_temp_data{$check_key}, $line, $preparse);
                }
            }
            last if $limit and $cnt++ > $limit;
        }
        $positions->{$log->{path}} = tell($fh);
    }

    my %results;

    for my $check_key (keys %$checks) {
        my $check = $checks->{$check_key};
        $check->{final_sub}->($check, $logs, $checks_temp_data{$check_key}, \%results);
    }

    return (\%results, $positions, \%stat_path);
}

sub _check {
    my ($condition, $arg) = @_;

    if (ref($condition) eq 'Regexp') {
        $arg =~ $condition;
    } elsif (ref($condition) eq 'CODE') {
        $condition->($arg);
    } else {
        $condition eq $arg;
    }
}

sub _seek_to_start {
    my ($log, $positions) = @_;

    my $fh;
    unless (open($fh, '<', $log->{path})) {
        INFOF "Skip processing %s (%s): %s", $log->{name}, $log->{path}, $!;
        return;
    }

    my $curr_last_pos = _get_file_last_pos($fh);
    my $last_processed_pos = $positions->{$log->{path}} // -1;

    unless ($curr_last_pos) {
        INFOF "Skip processing %s: %s is empty", $log->{name}, $log->{path};
        return;
    }

    if ($curr_last_pos == $last_processed_pos) {
        INFOF "Skip processing %s: %s has not changed", $log->{name}, $log->{path};
        return;
    }

    my $from_pos =
      $last_processed_pos > $curr_last_pos
      ? 0
      : $last_processed_pos + 1;

    seek($fh, $from_pos, SEEK_SET);

    return $fh;
}

sub write_log_positions {
    my ($log_position_file, $new_log_positions) = @_;

    my $data = $new_log_positions;

    eval {Storable::store($data, $log_position_file);};
    if ($@) {
        ERROR "Store positions file $log_position_file error: $@";
    }

    return 1;
}

sub read_log_positions {
    my ($log_position_file) = @_;

    my $log_positions = {};

    my $data = eval {Storable::retrieve($log_position_file)};
    if ($@) {
        INFO "Can't read log positions file $log_position_file: $@";
    } else {
        $log_positions = $data;
    }

    return $log_positions;
}

sub _get_file_last_pos {
    my ($fh) = @_;

    my $cur_pos = tell($fh);

    # Выставляем каретку в конец файла
    seek($fh, 0, SEEK_END);

    my $last_pos = tell($fh);

    seek($fh, $cur_pos, SEEK_SET);

    return $last_pos;
}

1;
