package Utils::Logger;

use qbit;

use lib::abs;
use Exporter;
use Clone qw(clone);
use Data::Dumper;
use Log::Log4perl;
use Log::Log4perl::Level;
use File::Slurp;
use Log4Perl::Message;

our @ISA       = qw(Exporter);
our @EXPORT_OK = qw(
  get_logger
  TRACE
  DEBUG
  INFO
  WARN
  ERROR
  FATAL
  TRACEF
  DEBUGF
  INFOF
  WARNF
  ERRORF
  FATALF
  );
our @EXPORT = @EXPORT_OK;

my $is_dataprinter = undef;

my $qbit_app;

my %sentry_level = (
    $FATAL => 'fatal',
    $ERROR => 'error',
    $WARN  => 'warning',
);

=head2 import

=cut

my %logger;

my $is_default_init_already = 0;

sub import {
    my ($class, @args) = @_;

    if (@args && ref($args[-1]) eq 'HASH') {
        my $opts   = pop @args;
        my $caller = (caller())[0];
        $logger{$caller} = Log::Log4perl->get_logger($opts->{logger});
    }

    $class->export_to_level(1, $class, @args);

    unless ($is_default_init_already) {
        $is_default_init_already = 1;
        init_and_watch();
    }
}

sub init {
    ($qbit_app) = @_;
    Log::Log4perl::init(get_config_file());
}

sub init_and_watch {
    my ($app) = @_;
    $qbit_app = $app;
    Log::Log4perl::init_and_watch(get_config_file(), 10);
}

sub set_app {
    my ($app) = @_;
    $qbit_app = $app;
}

sub get_config_file {
    my $path          = lib::abs::path('../..');
    my $dev_log_file  = $path . '/nginx/log4perl.conf';
    my $prod_log_file = '/etc/partner/log4perl.conf';
    my $dev_null      = '/dev/null';

    my $log_file;

    if (-e $prod_log_file) {
        $log_file = $prod_log_file;
    } elsif (-e $dev_log_file) {
        $log_file = $dev_log_file;

        my $term_log_file = $log_file . '.term';
        if (-t STDOUT || $ENV{TAP_VERSION} || $ENV{FORCE_LOGGER_TO_SCREEN}) {
            unless (-e $term_log_file) {
                my $content = File::Slurp::read_file($log_file, binmode => ':utf8');
                $content =~ s|(\.logger.*?=\s?\w+,).*|$1 Screen|g;

                File::Slurp::write_file($term_log_file, {binmode => ':utf8'}, $content);
            }
            $log_file = $term_log_file;
        }
    } else {
        $log_file                                      = $dev_null;
        $Log::Log4perl::Config::CONFIG_INTEGRITY_CHECK = 0;
    }

    return $log_file;
}

sub DEBUG {
    local $Log::Log4perl::caller_depth = $Log::Log4perl::caller_depth + 1;
    _log($DEBUG, @_);
}

sub DEBUGF {
    local $Log::Log4perl::caller_depth = $Log::Log4perl::caller_depth + 1;
    _log($DEBUG, sprintf(shift, @_));
}

sub ERROR {
    local $Log::Log4perl::caller_depth = $Log::Log4perl::caller_depth + 1;
    _log($ERROR, @_);
}

sub ERRORF {
    local $Log::Log4perl::caller_depth = $Log::Log4perl::caller_depth + 1;
    _log($ERROR, sprintf(shift, @_));
}

sub FATAL {
    local $Log::Log4perl::caller_depth = $Log::Log4perl::caller_depth + 1;
    _log($FATAL, @_);
}

sub FATALF {
    local $Log::Log4perl::caller_depth = $Log::Log4perl::caller_depth + 1;
    _log($FATAL, sprintf(shift, @_));
}

sub INFO {
    local $Log::Log4perl::caller_depth = $Log::Log4perl::caller_depth + 1;
    _log($INFO, @_);
}

sub INFOF {
    local $Log::Log4perl::caller_depth = $Log::Log4perl::caller_depth + 1;
    _log($INFO, sprintf(shift, @_));
}

sub TRACE {
    local $Log::Log4perl::caller_depth = $Log::Log4perl::caller_depth + 1;
    _log($INFO, @_);
}

sub TRACEF {
    local $Log::Log4perl::caller_depth = $Log::Log4perl::caller_depth + 1;
    _log($TRACE, sprintf(shift, @_));
}

sub WARN {
    local $Log::Log4perl::caller_depth = $Log::Log4perl::caller_depth + 1;
    _log($WARN, @_);
}

sub WARNF {
    local $Log::Log4perl::caller_depth = $Log::Log4perl::caller_depth + 1;
    _log($WARN, sprintf(shift, @_));
}

sub _args_to_exception {
    my (@args) = @_;
    local $Log::Log4perl::caller_depth = $Log::Log4perl::caller_depth + 1;

    if (@args == 1 && blessed($args[0]) && $args[0]->isa('Exception')) {
        # try {
        #      ....
        # } catch {
        #     my ($exception) = @_;
        #     ERROR $exception;
        # }
        return ($args[0], $args[0]->as_string(1));
    } elsif (
        @args == 1 && ref($args[0]) eq 'HASH' && grep {
            $args[0]{$_}
        } (qw(exception message messages))
      )
    {
        # try {
        #      ....
        # } catch {
        #     my ($exception) = @_;
        #     ERROR {
        #         (exception|message|messages) => ($exception|'something went wrong'|['msg1', $msg2]),
        #         extra       => {some_additional_data => 'available_from_context'},
        #         fingerprint => ['some', 'array', 'of', 'strings', 'that', 'identifies', 'issue', 'for', 'this', 'event'],
        #     };
        # }

        my $exception = $args[0]{exception}
          // Log4Perl::Message->new(join('', _get_pretty_dumper(@{$args[0]{messages} // [$args[0]{message}]})),
            caller_depth => $Log::Log4perl::caller_depth);

        if (ref($args[0]{extra}) eq 'HASH') {
            if (ref($exception->{sentry}{extra}) eq 'HASH') {
                %{$exception->{sentry}{extra}} = (%{$exception->{sentry}{extra}}, %{$args[0]{extra}});
            } else {
                $exception->{sentry}{extra} = $args[0]{extra};
            }
        }

        $exception->{sentry}{fingerprint} = $args[0]{fingerprint} if defined($args[0]{fingerprint});

        my $message = $args[0]{exception} ? $exception->as_string(1) : $exception->message;

        return ($exception, $message);
    } else {
        # ERROR 'just', ' ', 'regular' ,' ',  $message;
        my $exception =
          Log4Perl::Message->new(join('', _get_pretty_dumper(@args)), caller_depth => $Log::Log4perl::caller_depth);
        return ($exception, $exception->message);
    }
}

sub _log {
    my ($level, @args) = @_;
    local $Log::Log4perl::caller_depth = $Log::Log4perl::caller_depth + 1;
    my ($exception, $message) = _args_to_exception(@args);
    if (defined($qbit_app)) {
        my $time_sec      = $qbit_app->get_time();
        my $extra_message = "Time is: $time_sec sec";
        if (!$ENV{TAP_VERSION} && $sentry_level{$level} && !$exception->{dont_send_to_sentry}) {
            try {
                $qbit_app->sentry->send_exception($exception, level => $sentry_level{$level},);
            }
            catch {
                my ($e) = @_;
                $exception->{sentry}{error} = $e->message;
            };

            if ($exception->{sentry}) {
                my $sentry_text =
                  sprintf('Sentry url: %s', $exception->{sentry}->{event_url} // $exception->{sentry}->{error});
                $extra_message .= "\n    " . $sentry_text;
            }

        }
        $message =~ s/(Filename: .*)/$1\n    $extra_message/;
    }

    get_logger()->log($level, $message);

    return 1;
}

=head2 get_logger

=cut

sub get_logger {
    my $caller = (caller(2) // caller())[0];
    $logger{$caller} //= Log::Log4perl->get_logger($caller);
    return $logger{$caller};
}

sub _get_pretty_dumper {
    my @args = @_;

    if (-t STDOUT || $ENV{TAP_VERSION}) {
        unless (defined $is_dataprinter) {
            $is_dataprinter = (-e ($ENV{'HOME'} // '') . '/.dataprinter') ? 1 : 0;
        }

        local $Data::Dumper::Indent   = 1;
        local $Data::Dumper::Sortkeys = 1;

        for (my $i = 0; $i < @args; $i++) {
            if (ref $args[$i]) {
                if ($is_dataprinter) {
                    require DDP;
                    DDP->import({sort_keys => 1});
                    my $struct = clone($args[$i]);    # fix DDP bug that brokes iitital structure inexplicably
                    $args[$i] = Data::Printer::np($args[$i], colored => 1, caller_info => 0);
                } else {
                    $args[$i] = Data::Dumper::Dumper($args[$i]);
                }
            }
        }
    }

    return @args;
}

1;
