package Utils::MakeFile;

use qbit;
use lib::abs;

use File::Slurp qw(write_file read_file);

use PiConstants qw($CONFIG_FILE_PATH  $DB_CONFIG_FILE_PATH $CONFIG_FILE_NAME);
use PiSecrets;
use Utils::Deploy qw(
  get_revision_from_status
  get_stage_status
  resolve_db_endpoint
  wait_deploy_status_ready
  );
use Utils::Sandbox qw(
  create_and_start_sandbox_task
  wait_sandbox_task
  );
use Utils::Logger qw( INFO  INFOF );

our @EXPORT    = qw(get_java_jsonapi_port);
our @EXPORT_OK = @EXPORT;

my $ONESHOT_DEPLOY_STAGE_NAME     = 'partner-oneshot-db';
my $SQL_CONNECT_TRY               = 300;
my $SQL_CONNECT_WAIT              = 2;
my $SANDBOX_ONESHOT_REDEPLOY_TASK = 'PARTNER_REDEPLOY_ONESHOT_STAGE';

sub set_bk_on_prod {
    my $json = get_config_json_data($CONFIG_FILE_PATH);
    my $host = 'http://localhost:48000/';
    $json->{api_bk}->{url}      = $host;
    $json->{api_http_bk}->{url} = $host;
    _write_config($CONFIG_FILE_PATH, $json);
}

sub set_balance_on_prod {
    my $json = get_config_json_data($CONFIG_FILE_PATH);
    $json->{api_balance}->{url} = 'http://localhost:8002/xmlrpc';
    _write_config($CONFIG_FILE_PATH, $json);
}

sub set_mol_on_prod {
    my $json = get_config_json_data($CONFIG_FILE_PATH);
    $json->{api_http_mol}->{url} = 'http://bsfastexport.yandex.ru/';
    _write_config($CONFIG_FILE_PATH, $json);
}

sub set_db_on_prod {
    _change_db_config(get_secret('connection-to-partner2-prod-database'));
}

sub set_db_on_test {
    _change_db_config(get_secret('connection-to-partner2-testing-database'));
}

sub _change_db_config {
    my ($config) = @_;

    my $json = get_config_json_data($DB_CONFIG_FILE_PATH);

    $json->{partnerdb2} = {
        host => '127.0.0.1',
        %$config
    };

    _write_config($DB_CONFIG_FILE_PATH, $json);
}

sub _get_db_docker {
    my ($image_name, $port) = @_;

    if ($image_name xor $port) {
        die "image_name and port must be both defined or no";
    }
    unless ($port) {
        my $init_sh = lib::abs::path('../../nginx/init.sh');
        for (read_file($init_sh)) {
            if (/^\s*HTTP_PORT\s*=\s*(\d+)$/) {
                $port = $1 + 100;
                last
            }
        }
        die "cannot calculate port: '$init_sh'" unless $port;
    }
    $image_name = "beta_partner2_$port-db";
    return ($image_name, $port);
}

sub delete_db_on_docker {
    my ($image_name, $port) = _get_db_docker(@_);
    my $not_verbose = !$ENV{VERBOSE_DB_DOCKER};
    if (run_shell("docker ps -q -f name=$image_name", silent => $not_verbose)) {
        run_shell("docker rm -f $image_name", silent => $not_verbose);
    }
}

sub set_db_on_docker {
    my ($image_name, $port) = _get_db_docker(@_);

    delete_db_on_docker($image_name, $port);

    my $sql_connect_try = $ENV{SQL_CONNECT_TRY} || $SQL_CONNECT_TRY;
    my $not_verbose = !$ENV{VERBOSE_DB_DOCKER};

    my $repo = 'registry.yandex.net/partners/partner2-db-general';
    my @result = split /\s+/, run_shell("docker images $repo | tail -n +2 | sort | tail -n 1", silent => $not_verbose);

    run_shell("docker run --detach --publish $port:3306 --name $image_name $repo:$result[1]", silent => $not_verbose);

    my $cmd_wait_mysql = "
        docker exec -t $image_name bash -c \"
        for ((TRY=$sql_connect_try ; TRY > 0 ; TRY-- )); do
            if service mysql status >/dev/null;
                then break;
            fi;
            sleep $SQL_CONNECT_WAIT;
        done;
        echo -n \\\$TRY
        \"";

    unless (run_shell($cmd_wait_mysql, silent => $not_verbose)) {
        die "Service mysql on $image_name not running\n";
    }

    my $host  = '127.0.0.1';
    my $login = 'root';
    my $c_template =
        "mysql -e 'select 1'"
      . " -u $login"
      . " --host $host"
      . " --port $port"
      . " --default-character-set=utf8"
      . " --disable-column-names -B %s"
      . " 2>/dev/null";

    unless (run_shell(sprintf($c_template, ""), silent => $not_verbose, skip_exit_code => TRUE)) {
        die "Cannot connect to mysql on $image_name ($host:$port)\n";
    }

    unless (run_shell(sprintf($c_template, "-D partner"), silent => $not_verbose, skip_exit_code => TRUE)) {
        die "Cannot use database partner on $image_name ($host:$port)\n";
    }

    my $json = get_config_json_data($DB_CONFIG_FILE_PATH);
    $json->{partnerdb2} = {
        host     => $host,
        port     => $port,
        user     => $login,
        password => '',
    };
    _write_config($DB_CONFIG_FILE_PATH, $json);
}

sub set_db_on_oneshot {
    my ($host, $port) = resolve_db_endpoint('sas', $ONESHOT_DEPLOY_STAGE_NAME);
    my $db_config = {
        partnerdb2 => {
            host     => $host,
            port     => $port,
            user     => 'root',
            password => '',
        },
    };

    writefile(lib::abs::path('../../lib/DatabaseConfig.json'), to_json($db_config, pretty => 1));
}

sub setup_db_on_oneshot {
    INFOF 'Start update stage "%s"', $ONESHOT_DEPLOY_STAGE_NAME;

    my $prev_revision = get_revision_from_status(get_stage_status('', $ONESHOT_DEPLOY_STAGE_NAME));

    INFOF 'Deploy stage "%s" current revision %s', $ONESHOT_DEPLOY_STAGE_NAME, $prev_revision;

    my $task_id = create_and_start_sandbox_task($SANDBOX_ONESHOT_REDEPLOY_TASK);

    INFOF 'Start waiting sandbox task "%s" - https://sandbox.yandex-team.ru/task/%s', $SANDBOX_ONESHOT_REDEPLOY_TASK,
      $task_id;

    wait_sandbox_task($task_id);

    wait_deploy_status_ready('', $ONESHOT_DEPLOY_STAGE_NAME, 1200, $prev_revision);

    INFO 'Start updating DatabaseConfig';
    set_db_on_oneshot();

    INFOF 'Stage "%s" updated successfully', $ONESHOT_DEPLOY_STAGE_NAME;
}

sub set_ch_on_prod {

    my $prod_config = get_config_json_data(lib::abs::path('../../configs/production/Application.json'));

    $prod_config->{'clickhouse_db'}{'user'}     = get_secret('partner2-clickhouse-user-production');
    $prod_config->{'clickhouse_db'}{'password'} = get_secret('partner2-clickhouse-password-production');

    _merge_application_configs({clickhouse_db => $prod_config->{'clickhouse_db'}});
}

sub _merge_application_configs {
    my ($patch, $config) = @_;

    my $need_save = !defined($config);

    $config //= get_config_json_data($CONFIG_FILE_PATH);

    foreach my $param (sort keys(%$patch)) {
        if (ref($patch->{$param}) eq 'HASH') {
            _merge_application_configs($patch->{$param}, $config->{$param});
        } else {
            $config->{$param} = $patch->{$param};
        }
    }

    _write_config($CONFIG_FILE_PATH, $config) if $need_save;
}

sub set_ch_on_test {
    my $prod_config = get_config_json_data(lib::abs::path('../../configs/test/Application.json'));

    $prod_config->{'clickhouse_db'}{'user'}     = get_secret('partner2-clickhouse-user-testing');
    $prod_config->{'clickhouse_db'}{'password'} = get_secret('partner2-clickhouse-password-testing');

    _merge_application_configs({clickhouse_db => $prod_config->{'clickhouse_db'}});
}

sub _set_something_on_stage {
    my ($stage, $name) = @_;

    my $stage_config = get_config_json_data(lib::abs::path("../../configs/$stage/$CONFIG_FILE_NAME"));
    _merge_application_configs({$name => $stage_config->{$name}});
}

sub set_yt_on_prod {
    _set_something_on_stage('production', 'yt');
}

sub set_page_id_generator_on_prod {
    _set_something_on_stage('production', 'api_utils_partner2');
}

sub set_page_id_generator_on_test {
    _set_something_on_stage('test', 'api_utils_partner2');
}

sub get_config_json_data {
    my ($config_path) = @_;
    my $content = read_file($config_path, {binmode => ':utf8'});
    my $json = from_json($content);
    return $json;
}

sub get_prod_db_hosts {
    my $json = `curl -s "https://c.yandex-team.ru/api-cached/groups2hosts/partner_rtb_db?format=json"`;
    chomp($json);

    return from_json($json);
}

sub _write_config {
    my ($config_path, $data) = @_;
    write_file($config_path, {binmode => ':utf8'}, to_json($data, pretty => 1));
    return 1;
}

sub set_java_jsonapi_port {
    my $json              = get_config_json_data($CONFIG_FILE_PATH);
    my $token             = '[%JAVA_JSONAPI_PORT%]';
    my $java_jsonapi_port = get_java_jsonapi_port();
    foreach my $k (keys %$json) {
        if ('HASH' eq ref $json->{$k} && $json->{$k}{url}) {
            $json->{$k}{url} =~ s/\Q$token\E/$java_jsonapi_port/;
        }
    }
    _write_config($CONFIG_FILE_PATH, $json);
}

sub get_java_jsonapi_port {
    my $java_jsonapi_port = `command grep -oP 'JAVA_JSONAPI_PORT=\\K\\d+' nginx/init.sh`;
    $java_jsonapi_port =~ s/\D//g;
    return $java_jsonapi_port;
}

1;
