package Utils::MigrationApplier;

use qbit;

use base qw(Exporter);

use Pod::Usage;
use Getopt::Long qw();
use Application;
use File::Slurp qw(read_file);
use Utils::Logger qw(INFO INFOF ERROR ERRORF), {logger => 'Screen',};
use Utils::DB;
use File::Copy qw(copy);
use PiConstants qw($CONFIG_FILE_PATH  $DB_CONFIG_FILE_PATH);

use File::Basename;
use Exception::DB;

use lib::abs qw();

our @EXPORT    = qw(run_migrations);
our @EXPORT_OK = @EXPORT;

my $AFTER_RELEASE_DIR     = lib::abs::path('../../migrations/after_release');
my $BEFORE_RELEASE_DIR    = lib::abs::path('../../migrations/before_release');
my $PERL_LIB_DIR          = lib::abs::path('../../lib');
my $CARTON_LIB_DIR        = lib::abs::path('../../local/lib/perl5');
my $MIGRATIONS_EXCLUDE_RE = '\.(json|yson)$';
my %NON_CRITICAL_SKIP     = ('MANUAL RUN' => TRUE,);
my $CHECK_RIGHTS;

sub run_migrations {
    my ($app, $opts) = @_;

    throw "type and order - requred options" unless $opts->{type} && $opts->{order};

    $app->partner_db->lock(tables => {migrations => 'write'});

    my $completed_migrations = get_completed_migrations($app, $opts->{type});
    my @orders = ($opts->{order} eq 'all' ? qw(before after) : $opts->{order});
    my $result = TRUE;
    my %executed_migrations;
    foreach my $order (@orders) {
        my $new_migrations = get_new_migration_scripts($opts->{type}, $order, $completed_migrations);

        my ($migrations_tree, $skipped_migrations) =
          check_dependencies($new_migrations, $opts->{without_manual}, $completed_migrations);

        my $onemore;
        do {
            $onemore = FALSE;
            foreach my $migration (keys %$migrations_tree) {
                $migrations_tree->{$migration} = [grep {!$executed_migrations{$_}} @{$migrations_tree->{$migration}}];
                unless (@{$migrations_tree->{$migration}}) {
                    delete $migrations_tree->{$migration};
                    if (execute_migration($app, $migration, $opts->{dry_run})) {
                        $executed_migrations{basename($migration)} = TRUE;
                        $onemore = TRUE;
                    } else {
                        $result = FALSE;
                    }
                }
            }
        } while (%$migrations_tree && $onemore);

        if (%$migrations_tree) {
            push @{$skipped_migrations->{'INCOMPLETE DEPENDENCIES'}}, keys %$migrations_tree;
        }

        if (scalar keys %$skipped_migrations) {
            foreach my $reason (sort keys %$skipped_migrations) {
                if (scalar @{$skipped_migrations->{$reason}}) {
                    #При изменении следующей строки надо помнить, что в релизном роботе
                    #данный вывод грепается для определения наличия пропущенных миграций
                    #https://a.yandex-team.ru/arc_vcs/partner/infra/partner_ansible/roles/partner2_release_robot/files/script?rev=users%2Fdioksin%2FPI-28691_fix_migration_detection#L1594
                    my $msg = "SKIP because of $reason: " . join(', ', @{$skipped_migrations->{$reason}});
                    if ($NON_CRITICAL_SKIP{$reason}) {
                        INFO $msg;
                    } else {
                        ERROR $msg;
                    }
                } else {
                    ERRORF 'skip reason "%s" is defined but empty', $reason;
                }
            }
        }
    }

    $app->partner_db->unlock();

    reset_config();

    return $result ? 0 : $opts->{dont_fail} ? 0 : -1;
}

sub execute_migration {
    my ($app, $migration, $dry_run) = @_;

    #При изменении следующей строки надо помнить, что в релизном роботе
    #данный вывод грепается для определения наличия миграций
    #https://a.yandex-team.ru/arc_vcs/partner/infra/partner_ansible/roles/partner2_release_robot/files/script?rev=r9425542#L1596
    print "Try to execute $migration\n";
    return TRUE if $dry_run;

    my $start_time = curdate(oformat => 'db_time');
    my $type       = 'mysql';
    my $order      = 'before';
    $type  = 'clickhouse' if $migration =~ /\/clickhouse\//;
    $order = 'after'      if $migration =~ /\/after_release\//;

    my $result;
    my $result_code = 1;

    try {
        if ($migration =~ /\.(sql|ch)$/i) {
            execute_sql_migration($app, $type, $order, $migration);
            $result_code = 0;
        } elsif ($migration =~ /\.pl$/) {
            $result      = `perl -I $CARTON_LIB_DIR -I $PERL_LIB_DIR $migration 2>&1`;
            $result_code = $? >> 8;
            INFO $result;
            ERROR "Migration $migration exited with code $result_code" if $result_code;
        } elsif ($migration =~ /\.sh$/) {
            $result      = `$migration 2>&1`;
            $result_code = $? >> 8;
            INFO $result;
            ERROR "Migration $migration exited with code $result_code" if $result_code;
        } else {
            ERROR "Can't determine migration type for $migration";
            $result_code = 255;
        }
        INFOF 'Migration %s successfully executed', $migration unless $result_code;
    }
    catch {
        my ($e) = @_;
        ERROR "Migration $migration executed with errors " . $e->message;
        $result_code = 255;
    };
    mark_migration_as_executed($app, $type, $order, basename($migration), $start_time, $result_code);

    return $result_code ? FALSE : TRUE;
}

sub mark_migration_as_executed {
    my ($app, $type, $order, $migration, $start_time, $result_code) = @_;
    my $res = $app->partner_db->migrations->add(
        {
            source_type => 'perl',
            order_type  => $order,
            db_type     => $type,
            name        => $migration,
            time_start  => $start_time,
            time_stop   => curdate(oformat => 'db_time'),
            result_code => $result_code,
        },
        duplicate_update => TRUE,
    );
}

sub execute_sql_migration {
    my ($app, $type, $order, $migration) = @_;

    my $content = read_file($migration, binmode => ':utf8');
    if ($type eq 'mysql') {
        my $settings = Utils::DB::get_db_settings()->{partner_db};

        my $cmd = sprintf("/usr/bin/mysql --user=%s --password=%s --port=%s --host=%s -A partner -e 'source %s' 2>&1",
            $settings->{user}, $settings->{password}, $settings->{port}, $settings->{host}, $migration,);

        INFO "Executing query: $content";
        my $res = `$cmd`;
        $res =~ s/^mysql: \[Warning\] Using a password on the command line interface can be insecure.//;
        my @errors = ($res =~ /(ERROR.+\n)/);

        if ($? != 0 && @errors) {
            throw Exception::DB join("\n", @errors);
        } else {
            INFO "Result: $res";
        }
    } elsif ($type eq 'clickhouse') {
        my @commands = split /;\s*?\n\s*/, $content;
        no warnings 'once';
        local $QBit::Application::Model::DB::clickhouse::st::FORMAT = 'TabSeparated';
        foreach my $command (@commands) {
            my $res = $app->clickhouse_db->_do($command);
        }
    }
}

my @not_select = qw(update set insert replace delete create alter drop);
my @is_select  = qw(select);
my $qr         = join '|', map {"\\b$_\\b"} (@not_select, @is_select);

sub _is_select_sql {
    my ($sql) = @_;

    my $result = FALSE;
    if ($sql =~ /^.*?($qr)/is) {
        my $w = lc $1;
        $result = in_array($w, \@is_select) ? TRUE : FALSE;
    }
    return $result;
}

sub check_dependencies {
    my ($migrations, $without_manual, $completed_migrations) = @_;
    my %skiped;
    my $migration_tree;

    foreach my $type (keys %{$migrations}) {
        foreach my $order (keys %{$migrations->{$type}}) {
            foreach my $migration (@{$migrations->{$type}->{$order}}) {
                if ($without_manual && $migration =~ /MANUAL/) {
                    push @{$skiped{'MANUAL RUN'}}, $migration;
                    next;
                }
                if (open(FILE, $migration)) {
                    my $dependencies = <FILE>;
                    $dependencies = <FILE> unless $dependencies =~ /#dependencies/;
                    close(FILE);
                    if ($dependencies && $dependencies =~ /#dependencies/) {
                        $dependencies =~ s/.*#dependencies([^\n]*).*/$1/;
                        $dependencies =~ s/\s//g;
                        my @deps = split ',', $dependencies;
                        $migration_tree->{$migration} = [
                            grep {
                                     !$completed_migrations->{mysql}->{before}->{$_}
                                  && !$completed_migrations->{mysql}->{after}->{$_}
                                  && !$completed_migrations->{clickhouse}->{before}->{$_}
                                  && !$completed_migrations->{clickhouse}->{after}->{$_}
                              } @deps
                        ];
                    } else {
                        $migration_tree->{$migration} = [];
                    }
                } else {
                    push @{$skiped{'IO ERROR'}}, $migration;
                }
            }
        }
    }

    return $migration_tree, \%skiped;
}

sub get_scripts_from_dir {
    my ($dir, $type) = @_;
    my @scripts;
    my @dirs = ($dir);
    while (@dirs) {
        my $current_dir = shift @dirs;
        my $dh;
        unless (opendir $dh, $current_dir) {
            ERROR "Cannot open directory '$current_dir': $!";
            next;
        }
        while (my $entry = readdir $dh) {
            next if $entry =~ /^\./;
            next if $entry =~ /$MIGRATIONS_EXCLUDE_RE/;

            my $fullname = "$current_dir/$entry";

            if (-d $fullname) {
                push @dirs, $fullname;
            } elsif ($fullname =~ m|/$type/|) {
                push @scripts, $fullname;
            }
        }
        closedir $dh;
    }
    return @scripts;
}

sub get_new_migration_scripts {
    my ($type, $order, $completed_migrations) = @_;

    my $path;
    my $scripts = {};
    if ($order eq 'before' || $order eq 'all') {
        if ($type eq 'mysql' || $type eq 'all') {
            push @{$scripts->{mysql}->{before}},
              grep {!$completed_migrations->{mysql}->{before}->{basename($_)}}
              get_scripts_from_dir($BEFORE_RELEASE_DIR, 'mysql');
        }
        if ($type eq 'clickhouse' || $type eq 'all') {
            push @{$scripts->{clickhouse}->{before}},
              grep {!$completed_migrations->{clickhouse}->{before}->{basename($_)}}
              get_scripts_from_dir($BEFORE_RELEASE_DIR, 'clickhouse');
        }
    }
    if ($order eq 'after' || $order eq 'all') {
        if ($type eq 'mysql' || $type eq 'all') {
            push @{$scripts->{mysql}->{after}},
              grep {!$completed_migrations->{mysql}->{after}->{basename($_)}}
              get_scripts_from_dir($AFTER_RELEASE_DIR, 'mysql');
        }
        if ($type eq 'clickhouse' || $type eq 'all') {
            push @{$scripts->{clickhouse}->{after}},
              grep {!$completed_migrations->{clickhouse}->{after}->{basename($_)}}
              get_scripts_from_dir($AFTER_RELEASE_DIR, 'clickhouse');
        }
    }

    return $scripts;
}

sub get_completed_migrations {
    my ($app, $type, $order) = @_;
    my $completed_migrations;
    my $migrations = $app->partner_db->query->select(
        table  => $app->partner_db()->migrations,
        fields => [qw(db_type order_type name)],
        filter => [OR => [[result_code => '==' => \0], [result_code => '==' => \undef]]],
    )->get_all();

    $completed_migrations->{$_->{db_type}}{$_->{order_type}}{$_->{name}} = TRUE foreach (@{$migrations});

    return $completed_migrations;
}

my $config_copy;

sub reset_config {

    if ($config_copy) {
        unlink $CONFIG_FILE_PATH;
        unlink $DB_CONFIG_FILE_PATH;
        if ($config_copy->{app}) {
            try {
                copy($config_copy->{app}, $CONFIG_FILE_PATH);
            }
            catch {
                my ($e) = @_;
                ERROR {
                    exception => $e,
                    message   => 'Cannot restore ApplicationConfig',
                };
            };
        }
        if ($config_copy->{db}) {
            try {
                copy($config_copy->{db}, $DB_CONFIG_FILE_PATH);
            }
            catch {
                my ($e) = @_;
                ERROR {
                    exception => $e,
                    message   => 'Cannot restore DatabaseConfig',
                };
            };
        }
    }
}

sub int_handler {
    reset_config();
    exit 0;
}

sub init_config {
    my ($opts) = @_;

    if ($opts->{production}) {
        $SIG{__DIE__} = \&int_handler;
        $SIG{INT}     = \&int_handler;
        $SIG{QUIT}    = \&int_handler;
        $SIG{TERM}    = \&int_handler;
        $config_copy  = {};
        if (-e $CONFIG_FILE_PATH) {
            try {
                my $fn = "$CONFIG_FILE_PATH.$$.bak";
                copy($CONFIG_FILE_PATH, $fn);
                $config_copy->{app} = $fn;
            }
            catch {
                my ($e) = @_;
                ERROR {
                    exception => $e,
                    message   => 'Cannot save ApplicationConfig',
                };
            };
        }
        if (-e $DB_CONFIG_FILE_PATH) {
            try {
                my $fn = "$DB_CONFIG_FILE_PATH.$$.bak";
                copy($DB_CONFIG_FILE_PATH, $fn);
                $config_copy->{db} = $fn;
            }
            catch {
                my ($e) = @_;
                ERROR {
                    exception => $e,
                    message   => 'Cannot save DatabaseConfig',
                };
            };
        }
        system("make config_production");
    }
}

sub DESTROY {
    warn "in destroy";
    reset_config();
}

TRUE;
