package Utils::MonitoringUtils;

use qbit;

use Math::Round;
use Socket;

use Partner2::Juggler::API;
use PiSecrets;
use Utils::Logger qw(INFOF WARN);

our @ISA    = qw(Exporter);
our @EXPORT = qw(
  _format_data_for_monitoring_email_letter
  send_to_graphite
  get_pjapi
  );

our $host_cached;

my $APP;
my $PJA;

sub set_application {
    my ($app) = @_;

    weaken($app);

    $APP = $app;
}

=head1 send_to_graphite

    send_to_graphite(
        interval => 'five_min', # Дефолтное значение five_min
        path => 'number_of_v',  # Название параметра
        value => 2,             # Значение параметра
    );

Low level метод отправки данных в демон graphite-sender.

Данные в графите можно посмотреть по адресу:

http://ppcgraphite.yandex.ru/render?target=NNNNNN

Где, NNNNNN - это значение interval.host.path

Подробности: https://wiki.yandex-team.ru/partner/w/partner2-graphite

=cut

sub send_to_graphite {
    my (%opts) = @_;

    my $sender_host = '127.0.0.1';
    my $sender_port = '42000';

    my $interval = $opts{'interval'} // 'five_min';

    my $path = $opts{'path'};
    die 'No path' if !defined($path);

    $host_cached ||= get_hostname();
    my $host = $host_cached;
    $host =~ s/\./_/g;

    my $full_path = "$interval.$host.$path";

    my $value = $opts{'value'};
    die 'No value' if !defined($value);

    my $timestamp = $opts{'timestamp'} // time;

    my $data_to_send = "$full_path $value $timestamp\n";

    if ($opts{log_only}) {
        INFOF $data_to_send;
        return '';
    }

    if ($opts{'send_to_solomon'} // TRUE) {
        try {
            $APP->api_solomon->send(
                ts      => $timestamp,
                sensors => [
                    {
                        labels => {
                            sensor   => $path,
                            interval => $interval,
                            %{$opts{solomon} // {}},
                        },
                        kind  => 'DGAUGE',
                        value => $value,
                    }
                ],
            );
        }
        catch {
            WARN shift->message();
        };
    }

    return '';
}

sub _fix_value_to_view {
    my ($value, $type) = @_;

    my $result = format_number(
        $value,
        thousands_sep => ' ',
        decimal_point => ',',
        ($type eq 'number' ? (precision => 0) : (precision => 2))
    );

    $result =~ s/^-/&ndash;&thinsp;/;

    return $result;
}

sub _format_data_for_monitoring_email_letter {
    my ($pk, $data_f, $data_s, $fields) = @_;

    my $result = {pk => $pk};

    $result->{'week_ago'}{$_} = $data_f->{$_} foreach grep {!$fields->{$_}} keys(%$data_f);
    $result->{'date'}{$_}     = $data_s->{$_} foreach grep {!$fields->{$_}} keys(%$data_s);

    foreach my $field (keys(%$fields)) {
        $data_f->{$field} = 0 if !$data_f->{$field} || $data_f->{$field} eq '-';
        $data_s->{$field} = 0 if !$data_s->{$field} || $data_s->{$field} eq '-';

        my $diff = $data_s->{$field} - $data_f->{$field};
        $result->{'diff_sort'}{$field} = abs($diff);

        if ($data_f->{$field} + 0) {
            my $rate = ($diff * 100) / $data_f->{$field};
            $rate = _fix_value_to_view($rate, 'number');
            $result->{'rate'}{$field} = "$rate&thinsp;%";
        } elsif ($diff == 0) {
            $result->{'rate'}{$field} = '0&thinsp;%';
        } else {
            $result->{'rate'}{$field} = '100&thinsp;%';
        }

        $result->{'diff'}{$field} = _fix_value_to_view($diff, $fields->{$field}{'type'});

        $result->{'week_ago'}{$field} = _fix_value_to_view($data_f->{$field}, $fields->{$field}{'type'});
        $result->{'date'}{$field}     = _fix_value_to_view($data_s->{$field}, $fields->{$field}{'type'});
    }

    return $result;
}

sub get_pjapi {
    my ($self) = @_;

    my $api_juggler = $self->get_option('api_juggler');
    return Partner2::Juggler::API->new(
        juggler_token => get_secret('juggler-token'),
        host          => $api_juggler->{'host'},
        namespace     => $api_juggler->{'namespace'},
    );
}

TRUE;
