package Utils::Oneshots::AddBlockBySample;

=encoding UTF-8

=head1 DESCRIPTION

Модуль предоставляет возможность добавлять блоки на пейджи по образцу

=head1 USAGE

  perl -Ilib bin/oneshots/add_blocks_by_template.pl

=head1 OPTIONS

  page_ids                 - Список ID площадок через запятую
  page_list                - Список ID площадок в файле (построчно)
  page_multistate_filter   - Дополнительный фильтр для пейджей по статусу
  sample                   - Список блоков-образцов для создания
  rollback                 - лог для отката
  exclude_page_ids         - площадки-исключения, список
  exclude_page_list        - площадки-исключения, файл
  skip_empty               - пропускать площадки без блоков
  skip_without_stat        - пропускать площадки, где нет блоков со статистикой
  delete_blocks            - удалять все блоки с площадки перед копированием на неё
  delete_blocks_from_model - удалять блоки только с этих моделей
  delete_blocks_filter     - дополнительный фильтр на удаляемые блоки

=cut

use qbit;
use Utils::PublicID;
use File::Slurp qw(read_file);

sub main {
    my ($app, $opts) = @_;

    if ($opts->{exclude_page_ids}) {
        $opts->{exclude_page_ids} = [split /,/, $opts->{exclude_page_ids}];
    }
    if ($opts->{exclude_page_list}) {
        $opts->{exclude_page_ids} = [read_file($opts->{exclude_page_list}, chomp => 1)];
    }
    if ($opts->{page_ids}) {
        $opts->{page_ids} = [split /,/, $opts->{page_ids}];
    }
    if ($opts->{page_list}) {
        $opts->{page_ids} = [read_file($opts->{page_list}, chomp => 1)];
    }
    if ($opts->{sample}) {
        $opts->{sample} = [split /,/, $opts->{sample}];
    }
    if ($opts->{delete_blocks_from_model}) {
        $opts->{delete_blocks_from_model} = [split /,/, $opts->{delete_blocks_from_model}];
    }

    if ($opts->{rollback}) {
        _rollback__archive_blocks($app, $opts);
    } else {
        _add_blocks($app, $opts);
    }
}

sub get_samples {
    my ($app, $opts) = @_;

    my @result;
    my $model_by_prefix = Utils::PublicID::get_block_accessors_by_public_id_prefixes($app);
    for my $pid (@{$opts->{sample}}) {
        my ($prefix, $page_id, $id) = split_block_public_id($pid);
        my $model = $model_by_prefix->{$prefix}[0];

        my $add_fields = $app->$model->get_add_fields();
        delete @{$add_fields}{qw(aadfox_block articles)};

        my $sample_block = $app->$model->get($pid, 'fields', [sort keys %$add_fields]);
        delete $sample_block->{page_id};

        push @result, {model => $model, data => $sample_block, sample => $pid};
    }
    return \@result;
}

sub _add_blocks {
    my ($app, $opts) = @_;

    my $samples = get_samples($app, $opts);

    my %exclude;
    if ($opts->{exclude_page_ids}) {
        @exclude{@{$opts->{exclude_page_ids}}} = ();
    }

    my @readonly;
    if ($opts->{make_readonly}) {
        @readonly = (readonly => 1);
    }
    my %pages;
    foreach my $page (@{$opts->{'page_ids'}}) {
        if (exists $exclude{$page}) {
            print logstr("EXCLUDE_PAGE: $page");
            next;
        }

        my $page_data = $app->partner_db->all_pages->get_all(
            fields => [qw(page_id model is_deleted multistate)],
            filter => [page_id => '=' => \$page],
        )->[0];
        if (!defined($page_data) || $page_data->{is_deleted}) {
            print logstr("DELETED_PAGE: $page");
            next;
        }

        my $page_model_name = $page_data->{model};
        my $page_model      = $app->$page_model_name;

        if (
            defined($opts->{'page_multistate_filter'})
            && !in_array(
                $page_data->{'multistate'},
                $page_model->get_multistates_by_filter($opts->{'page_multistate_filter'})
            )
           )
        {
            print logstr("FILTERED_BY_MULTISTATE_PAGE: $page");
            next;
        }

        if ($opts->{skip_empty} || $opts->{skip_without_stat}) {
            my $key = 'PAGE_NO_BLOCK';
            my @filter = ([multistate => '&' => \2]);
            if ($opts->{skip_without_stat}) {
                push @filter, {not => [[multistate => '&' => \4]]};
                $key = 'PAGE_NO_STAT';
            }

            my $data = $app->partner_db->query->select(
                table  => $app->partner_db->all_blocks,
                fields => {cnt => {count => ['id_autoincrement']}},
                filter => [AND => [[page_id => '=' => \$page], @filter]],
            )->get_all();

            unless ($data->[0]{cnt}) {
                print logstr("$key: $page");
                next;
            }
        }

        if ($opts->{delete_blocks}) {
            my @block_models = ();
            if ($opts->{delete_blocks_from_model}) {
                @block_models =
                  map {$app->$_}
                  grep {in_array($_, $page_model->get_block_model_names())} @{$opts->{delete_blocks_from_model}};
            } else {
                @block_models = $page_model->get_block_models();
            }

            foreach my $block_model (@block_models) {
                print logstr('DELETE_FROM_MODEL:', $block_model->accessor);

                my $blocks = $block_model->get_all(
                    fields => [qw(page_id id multistate adfox_block)],
                    filter => [
                        AND =>
                          [{page_id => $page}, {multistate => 'not deleted'}, ($opts->{delete_blocks_filter} // ()),]
                    ]
                );
                foreach my $block (@$blocks) {
                    if ($block->{adfox_block}) {
                        print logstr('DELETE_BLOCK_SKIP:', $block);
                        next;
                    }
                    print logstr("DELETE_BLOCK:", $block);
                    unless ($opts->{'dry_run'}) {
                        try {
                            $block_model->maybe_do_action($block, 'stop');
                            $block_model->do_action($block, 'delete');
                            $pages{$page} = TRUE;
                        }
                        catch {
                            print logstr("DELETE_BLOCK_FAIL:", $block, $@->message);
                        }
                    }
                }
            }
        }

        for my $sample (@$samples) {
            my $block2add = {%{$sample->{data}}, page_id => $page, @readonly};
            print logstr('PAGE', $page, $sample->{sample}, $block2add);

            try {
                unless ($opts->{'dry_run'}) {
                    my $model     = $sample->{model};
                    my $public_id = $app->$model->add(%$block2add);
                    print logstr("BLOCK_ADD", $page, $public_id);
                    $pages{$page} = TRUE;
                }
            }
            catch {
                my ($e) = @_;
                print logstr("BLOCK_FAIL: $page", $sample->{sample}, $e->message);
            };
        }
    }
    my @pages = keys %pages;
    $app->all_pages->mark_pages_for_async_update(page_ids => \@pages);
    print logstr("MARK_TO_RESEND: ", scalar(@pages));
}

sub _rollback__archive_blocks {
    my ($app, $opts) = @_;

    my %exclude;
    if ($opts->{exclude_page_ids}) {
        @exclude{@{$opts->{exclude_page_ids}}} = ();
    }
    if (open FH, $opts->{rollback}) {
        my %pages;
        my $model_by_prefix = Utils::PublicID::get_block_accessors_by_public_id_prefixes($app);

        my @block_list = map {/BLOCK_ADD\s+\d+\s+(\w+-\w+-\d+-\d+)/ ? $1 : ()} <FH>;
        for my $block_public_id (@block_list) {
            my ($prefix, $page_id, $id) = split_block_public_id($block_public_id);
            if (exists $exclude{$page_id}) {
                print logstr("SKIP_BLOCK: $block_public_id");
                next;
            }
            try {
                my $model = $model_by_prefix->{$prefix}[0];
                $app->$model->do_action($block_public_id, 'delete') unless ($opts->{dry_run});
                $pages{$page_id}++;
                print logstr("DELETE_BLOCK: $block_public_id");
            }
            catch {
                print logstr("FAIL_DELETE: $block_public_id", $_[0]->message);
            };
        }
        close FH;

        my @pages = keys %pages;
        $app->all_pages->mark_pages_for_async_update(page_ids => \@pages);
        print logstr("MARK_TO_RESEND: ", scalar(@pages));
    } else {
        print logstr("!could not open rollback_log file: $!");
    }
}

1;
