package Utils::Oneshots::AddPageBySample;

=encoding UTF-8

=head1 DESCRIPTION

Модуль предоставляет возможность добавлять блоки на пейджи по образцу

=head1 USAGE

  perl -Ilib bin/oneshots/add_pages_by_sample.pl

=head1 OPTIONS

=cut

use qbit;
use Utils::PublicID;
use File::Slurp qw(read_file);
use PiConstants qw($DSP_DIRECT_ID $OVERLAY);
use DDP;

sub main {
    my ($app, $opts) = @_;

    _prepare_opts($app, $opts);

    _add_pages($app, $opts);
}

sub _prepare_opts {
    my ($app, $opts) = @_;

    $opts->{'captions'} =
        $opts->{'caption_list'} ? [split(/,/, $opts->{'caption_list'} // '')]
      : $opts->{'caption_file'} ? [read_file($opts->{caption_file}, chomp => 1)]
      :                           [];

    unless (@{$opts->{'captions'}}) {
        print STDERR logstr('caption_list or caption_file should be defined...');
        exit;
    }

    if ($opts->{login}) {
        $opts->{login} = $app->users->get_all(
            fields => ['login'],
            filter => ['login', '=', $opts->{login}],
        )->[0]->{'login'};
        unless ($opts->{login}) {
            print STDERR logstr('login not found...');
            exit;
        }
    }

    $opts->{'target_tags'} = [split /,/, $opts->{'target_tags'}]
      if $opts->{'target_tags'};

    for my $opt_name (
        qw(
        caption_mask
        domain
        is_internal
        sample
        sample_type_only
        set_dsp_for_overlay_blocks
        partner_type
        )
      )
    {
        $opts->{$opt_name} = $opts->{$opt_name} // '';
    }
}

# moneymap page_id=1232045
sub _get_moneymap {
    my ($opts) = @_;
    return {
        'abc_id'               => 1558,
        'abc_oebs_id'          => 1558,
        'is_internal'          => $opts->{is_internal} ? TRUE : FALSE,
        'is_sandbox'           => FALSE,
        'login'                => undef,
        'os'                   => undef,
        'platform'             => undef,
        'traffic_sub_sub_type' => undef,
        'traffic_sub_type'     => 'video:video',
        'traffic_type'         => 'videoresources',
    };
}

sub _get_required_fields {
    return {
        platform            => '1',
        skip_delay          => '5',
        skip_time_left_show => '1',
        time_left_show      => '1',
        title               => 'Благодаря рекламе это видео для Вас бесплатно'
    };
}

sub _add_pages {
    my ($app, $opts) = @_;

    my $sample_page = $app->all_pages->get_all(filter => ['page_id', '=', $opts->{sample}],)->[0];

    my $page_model = $sample_page->{model};

    my $sample = $app->$page_model->get_all(
        fields => [keys %{$app->$page_model->get_add_fields($sample_page)}],
        filter => {'id' => $sample_page->{page_id}},
    )->[0];

    my @models_to_create =
      grep {$_ =~ /$opts->{sample_type_only}/} @{$app->$page_model->get_block_model_names()};

    print logstr('PAGE SAMPLE', $sample);
    for my $caption (@{$opts->{'captions'}}) {
        $caption = sprintf($opts->{caption_mask}, $caption)
          if $opts->{caption_mask};

        my $page2add = {
            %{_get_required_fields()},
            moneymap => _get_moneymap($opts),
            caption  => $caption
        };

        for (qw/login domain target_tags partner_type/) {
            $page2add->{$_} = $opts->{$_} if $opts->{$_};
        }

        try {
            unless ($opts->{'dry_run'}) {
                print logstr("PAGE_ADD", $page_model, $caption, $page2add);
                $app->partner_db->transaction(
                    sub {
                        my $public_id = $app->$page_model->add(%$page2add);
                        print logstr("PAGE_ADDED", $public_id);
                        _add_blocks(
                            $app,
                            {
                                %$opts,
                                page_id     => $sample_page->{page_id},
                                new_page_id => $public_id,
                                models      => \@models_to_create,
                            }
                        );
                    }
                );
            }
        }
        catch {
            my ($e) = @_;
            print logstr("PAGE_FAIL: $caption", $e->message);
        };
    }
}

sub _add_blocks {
    my ($app, $opts) = @_;

    for my $block_model (@{$opts->{models}}) {
        my $block_page_id_name = $app->$block_model->get_page_id_field_name();

        my $blocks = $app->$block_model->get_all(
            fields => [keys %{$app->$block_model->get_add_fields}],
            filter => [AND => [{multistate => 'working'}, {$block_page_id_name => $opts->{page_id}}]],
            order_by => [$block_page_id_name, 'id'],
        );

        map {delete $_->{moneymap}} @$blocks;

        print logstr('BLOCKS', scalar @$blocks, 'blocks will be created');

        my $dsp;
        $dsp = $app->dsp->get($DSP_DIRECT_ID) if $opts->{set_dsp_for_overlay_blocks};

        foreach my $block (@$blocks) {
            if ($opts->{set_dsp_for_overlay_blocks} && $block->{type} eq $OVERLAY) {
                $block->{dsps} = [{dsp_id => $dsp->{id}}];
            }

            my $block2add = {%$block, page_id => $opts->{new_page_id}};

            # only articles or picategories value enabled at same time
            delete $block2add->{articles};

            print logstr('BLOCK', $block->{caption}, $block2add);

            try {
                unless ($opts->{'dry_run'}) {
                    print logstr("BLOCK_ADD", $opts->{new_page_id});
                    my $public_id = $app->$block_model->add(%$block2add);
                    print logstr("BLOCK_START", $public_id);
                    my $tmp_rights = $app->add_all_tmp_rights();
                    $app->$block_model->do_action($public_id, 'start');
                }
            }
            catch {
                my ($e) = @_;
                print logstr("BLOCK_FAIL", $block->{caption}, $e->message);
            };
        }
    }
}

1;
