package Utils::Oneshots::ResendToBk;

use qbit;
use Time::HiRes qw(gettimeofday);

use Utils::Oneshots;
use Utils::MakeFile;

use PiConstants qw($VIP_PROTECTED_MAX_PAGE_ID);

my @fields = qw(id model multistate);

sub main {
    my ($app, $opts) = @_;

    check_java_jsonapi_is_running($app, $opts);

    my $page_ids = $opts->{page_ids};
    silent($app);

    local $PiConstants::VIP_PROTECTED_MAX_PAGE_ID = $VIP_PROTECTED_MAX_PAGE_ID;
    allow_send_special($app, $opts) if $opts->{allow_special};

    my $all_count = scalar(@$page_ids);
    my $count     = 1;

    print logstr('pages', sprintf 'total=%s', $all_count);

    my $has_error;
    my $need_exit = 0;
    foreach my $page_id (@$page_ids) {
        last if $need_exit;

        my $t1 = gettimeofday();

        my $filter = [page_id => '=' => \$page_id];
        if ($opts->{age}) {
            $filter = [AND => [$filter, [update_time => '<' => \$opts->{age}]]];
        }
        my $error;
        my $model = '--';

        try {
            my $page = Utils::Oneshots::repeater(
                sub {
                    $app->partner_db->all_pages->get_all(
                        fields => \@fields,
                        filter => $filter,
                    );
                },
                $opts->{timeout},
                $opts->{tries}
            )->[0];

            if ($page) {
                $model = $page->{model};
                my $page_id_field    = $app->$model->get_page_id_field_name();
                my $multistate_names = $app->$model->get_multistate_names();

                if (  !$app->$model->may_send_not_balance_registered
                    && $multistate_names->{balance_registered}
                    && !$app->$model->check_multistate_flag($page->{multistate}, 'balance_registered'))
                {
                    throw Exception gettext('Page ID %d not balance_registered', $page_id);
                }

                foreach my $status (grep {!$opts->{special_flags}{$_}} keys %{$opts->{special_flags}}) {
                    if (   $multistate_names->{$status}
                        && $app->$model->check_multistate_flag($page->{multistate}, $status))
                    {
                        throw Exception gettext('Page ID %d is %s', $page_id, $status);
                    }
                }

                # NOTE! Транзакция нужна чтобы выставленный 'need_update' не подхватил скрипт асинхронной отправки
                Utils::Oneshots::repeater(
                    sub {
                        $app->partner_db->transaction(
                            sub {
                                $app->$model->do_action($page->{id}, 'set_need_update') unless $opts->{logbroker_only};
                                $app->$model->update_in_bk(
                                    {$page_id_field => $page_id},
                                    do_not_change_multistate => $opts->{do_not_change_multistate},
                                    logbrocker_only          => $opts->{logbroker_only},
                                    protobuf_only            => $opts->{protobuf_only},
                                    is_from_java             => $opts->{from_java}
                                );
                            }
                        );
                    },
                    $opts->{timeout},
                    $opts->{tries}
                );
            } else {
                throw Exception gettext('Page with Page ID %s not found', $page_id);
            }

        }
        catch {
            my ($e) = @_;

            $error = $e->message();
            $error =~ s/\s+/ /g;
            if ($opts->{log_failed}) {
                my $elapsed = sprintf "%0.2f", gettimeofday() - $t1;

                writefile(
                    $opts->{log_failed},
                    logstr(
                        $opts->{cur_part_idx},
                        sprintf('%s/%s',      $count, $all_count),
                        sprintf('page_id=%s', $page_id),
                        'ERROR',
                        $error,
                        sprintf('model=%s',     $model),
                        sprintf('(elapsed %s)', $elapsed)
                    ),
                    append => TRUE,
                    lock   => TRUE
                );
            }
            $need_exit = 1 if ref $e eq 'Exception::OneshotsRepeaterFail';
        };

        my $elapsed = sprintf "%0.2f", gettimeofday() - $t1;

        if ($error) {
            print logstr(
                $opts->{cur_part_idx}, sprintf('%s/%s', $count, $all_count),
                'PAGE', sprintf('page_id=%s', $page_id),
                'ERROR', $error,
                sprintf('model=%s',     $model),
                sprintf('(elapsed %s)', $elapsed)
            );
        } else {
            print logstr(
                $opts->{cur_part_idx},
                sprintf('%s/%s',      $count, $all_count),
                sprintf('page_id=%s', $page_id),
                'OK', sprintf('(elapsed %s)', $elapsed)
            );
        }
        $count++;
        $has_error ||= !!$error;
    }

    return $has_error;
}

sub check_java_jsonapi_is_running {
    my ($app, $opts) = @_;

    if ($app->get_option('stage') eq 'dev'
        && ($opts->{from_java} || ('java' eq ($app->kv_store->get('bk_data_blocks_from') // ''))))
    {

        my $java_actuator_port = Utils::MakeFile::get_java_jsonapi_port() + 100;

        my $out = `curl -s http://localhost:$java_actuator_port/actuator/health || true`;
        my $data = eval {from_json($out)};

        unless ($data && $data->{'status'} eq 'UP') {
            print logstr('jsonapi', "No running JsonAPI found on port $java_actuator_port");
            print logstr('jsonapi', "This message may be ignored during oneshots on production");
        } else {
            print logstr('jsonapi', 'OK');
        }

    }

    return 1;
}

sub allow_send_special {
    my ($app, $opts) = @_;

    foreach my $page_accessor (@{$app->product_manager->get_page_model_accessors}) {

        my $multistate_graph = $app->$page_accessor->get_multistates_graph_definition();

        foreach my $action_data (@{$multistate_graph->{'multistate_actions'}}) {
            if ($action_data->{'action'} eq 'set_need_update') {
                # Разрешаем отправку из статуса deleted/rejected/protected
                $action_data->{'from'} = fix_action_condition_to_allow_statuses($action_data->{'from'},
                    [grep {$opts->{special_flags}{$_}} keys %{$opts->{special_flags}}]);
            }
        }

        # Применяем новый граф
        $app->$page_accessor->multistates_graph($multistate_graph);
    }

    if ($opts->{special_flags}->{'protected'}) {
        # позволяем отправлять пейджи с id < 1000
        $PiConstants::VIP_PROTECTED_MAX_PAGE_ID = 0;
    }

    return 1;
}

sub fix_action_condition_to_allow_statuses {
    my ($from, $special_statuses) = @_;

    foreach my $name (@$special_statuses) {
        foreach my $operator (qw( and  or )) {
            $from =~ s/$name $operator //;
            $from =~ s/ $operator $name//;
        }
        $from =~ s/$name//;
    }

    $from =~ s/(not\s*)?\(\s*\)//g;
    $from =~ s/^not $//;

    foreach my $operator (qw( and  or )) {
        $from =~ s/^ $operator //;
    }

    if ($from =~ /^\s*$/) {
        $from = '__EMPTY__ or not __EMPTY__';
    }

    return $from;
}

sub silent {
    my ($app) = @_;

    no warnings 'redefine';
    no strict 'refs';

    require Application::Model::API::Yandex::BK;
    require QBit::Application::Model::API::HTTP;

    *{'QBit::Application::Model::API::HTTP::INFO'}        = sub { };
    *{'Application::Model::API::Yandex::BK::INFOF'}       = sub { };
    *{'Application::Model::Role::JavaJsonApiProxy::INFO'} = sub { };
}

1;
