package Utils::Oneshots::Validator;

use qbit;
use Exception::DB;
use Utils::Oneshots;

=encoding UTF-8

=head1 DESCRIPTION

Скрипт для валидации площадок.

=head1 USAGE

  perl bin/validator.pl --page_ids=12345,54321,..
  perl bin/validator.pl --file_path=pageids.txt

=head1 OPTIONS

  page_ids        - Список ID площадок через запятую (необязательный)
  file_path       - Путь к файлу со списком площадок (разделитель перевод строки или запятая)
  split           - Разбить задание на указанное количество потоков
  part            - Размер пачки для проверки
  result          - Путь к файлу с итоговым результатом

=cut

sub main {
    my ($app, $opts) = @_;

    silent();

    my $page_ids = $opts->{page_ids};

    my $qvc         = $app->qbit_validator_checker;
    my %valid_model = map {$_ => TRUE} $qvc->get_model_names_that_use_qbit_validator();
    my $by_model    = split_page_by_model($app, $page_ids, $opts);

    my %errors;
    for my $accessor (keys %$by_model) {
        next unless $valid_model{$accessor};
        my $total      = @{$by_model->{$accessor}};
        my $maked      = 0;
        my $with_error = 0;
        while (my @part = splice @{$by_model->{$accessor}}, 0, $opts->{part}) {
            my ($is_ok, $details) = Utils::Oneshots::repeater(
                sub {
                    $qvc->check_all_elements_in_model(accessor => $accessor, page_ids => \@part);
                },
                $opts->{timeout},
                $opts->{tries}
            );

            unless ($is_ok) {
                my $r = add_report($app, \%errors, $details);
                $with_error += @{$details->{error_elements}};
                unless ($opts->{file_result}) {
                    print logstr "ERROR", $opts->{cur_part_idx}, $r;
                }
            }
            $maked += @part;
            print logstr "PROGRESS", $opts->{cur_part_idx}, $accessor, $maked, $total, $with_error;
        }
    }
    if (%errors) {
        my $data = to_json(\%errors, pretty => TRUE);
        if ($opts->{file_result}) {
            writefile($opts->{file_result}, $data, append => TRUE, lock => TRUE);
        }
        print logstr "ERROR", $opts->{cur_part_idx}, $data;
        return 1;
    } else {
        print logstr "OK", $opts->{cur_part_idx};
        return 0;
    }
}

sub split_page_by_model {
    my ($app, $list, $opts) = @_;

    my %result;

    for my $row (
        @{
            Utils::Oneshots::repeater(
                sub {
                    $app->partner_db->all_pages->get_all(
                        fields => ["model", "page_id"],
                        filter => [page_id => "IN" => \$list]
                    );
                },
                $opts->{timeout},
                $opts->{tries}
            )
        }
      )
    {
        my $accessor = $row->{model};
        push @{$result{$accessor}}, $row->{page_id};
        for my $block_accessor (@{$app->$accessor->get_block_model_names}) {
            push @{$result{$block_accessor}}, $row->{page_id};
        }
    }
    return \%result;
}

sub add_report {
    my ($app, $result, $detail) = @_;

    my $accessor = $detail->{model_name};
    my $model    = $app->$accessor;
    my %r;
    for my $e (@{$detail->{error_elements}}) {
        my $pid = $model->public_id($e->{pk});
        $result->{$pid} = $e->{error_fields};
        $r{$pid} = $e->{error_fields};
    }

    return \%r;
}

sub silent {

    no warnings 'redefine';
    no strict 'refs';

    require Application::Model::QBitValidatorChecker;
    require QBit::Application::Model::API::HTTP;

    *{'QBit::Application::Model::API::HTTP::INFO'}       = sub { };
    *{'Application::Model::QBitValidatorChecker::ERROR'} = sub { };
    *{'Application::Model::QBitValidatorChecker::INFOF'} = sub { };
    *{'Application::Model::QBitValidatorChecker::INFO'}  = sub { };
    *{'Application::Model::QBitValidatorChecker::WARN'}  = sub { };
    *{'Application::Model::QBitValidatorChecker::WARNF'} = sub { };
}

1;
