package Utils::PublicID;

=encoding UTF-8

=head1 DESCRIPTION

Набор функций по работе с Public ID.

L<https://wiki.yandex-team.ru/partner/w/partner2-public-id/>

=cut

our @ISA    = qw(Exporter);
our @EXPORT = qw(
  check_public_id
  check_public_ids
  split_block_public_id
  check_public_id
  validate_public_ids
  group_public_ids_by_model
  );

use qbit;

use Exception::IncorrectParams;
use Exception::Validation::BadArguments;

my %MODELS_BY_PUBLIC_ID_PREFIXES = (
    #    'R-A-'  => ['context_on_site_rtb'],
    #    'D-A-'  => ['context_on_site_direct', 'search_on_site_direct'],
    #    ...
);

=head2 split_block_public_id

    my ($prefix, $page_id, $block_id) = split_block_public_id('R-A-41443-1');
    # 'R-A-', 41443, 1

Функция принимает на вход строку, на входе отдает 3 значения. Функция не взаимодействует
с базой, так что совершенно нет никаких гарантий что в системе есть
блок с таким Public ID.

Prefix никак не валидируется, кроме того что он должен соответствовать
регулярному выражению (Т.е. строка Z-LLL-123-1 считется валидной).

В том случае если указананя строка не соответствует строкам вида 'R-A-41443-1',
то будет брошено исключение Exception::Validation::BadArguments
Функция не делает trim, в случае начальных / конечных пробелов функция бросает это же исключение.

В ПИ1 у RTB блоков был другой формат id - 'R-41443-1', вот эта функция НЕ умеет работать
с такими строками. Кроме строк вида 'R-A-41443-1' у нас еще бывают и другие виды public_id,
например "1", "2563070" (в случае DSP) или "123_4" (в случае block_tags). Эта функция так же
не умеет работать с такими id.

=cut

sub split_block_public_id {
    my ($public_id, $need_id) = @_;

    if ($public_id =~ /^([A-Z]+-[A-Z]+-)([1-9][0-9]*)-([1-9][0-9]*)\z/) {
        if (wantarray) {
            return ($1, $2, $3);
        } else {
            my @fn = ('prefix', 'page_id', ($need_id ? 'id' : 'block_id'), 'public_id');
            my %result;
            @result{@fn} = ($1, $2, $3, $public_id);
            return \%result;
        }
    } else {
        throw Exception::Validation::BadArguments gettext("'%s' is not valid", $public_id);
    }
}

sub check_public_id {
    my ($public_id, $type) = @_;

    my $wrong = check_public_ids([$public_id], $type);

    return @$wrong ? 0 : 1;
}

sub check_public_ids {
    my ($public_ids, $type) = @_;

    my $publicid_regexes = {
        'simple_id'          => qr/^[1-9][0-9]*\z/,
        'prefix-page-block'  => qr/^[A-Z]+-[A-Z]+-[1-9][0-9]*-[1-9][0-9]*\z/,
        'prefix-?page-block' => qr/^(?:[A-Z]+-[A-Z]+-)?[1-9][0-9]*-[1-9][0-9]*\z/,
    };

    my $regex = $publicid_regexes->{$type} // throw Exception::IncorrectParams 'Unknown PublicID type';

    my @wrong_public_ids = grep {!$_ || ref($_) || $_ !~ m/$regex/} @$public_ids;

    return \@wrong_public_ids;
}

sub validate_public_ids {
    my ($app, $public_ids) = @_;

    my $block_accessor_public_ids = group_public_ids_by_model($app, $public_ids);

    my $found = {
        #  <public_id> => <accesor>
    };

    foreach my $block_accessor (sort keys %$block_accessor_public_ids) {
        my $model_public_ids = $block_accessor_public_ids->{$block_accessor};
        my $res              = $app->$block_accessor->get_all(
            fields => [qw(page_id id)],
            filter => {id => $model_public_ids},
        );

        my $page_id_field_name = $app->$block_accessor->get_page_id_field_name();

        map {
            $found->{$app->$block_accessor->public_id({$page_id_field_name => $_->{'page_id'}, id => $_->{'id'}})} =
              $block_accessor
        } @$res;
    }

    my @not_found = grep {!$found->{$_}} @$public_ids;

    return ($found, \@not_found);
}

sub group_public_ids_by_model {
    my ($app, $public_ids) = @_;

    my $prefix_accessors = get_block_accessors_by_public_id_prefixes($app);

    my $block_accessor_public_ids = {
        #  'context_on_site_direct' => [ 'D-A-100-1', ... ],
        #  'search_on_site_direct'  => [ 'D-A-100-1', ... ],
    };
    foreach my $public_id (@$public_ids) {
        my ($prefix) = ($public_id =~ m/^(.*\-)\d+\-\d+$/);

        throw Exception::Validation::BadArguments gettext('No info for zero block')
          if $public_id =~ m/-0\z/;

        throw Exception::Validation::BadArguments gettext('Wrong public_id "%s"', $public_id)
          unless $prefix;

        my $block_accessors = $prefix_accessors->{$prefix};

        throw Exception::Validation::BadArguments gettext('Unknown public_id "%s"', $public_id)
          unless $block_accessors;

        foreach my $block_accessor (@$block_accessors) {
            my $ar = $block_accessor_public_ids->{$block_accessor} //= [];
            push @$ar, $public_id;
        }
    }

    return $block_accessor_public_ids;
}

sub get_block_accessors_by_public_id_prefixes {
    my ($app) = @_;

    my $prefix_accessors = \%MODELS_BY_PUBLIC_ID_PREFIXES;
    unless (%$prefix_accessors) {

        foreach my $product_accessor (sort @{$app->product_manager->get_page_model_accessors()}) {
            foreach my $block_model ($app->$product_accessor->get_block_models()) {
                my $prefix = $block_model->public_id_prefix();
                my $ar = $prefix_accessors->{$prefix} //= [];
                push @$ar, $block_model->accessor;
            }
        }
    }

    return $prefix_accessors;
}

TRUE;
