package Utils::Sandbox;

use qbit;
use Exporter;
use HTTP::Request;
use PiSecrets qw(get_secret);

our @ISA       = qw(Exporter);
our @EXPORT_OK = qw(
  create_and_start_sandbox_task
  wait_sandbox_task
  );
our @EXPORT = @EXPORT_OK;

my $SANDBOX_API_URL      = 'https://sandbox.yandex-team.ru/api/v1.0';
my $DEFAULT_WAIT_TIMEOUT = 1200;                                                                  # 20 minutes
my %TASK_ERROR_STATUSES  = map {$_ => 1} qw (FAILURE EXCEPTION NO_RES STOPPED EXPIRED TIMEOUT);

sub create_and_start_sandbox_task {
    my ($type, $fields) = @_;

    # Create new task with type $type
    my %create_data = (
        type          => $type,
        owner         => 'PARTNER',
        custom_fields => $fields || {},
    );

    my $response_data = sandbox_api_request('POST', $SANDBOX_API_URL . '/task', %create_data);

    my $task_id = $response_data->{'id'};

    die('Can not create sandbox task', 'red') unless defined($task_id);

    # Start task
    $response_data = sandbox_api_request(
        'PUT',
        $SANDBOX_API_URL . '/batch/tasks/start',
        (
            comment => 'Start task by partner robot',
            id      => [$task_id]
        )
    );

    my $status = $response_data->[0]{'status'};

    die('Can not start sandbox task. ID:' . $task_id . '. Status: ' . $status) unless $status eq 'SUCCESS';

    return $task_id;
}

sub wait_sandbox_task {
    my ($task_id, $wait_sec) = @_;

    $wait_sec //= $DEFAULT_WAIT_TIMEOUT;

    my $wait_time = time + $wait_sec;

    while (time < $wait_time) {
        sleep 20;

        my $task_info = get_sandbox_task_info($task_id);
        my $status = $task_info->{'status'} // 'undef';

        if ($status eq 'SUCCESS') {
            return $task_info->{'output_parameters'};
        }

        if (defined $TASK_ERROR_STATUSES{$status}) {
            die("Error during run sandbox task $task_id.");
        }
    }

    die("Sandbox task $task_id wait timeout.");
}

sub get_sandbox_task_info {
    my ($task_id) = @_;

    my $sandbox_check_status_url = $SANDBOX_API_URL . '/task/' . $task_id;

    my $info = sandbox_api_request('GET', $sandbox_check_status_url);

    return $info;
}

sub sandbox_api_request {
    my ($method, $url, %request_data) = @_;

    my $sandbox_token = get_secret('sandbox-token');

    my $request = HTTP::Request->new(
        $method => $url,
        [
            'Authorization' => sprintf('OAuth %s', $sandbox_token),
            'Content-Type'  => 'application/json',
        ],
        to_json(\%request_data)
    );

    my $browser = LWP::UserAgent->new(timeout => 60,);

    my $data;
    my $try = 3;
    while ($try--) {
        eval {
            my $response = $browser->request($request);

            if ($response->is_success()) {
                $data = from_json($response->decoded_content);
            }
        };

        last if $data;
        sleep(3);
    }

    die('Error while sandbox request') unless defined($data);

    return $data;
}

TRUE;
