package Utils::ScriptWrapper;

use qbit;

use base qw(Exporter);

use Pod::Usage;
use Pod::Find qw(pod_where);
use Getopt::Long qw();

use Application;

use Utils::Stream::DataSource::Callback;
use Utils::Stream::Serializer::JSON;
use Utils::Stream::Writer::OutStream;
use Utils::Safe;

use PiConstants qw($STREAMING_API_BUFFER_SIZE $SYSTEM_CRON_USER_ID);

our @EXPORT    = qw(run restore_check_rights _create_trigger);
our @EXPORT_OK = @EXPORT;

my $CHANGED_TABLES = {};

my $EXIT_OK   = 0;
my $EXIT_FAIL = 1;

my $OUT_TEE;
my $ERR_TEE;

my $MODE;
my $opts_by_mode = {
    oneshot => {
        params => {
            'append_logs!'       => 'append_logs',
            'over_logs!'         => 'over_logs',
            'debug!'             => 'debug',
            'dry_run|dry-run!'   => 'dry_run',
            'commit!'            => 'commit',
            'help|?|h'           => 'help',
            'preload_accessors!' => 'preload_accessors',
            'skip_logs!'         => 'skip_logs',
            'rollback:s'         => 'rollback',
            'ticket:s'           => 'ticket',
            'debug_count:i'      => 'debug_count',
            'limit:i'            => 'limit',
            'file_name:s'        => 'file_name',
        },
        force => {},
    },
    util => {
        params => {
            'dry_run|dry-run!' => 'dry_run',
            'commit!'          => 'commit',
            'help|?|h'         => 'help',
            'split:i'          => 'split',
            'limit:i'          => 'limit',
            'file_name:s'      => 'file_name',
        },
        force => {'skip_logs' => 1,},
    },
};

my $CHECK_RIGHTS;
my @ARGV_ORIG = ();

sub import {
    my $package = shift;
    my $mode    = shift;

    @ARGV_ORIG = @ARGV;

    Utils::ScriptWrapper->export_to_level(1, @_);

    unless ($MODE) {
        unless ($mode) {
            $MODE = 'oneshot';
        } elsif ($opts_by_mode->{$mode}) {
            $MODE = $mode;
        } else {
            die "invalid mode '$mode'";
        }
    }
}

END {
    close(STDOUT);
    close(STDERR);
    close($OUT_TEE) if defined($OUT_TEE);
    close($ERR_TEE) if defined($ERR_TEE);
}

sub run {
    my ($sub) = @_;

    confirm_connect_on_production();
    my $opts = _parse_argv();

    $ENV{'LAZY_LOAD'} = !$opts->{'preload_accessors'};

    my $file = $opts->{'file_name'};
    $file =~ s/"/\\"/g;

    my $tee_params = '';
    if ($opts->{'append_logs'}) {
        $tee_params = '-a';
    } elsif (!$opts->{skip_logs} && !$opts->{over_logs}) {
        my @exists_files = grep {-e $_} map {"$file.$_"} qw(out err);

        die sprintf('Files already exists: %s', join(', ', map {"\"$_\""} @exists_files)) if @exists_files;
    }

    unless ($opts->{skip_logs}) {
        open $OUT_TEE, "|-", sprintf('tee %s -- "%s"', $tee_params, "$file.out");
        open $ERR_TEE, "|-", sprintf('tee %s -- "%s"', $tee_params, "$file.err");

        open(STDOUT, '>&', $OUT_TEE);
        open(STDERR, '>&', $ERR_TEE);
    }

    my $code_exit = $EXIT_OK;
    try {
        my $argv_filtered = substr(join(' ', @ARGV_ORIG), 0, 1000);
        $argv_filtered =~ s/(token[= ]+)\S+/$1\[FILTERED TOKEN]/;
        print logstr('START WITH ARGS:', $argv_filtered);

        my $max_len       = 500;
        my $max_vals      = 100;
        my $filtered_opts = {};
        foreach my $key (keys %$opts) {
            my $val = $opts->{$key};
            next unless defined $val;
            if (ref $val eq 'ARRAY') {
                if (scalar(@$val) > $max_vals) {
                    $key .= ' [len=' . scalar(@$val) . ']';
                    $val = [@$val[0 .. $max_vals - 1], '...'];
                }
            } elsif (ref $val eq 'HASH') {
                $val = {map {$_ => $val->{$_}} grep {defined $val->{$_}} keys %$val};
            } elsif ($key =~ m/token/i) {
                $val = '[FILTERED TOKEN]';
            } elsif (length($val) > $max_len) {
                $key .= ' [len=' . length($val) . ']';
                $val = substr($val, 0, $max_len) . '...';
            }
            $filtered_opts->{$key} = $val;
        }

        print logstr('START WITH OPTS:', to_json($filtered_opts, canonical => TRUE));

        $opts->{cur_part_idx} = 0;
        my $skip;
        if ($opts->{split}) {
            printf logstr("RUN_PROC_CNT", $opts->{split});
            $opts->{$opts->{split_key}} = [];

            my $child;
            for (my $idx = 0; $idx < $opts->{split}; $idx++) {
                unless ($child = fork()) {
                    $opts->{$opts->{split_key}} = $opts->{split_data}[$idx];
                    $opts->{cur_part_idx} = $idx + 1;
                    last;
                }
            }

            while (wait != -1) {
                my $code = $? >> 8;
                $code_exit ||= $code;
            }
            $skip = $child;
        }

        unless ($skip) {
            print logstr("START", $opts->{cur_part_idx});

            my $app = get_app($opts);

            if ($MODE eq 'oneshot') {
                die sprintf('Argument "--ticket" is required') unless defined($opts->{'ticket'});

                $app->system_events->start('script', ticket => $opts->{'ticket'}) unless $opts->{'dry_run'};
            }

            my $code = $sub->($app, $opts);
            $code_exit ||= $code;

            print logstr('END', $opts->{cur_part_idx});
            print logstr('OUTFILE:', "\"$file.out\"", "\"$file.err\"") unless $opts->{skip_logs};

            if ($opts->{'debug'}) {
                generate_mysql_diff_report($app);
            }
            $app->post_run();
        } else {
            printf logstr("DONE_ALL", $opts->{split});
        }
    }
    catch {
        my ($err) = @_;

        $code_exit = $EXIT_FAIL;
        if (defined $ERR_TEE) {
            $ERR_TEE->print($err);
        } else {
            print STDERR $err;
        }
    };
    exit($code_exit) if $code_exit;
}

sub _parse_argv {
    my $opts = {};

    my ($class, $script) = caller(1);
    $opts->{class} = $class;

    if ($script eq '-e') {
        $script = 'perl_script';
    }

    ($opts->{'file_name'}) = reverse split(/\//, $script);
    $opts->{'file_name'} =~ s/(?<=.)\.[^\.]+$//;

    my %args = ();
    no strict 'refs';
    if (defined(*{$class . '::args'})) {
        %args = *{$class . '::args'}->($opts);
    }

    my $p = $opts_by_mode->{$MODE}{params};
    my $result = Getopt::Long::GetOptions(%args, map {$_ => \$opts->{$p->{$_}}} keys %$p,);

    if ($MODE eq 'oneshot') {
        $opts->{'file_name'} =~ s/PI-\d+/$opts->{ticket}/;
    }

    if (!$result || $opts->{'help'}) {
        pod2usage(
            -exitval   => $EXIT_FAIL,
            -message   => _get_default_pod_message(),
            -verbose   => 2,
            -noperldoc => 1,
            -input     => pod_where({-inc => 1}, $class)
        );
    }

    try {
        if (my $sub = $opts->{class}->can('prepare_args')) {
            $sub->($opts);
        }

        my $f = $opts_by_mode->{$MODE}{force};
        @{$opts}{keys %$f} = values %$f;

        if (my $split = $opts->{split}) {
            if (my $sub = $opts->{class}->can('key_data_for_split')) {
                my $key = $sub->($opts);
                unless (ref $opts->{$key} eq 'ARRAY') {
                    die "data in '$key' not arrayref\n";
                }
                my @parts;
                my $count = 0;
                foreach my $id (@{$opts->{$key}}) {
                    my $idx = $count++ % $split;
                    push @{$parts[$idx]}, $id;
                }

                if (@parts > 1) {
                    $opts->{split}      = @parts;
                    $opts->{split_data} = \@parts;
                    $opts->{split_key}  = $key;
                } else {
                    $opts->{split} = 0;
                }
            } else {
                die "Need 'sub key_data_for_split' in '$opts->{class}' for use '--split'\n";
            }
        }
    }
    catch {
        my ($e) = @_;
        warn $e->message, "\n";
        pod2usage(-exitval => $EXIT_FAIL, -verbose => 2, -noperldoc => 1, -input => pod_where({-inc => 1}, $class));
    };

    return $opts;
}

sub mock_table_methods {
    my ($app) = @_;

    $app->partner_db;

    no warnings 'redefine';

    my $add_sub = \&QBit::Application::Model::DB::mysql::Table::add_multi;
    *QBit::Application::Model::DB::mysql::Table::add_multi = sub {
        my ($self) = @_;

        my $table_name = $self->name;

        unless ($CHANGED_TABLES->{$table_name}) {
            _create_trigger($app, $table_name);
        }

        goto &$add_sub;
    };

    my $delete_sub = \&QBit::Application::Model::DB::mysql::Table::delete;
    *QBit::Application::Model::DB::mysql::Table::delete = sub {
        my ($self) = @_;

        my $table_name = $self->name;

        unless ($CHANGED_TABLES->{$table_name}) {
            _create_trigger($app, $table_name);
        }

        goto &$delete_sub;
    };

    my $edit_sub = \&QBit::Application::Model::DB::mysql::Table::edit;
    *QBit::Application::Model::DB::mysql::Table::edit = sub {
        my ($self) = @_;

        my $table_name = $self->name;

        unless ($CHANGED_TABLES->{$table_name}) {
            _create_trigger($app, $table_name);
        }

        goto &$edit_sub;
    };
}

sub _create_trigger {
    my ($app, $table_name) = @_;

    print logstr('CREATE TRIGGERS:', $table_name);

    # нельзя удалять или создавать таблицы внутри транзакции
    my $save_points = $app->partner_db->{'__SAVEPOINTS__'};
    while ($app->partner_db->{'__SAVEPOINTS__'}) {
        $app->partner_db->commit();
    }

    my $new_table_name = 'dump__' . $table_name;

    $app->partner_db->make_tables({$new_table_name => $table_name});

    my $new_table = $app->partner_db->$new_table_name;

    $new_table->drop(if_exists => TRUE);

    foreach (qw(foreign_keys indexes)) {
        delete($new_table->{$_});
    }

    my $field = $new_table->_get_field_object(
        name     => '__action__',
        type     => 'VARCHAR',
        length   => 6,
        not_null => TRUE,
        db       => $new_table->db,
        table    => $new_table
    );
    push(@{$new_table->{'fields'}}, $field);
    $new_table->{'__FIELDS_HS__'}{'__action__'} = $field;

    $new_table->create();

    my @fields = $app->partner_db->$table_name->field_names();

    my $quote_fields = join(', ', map {$app->partner_db->quote_identifier($_)} @fields, '__action__');

    $app->partner_db->_do("DROP TRIGGER IF EXISTS `insert__$table_name`;");
    $app->partner_db->_do(
        qq[
CREATE TRIGGER insert__$table_name AFTER INSERT ON $table_name FOR EACH ROW BEGIN INSERT IGNORE INTO $new_table_name ($quote_fields) VALUES (]
          . join(', ', (map {"NEW.$_"} @fields), '"insert"') . ');END;'
    );

    $app->partner_db->_do("DROP TRIGGER IF EXISTS `delete__$table_name`;");
    $app->partner_db->_do(
        qq[
CREATE TRIGGER delete__$table_name BEFORE DELETE ON $table_name FOR EACH ROW BEGIN INSERT IGNORE INTO $new_table_name ($quote_fields) VALUES (]
          . join(', ', (map {"OLD.$_"} @fields), '"delete"') . ');END;'
    );

    $app->partner_db->_do("DROP TRIGGER IF EXISTS `update__$table_name`;");
    $app->partner_db->_do(
        qq[
CREATE TRIGGER update__$table_name BEFORE UPDATE ON $table_name FOR EACH ROW BEGIN INSERT IGNORE INTO $new_table_name ($quote_fields) VALUES (]
          . join(', ', (map {"OLD.$_"} @fields), '"update"') . ');END;'
    );

    while ($save_points) {
        $app->partner_db->begin();
        $save_points--;
    }

    $CHANGED_TABLES->{$table_name} = $new_table_name;
}

sub generate_mysql_diff_report {
    my ($app) = @_;

    my @tables = ();
    foreach my $table_name (sort keys(%$CHANGED_TABLES)) {
        print logstr('GENERATE DEBUG DUMPS:', $table_name);

        my $table = $app->partner_db->$table_name;

        push(
            @tables,
            {
                name        => $table->name(),
                fields      => [$table->field_names()],
                primary_key => $table->primary_key(),
            }
        );

        my $dump_table_name = $CHANGED_TABLES->{$table_name};

        _dump_table($app, $dump_table_name, 'is_dump');
        _dump_table($app, $table_name);
    }

    writefile('./mysql_diff/src/json/tables.json', to_json(\@tables, pretty => TRUE));
}

sub _dump_table {
    my ($app, $table, $is_dump) = @_;

    my $offset = 0;
    my $limit  = 10_000;

    open(my $fh, '>', "./mysql_diff/src/json/$table.json") or die $!;
    binmode($fh, ':utf8');

    my $writer = Utils::Stream::Writer::OutStream->new(output => $fh);

    my $pk = $app->partner_db->$table->primary_key();

    my $filter;
    if ($is_dump) {
        $filter = ['__action__', '<>', \'insert'];
    } else {
        my $dump_table_name = 'dump__' . $table;

        $filter = [
            {'' => $pk},
            'IN', $app->partner_db->query->select(table => $app->partner_db->$dump_table_name, fields => $pk)
        ];
    }

    my $rows = [];

    my $serializer = Utils::Stream::Serializer::JSON->new(
        data => Utils::Stream::DataSource::Callback->new(
            source => sub {
                unless (@$rows) {
                    $rows = $app->partner_db->$table->get_all(
                        filter   => $filter,
                        order_by => $pk,
                        offset   => $offset,
                        limit    => $STREAMING_API_BUFFER_SIZE
                    );

                    $offset += @$rows;
                }

                return shift(@$rows);
            }
        ),
        pretty => TRUE,
    );

    $serializer->set_writer($writer);
    $serializer->serialize_full($STREAMING_API_BUFFER_SIZE);

    close($fh) or $!;
}

sub get_app {
    my ($opts) = @_;

    if (my $sub = $opts->{class}->can('configure_app')) {
        $sub->($opts);
    }

    my $app = Application->new();
    $app->pre_run();
    $app->set_cur_user({id => $SYSTEM_CRON_USER_ID, login => 'system-cron'});
    $app->set_option('find_app_mem_cycle', 0);

    {
        no warnings 'redefine';
        no strict 'refs';

        $CHECK_RIGHTS = \&QBit::Application::check_rights;

        *QBit::Application::check_rights = sub {1};

        if ($opts->{'debug'}) {
            mock_table_methods($app);
        }
    }

    if (my $sub = $opts->{class}->can('prepare_app')) {
        $sub->($app, $opts);
    }

    return $app;
}

sub restore_check_rights {
    no warnings 'redefine';
    no strict 'refs';

    *QBit::Application::check_rights = $CHECK_RIGHTS;
}

sub _get_default_pod_message {
    my $message = "\nOPTIONS(ScriptWrapper)\n";
    foreach (keys(%{$opts_by_mode->{$MODE}{'params'}})) {
        $message .= (' ' x 6) . $_ . "\n";
    }

    return $message;
}

TRUE;
