
=head1 Name

Utils::Stream::Serializer - модуль для валидации и сериализации данных

=head1 Description

БНФ входной структуры для сериализации.

data      := scalar   | reference
reference := arrayref | hashref
arrayref  := [ data* ]
hashref   := { (scalar data)* }

Знает про пакеты:
    Utils::Stream::Writer
    Utils::Stream::DataSource

Можно использовать в двух режимах, которые несовместимы между собой:
   1. Ручной режим
   2. Автоматический режим

writer - обязательный аргумент конструктора.
data - обязательный аргумент конструктора для автоматического режима.

В ручном режиме пользователь сам вызывает с нужными аргументами следующие функции:
   hash_begin
   hash_item
   hash_key
   hash_value_begin
   hash_value_end
   hash_end
   array_begin
   array_item
   array_item_begin
   array_item_end
   array_end
   scalar.
Внутри этих функций также происходит и валидация корретности очередности вызова.

В автоматическом предполагается, что все эти функции будут вызываться автоматически:
Пользователью необходимо:
   а) передать параметр data в конструктор
   б) вызывать serialize_batch(n), n > 0

Внутри себя, автоматический режим пользуется теми же фнукциями ручного режима.

Недостаток автоматического режима - использует много памяти (метод _linearize).
Поэтому, предполагается, что входные данные в этом режиме небольшие и упор сделан на DataSource.
Т.к. DataSource практически не увеличивает используемую для сериализации память.

=cut

package Utils::Stream::Serializer;

use base qw(QBit::Class);

use qbit;

use Utils::Stream::Writer::Buffer;

sub init {
    my ($self) = @_;

    $self->{_stack_} = [];
    $self->{_context_} = {expect => 'data',};
    $self->{writer} //= Utils::Stream::Writer::Buffer->new();

    return;
}

=head1 Methods

=head2 _level

Возвращает текущий уровень вложенности

=cut

sub _level {
    my ($self) = @_;
    return scalar @{$self->{_stack_}};
}

=head1 Methods

=head2 hash_begin

Говорит, что дальше все действия будут происходить внутри хеша

=cut

sub hash_begin {
    my ($self) = @_;

    my $expected = $self->{_context_}{expect};

    throw "Expected $expected" if $expected ne 'data';

    push @{$self->{_stack_}}, $self->{_context_};

    $self->{_context_} = {
        in_hash => 1,
        expect  => 'scalar',
    };
    return;
}

=head1 Methods

=head2 hash_item

Говорит, что дальше пара hash_key: hash_value, причем оба скаляры.

B<Arguments:>

=over

=item

B<$key> - ключ для хэша

=item

B<$value> - значение для ключа

=back

=cut

sub hash_item {
    my ($self, $key, $value) = @_;

    throw 'Not in hash' unless $self->{_context_}{in_hash};
    throw 'Expected hash value' if $self->{_context_}{expect} eq 'data';
    throw 'Hash key must be an scalar' if not defined($key) or ref($key);
    throw 'Expected scalar hash value' if ref($value) and ref($value) ne 'SCALAR';

    return;
}

=head1 Methods

=head2 hash_key

Говорит, что дальше ключ хэша.

B<Arguments:>

=over

=item

B<$key> - ключ для хэша

=back

=cut

sub hash_key {
    my ($self, $key) = @_;

    throw 'Not in hash' unless $self->{_context_}{in_hash};
    throw 'Expected hash value' if $self->{_context_}{expect} eq 'data';
    throw 'Hash key must be an scalar' if not defined($key) or ref($key);

    $self->{_context_}{key}    = $key;
    $self->{_context_}{expect} = 'data';

    return;
}

=head1 Methods

=head2 hash_item

Говорит, что дальше hash_value

=cut

sub hash_value_begin {
    my ($self) = @_;

    throw 'Not in hash' unless $self->{_context_}{in_hash};
    throw 'Expected hash key' if $self->{_context_}{expect} eq 'scalar';

    push @{$self->{_stack_}}, $self->{_context_};

    $self->{_context_} = {
        in_hash_value => 1,
        expect        => 'data',
    };
    return;
}

sub hash_value_end {
    my ($self) = @_;

    throw 'Not in hash value' unless $self->{_context_}{in_hash_value};

    $self->{_context_} = pop @{$self->{_stack_}};
    $self->{_context_}{expect} = 'scalar';

    return;
}

sub hash_end {
    my ($self) = @_;

    throw 'Not in hash' unless $self->{_context_}{in_hash};
    throw 'Odd elements in hash' if $self->{_context_}{expect} eq 'data';

    $self->{_context_} = pop @{$self->{_stack_}};
    $self->{_context_}{expect} = 'nothing' if $self->_level() == 0;

    return;
}

sub array_begin {
    my ($self) = @_;

    my $expected = $self->{_context_}{expect};

    throw "Expected $expected" if $expected ne 'data';

    push @{$self->{_stack_}}, $self->{_context_};

    $self->{_context_} = {
        in_array => 1,
        expect   => 'data',
    };
    return;
}

sub array_item {
    my ($self, $item) = @_;

    throw 'Not in array' unless $self->{_context_}{in_array};
    throw 'Expected scalar array item' if ref($item) and ref($item) ne 'SCALAR';

    return;
}

sub array_item_begin {
    my ($self) = @_;

    throw 'Not in array' unless $self->{_context_}{in_array};

    push @{$self->{_stack_}}, $self->{_context_};

    $self->{_context_} = {
        in_array_item => 1,
        expect        => 'data',
    };

    return;
}

sub array_item_end {
    my ($self) = @_;

    throw 'Not in array item' unless $self->{_context_}{in_array_item};

    $self->{_context_} = pop @{$self->{_stack_}};

    return;
}

sub array_end {
    my ($self) = @_;

    throw 'Not in array' unless $self->{_context_}{in_array};

    $self->{_context_} = pop @{$self->{_stack_}};
    $self->{_context_}{expect} = 'nothing' if $self->_level() == 0;

    return;
}

sub scalar {
    my ($self, $scalar) = @_;

    throw 'Expected nothing' if $self->{_context_}{expect} eq 'nothing';
    throw "Argument must be an scalar value" if ref($scalar) and ref($scalar) ne 'SCALAR';

    $self->{_context_}{expect} = 'nothing' if $self->_level() == 0;
    return;
}

sub end {
    my ($self) = @_;

    throw 'Incorrect end' if $self->_level() > 0;
    throw "Expected 'data'" if $self->{_context_}{expect} ne 'nothing';

    return;
}

sub handle_file_handler {
    my ($self, $data_source, $batch_size) = @_;

    while (1) {
        my $bulk = $data_source->get_bulk($batch_size);

        return unless @$bulk;

        my $data = join('', @$bulk);

        $self->{writer}->write($data);
    }

    return;
}

=head1 Methods

=head2 serialize_batch

Функция автоматического режима. Сериализирует, пока из DataSource
не будут извлечены следующие n элементов.

B<Arguments:>

=over

=item

B<$n> - сколько элементов извлечь из DataSource

=back

=cut

sub serialize_batch {
    my ($self, $batch_size) = @_;

    throw "'data' param is not provided" unless $self->{'data'};

    $self->_linearize($self->{'data'}, $batch_size) unless $self->{_linearized_};

    return unless @{$self->{_linearized_}};

    while ($batch_size > 0) {
        last unless @{$self->{_linearized_}};
        my $elem = $self->{_linearized_}[0];
        if (blessed($elem)) {
            $self->_handle_batch($elem, $batch_size);
        } else {
            my $method = $elem->{method};
            my $args = $elem->{args} // [];
            $self->$method(@$args);
        }
        shift @{$self->{_linearized_}} unless blessed($elem) and $elem->has_next();
    }

    $self->end() unless @{$self->{_linearized_}};

    return;
}

sub _handle_batch {
    my ($self, $data_source, $batch_size) = @_;

    while ($batch_size and $data_source->has_next()) {
        my $next = $data_source->next();
        if (ref($next) and ref($next) ne 'SCALAR') {
            $self->array_item_begin();
            $self->_traverse($next);
            $self->array_item_end();
        } else {
            $self->array_item($next);
        }
        --$batch_size;
    }
    return;
}

=head1 Methods

=head2 _linearize

Вспомогательный метод для автоматического режима.
Трансформирует иерархическую структуру в плоскую.
Благодаря этому, можем прервать сериализацию в любом месте,
а позже продолжить с того же самого места.

=cut

sub _linearize {
    my ($self, $data, $batch_size) = @_;

    $self->{_linearized_} //= [];

    if (not ref($data) or ref($data) eq 'SCALAR') {
        push @{$self->{_linearized_}}, {method => 'scalar', args => [$data]};
    } elsif (blessed($data)) {
        if ($data->isa('Utils::Stream::DataSource::FileHandler')) {
            push @{$self->{_linearized_}}, {method => 'handle_file_handler', args => [$data, $batch_size]};
        } elsif ($data->isa('Utils::Stream::DataSource')) {
            push @{$self->{_linearized_}}, {method => 'array_begin'};
            push @{$self->{_linearized_}}, $data;
            push @{$self->{_linearized_}}, {method => 'array_end'};
        } else {
            throw gettext('Can\'t convert object');
        }
    } elsif (ref($data) eq 'ARRAY') {
        push @{$self->{_linearized_}}, {method => 'array_begin'};
        for my $i (0 .. $#$data) {
            if (ref($data->[$i]) and ref($data->[$i]) ne 'SCALAR') {
                push @{$self->{_linearized_}}, {method => 'array_item_begin'};
                $self->_linearize($data->[$i], $batch_size);
                push @{$self->{_linearized_}}, {method => 'array_item_end'};
            } else {
                push @{$self->{_linearized_}}, {method => 'array_item', args => [$data->[$i]]};
            }
        }
        push @{$self->{_linearized_}}, {method => 'array_end'};
    } elsif (ref($data) eq 'HASH') {
        push @{$self->{_linearized_}}, {method => 'hash_begin'};
        for my $key (sort keys %$data) {
            if (ref($data->{$key}) and ref($data->{$key}) ne 'SCALAR') {
                push @{$self->{_linearized_}}, {method => 'hash_key', args => [$key]};
                push @{$self->{_linearized_}}, {method => 'hash_value_begin'};
                $self->_linearize($data->{$key}, $batch_size);
                push @{$self->{_linearized_}}, {method => 'hash_value_end'};
            } else {
                push @{$self->{_linearized_}}, {method => 'hash_item', args => [$key, $data->{$key}]};
            }
        }
        push @{$self->{_linearized_}}, {method => 'hash_end'};
    } else {
        throw gettext('Can\'t convert object');
    }

    return;
}

=head1 Methods

=head2 _traverse

Т.к. мы заранее не знаем содержимое DataSource, не можем
трансформировать его в плоскую структуру. Следовательно, каждый раз,
когда извлекаем очередной элемент из DataSource, мы должны его обойти отдельно.
Эта функция как раз таки этим и занимается.

=cut

sub _traverse {
    my ($self, $data) = @_;

    if (not ref($data) or ref($data) eq 'SCALAR') {
        $self->scalar($data);
    } elsif (ref($data) eq 'ARRAY') {
        $self->array_begin();
        for my $i (0 .. $#$data) {
            if (ref($data->[$i]) and ref($data->[$i]) ne 'SCALAR') {
                $self->array_item_begin();
                $self->_traverse($data->[$i]);
                $self->array_item_end();
            } else {
                $self->array_item($data->[$i]);
            }
        }
        $self->array_end();
    } elsif (ref($data) eq 'HASH') {
        $self->hash_begin();
        for my $key (sort keys %$data) {
            if (ref($data->{$key}) and ref($data->{$key}) ne 'SCALAR') {
                $self->hash_key($key);
                $self->hash_value_begin();
                $self->_traverse($data->{$key});
                $self->hash_value_end();
            } else {
                $self->hash_item($key, $data->{$key});
            }
        }
        $self->hash_end();
    } else {
        throw gettext('Can\'t convert object');
    }

    return;
}

sub serialize_full {
    my ($self, $batch_size) = @_;

    do {
        $self->serialize_batch($batch_size);
    } while (@{$self->{_linearized_}});

    return;
}

sub set_writer {
    my ($self, $writer) = @_;

    if ($self->{writer}->isa('Utils::Stream::Writer::Buffer')) {
        $writer->write($self->{writer}->content());
        $self->{writer}->clear();
    }
    $self->{writer} = $writer;

    return;
}

1;
