package Utils::Stream::Serializer::JSON;

use base qw(Utils::Stream::Serializer);

use qbit;

sub init {
    my ($self) = @_;
    $self->SUPER::init();

    $self->{pretty} = $self->{pretty} ? 1 : 0;
    $self->{indent} = ($self->{indent} // 3) + 0;

    $self->{_colon_} = $self->{pretty} ? ' : ' : ':';

    $self->{_shift_} = 0;

    return;
}

sub _indent {
    my ($self) = @_;
    return '' if not $self->{pretty};
    return "\n" . ' ' x ($self->{indent} * ($self->_level() - $self->{_shift_}));
}

sub _comma {
    my ($self) = @_;
    $self->{writer}->write(',') if $self->{_context_}{has_item};
    return;
}

sub _write {
    my ($self, $str) = @_;

    $self->{writer}->write($self->_indent() . $str);
    return;
}

sub hash_begin {
    my ($self) = @_;
    if ($self->{_context_}{in_hash_value} or $self->_level() == 0) {
        $self->{writer}->write('{');
    } else {
        $self->_write('{');
    }
    $self->SUPER::hash_begin();
    return;
}

sub hash_item {
    my ($self, $key, $value) = @_;
    $self->SUPER::hash_item($key, $value);
    $self->_comma();

    $key = $self->_quote($self->_json_escape($key));

    $value = $$value if ref($value) eq 'SCALAR';

    if (defined($value)) {
        $value = $self->_quote($self->_json_escape($value));
    } else {
        $value = 'null';
    }

    $self->_write("$key" . $self->{_colon_} . $value);

    $self->{_context_}{has_item} = 1;
    return;
}

sub hash_key {
    my ($self, $key) = @_;

    $self->SUPER::hash_key($key);
    $self->_comma();

    $key = $self->_quote($self->_json_escape($key));

    $self->_write("$key" . $self->{_colon_});
    return;
}

sub hash_value_begin {
    my ($self) = @_;
    $self->SUPER::hash_value_begin();
    $self->{_shift_}++;
    return;
}

sub hash_value_end {
    my ($self) = @_;
    $self->SUPER::hash_value_end();
    $self->{_shift_}--;
    $self->{_context_}{has_item} = 1;
    return;
}

sub hash_end {
    my ($self) = @_;
    $self->SUPER::hash_end();
    my $context = $self->{_context_};
    $self->_write('}');
    return;
}

sub array_begin {
    my ($self) = @_;
    if ($self->{_context_}{in_hash_value} or $self->_level() == 0) {
        $self->{writer}->write('[');
    } else {
        $self->_write('[');
    }
    $self->SUPER::array_begin();
    return;
}

sub array_item {
    my ($self, $item) = @_;
    $self->_comma();
    $self->SUPER::array_item($item);

    $item = $$item if ref($item) eq 'SCALAR';

    if (defined($item)) {
        $item = $self->_quote($self->_json_escape($item));
    } else {
        $item = 'null';
    }

    $self->_write($item);

    $self->{_context_}{has_item} = 1;
    return;
}

sub array_item_begin {
    my ($self) = @_;
    $self->_comma();
    $self->SUPER::array_item_begin();
    $self->{_shift_}++;
    return;
}

sub array_item_end {
    my ($self) = @_;
    $self->SUPER::array_item_end();
    $self->{_shift_}--;
    $self->{_context_}{has_item} = 1;
    return;
}

sub array_end {
    my ($self) = @_;
    $self->SUPER::array_end();
    my $context = $self->{_context_};
    $self->_write(']');
    return;
}

sub scalar {
    my ($self, $scalar) = @_;

    $self->SUPER::scalar($scalar);

    $scalar = $$scalar if ref($scalar) eq 'SCALAR';

    if (defined($scalar)) {
        $scalar = $self->_quote($self->_json_escape($scalar));
    } else {
        $scalar = 'null';
    }

    $self->_write($scalar);
    return;
}

sub end {
    my ($self) = @_;
    $self->{writer}->write("\n") if $self->{pretty};
    $self->SUPER::end();
    return;
}

sub _json_escape {
    my ($class, $str) = @_;

    local $_ = $str;

    s/\\/\\\\/g;
    s/\f/\\f/g;
    s/\n/\\n/g;
    s/\r/\\r/g;
    s/\t/\\t/g;
    s/"/\\"/g;

    return $_;
}

sub _quote {
    my ($class, $str) = @_;
    return '"' . $str . '"';
}

sub _handle_batch {
    my ($self, $data_source, $batch_size) = @_;

    my $bulk = $data_source->get_bulk($batch_size);

    return unless @$bulk;

    # we already have one wrap
    my $wrap_count = $self->_level() - $self->{_shift_} - 1;

    # засовываем рекурсивно в arrayref, пока не получим требуемый indent на выходе
    # т.к. JSON::XS не позволяет задавать indent
    $bulk = [$bulk] for (1 .. $wrap_count);

    my $json = to_json($bulk, $self->{pretty} ? (pretty => 1) : (canonical => 1));

    # теперь должны понять, что мы должны удалить с головы и с хвоста
    my ($cr, $indent) = $self->{pretty} ? ("\n", '   ') : ('', '');

    my $head = '[';
    for my $i (1 .. $wrap_count) {
        $head .= $cr . $indent x $i . '[';
    }

    my $tail = $cr;
    for my $i (reverse(0 .. $wrap_count)) {
        $tail .= $indent x $i . "]$cr";
    }

    $json = substr($json, length($head), -length($tail));

    $self->{writer}->write($json);

    $self->{writer}->write(',') if $data_source->has_next();

    return;
}

1;
