package Utils::Stream::Serializer::TSV;

use base qw(Utils::Stream::Serializer);

use qbit;
use Utils::TSV qw(tsv_with_fields);

sub init {
    my ($self) = @_;
    $self->SUPER::init();

    $self->{_print_headers_} = not $self->{no_headers};

    if ($self->{fields}) {
        throw "'fields' must be an 'ARRAY' reference" if ref($self->{fields}) ne 'ARRAY';
        $self->_write(@{$self->{fields}}) if $self->{_print_headers_};
    }

    $self->{end_marker} //= "#END";

    return;
}

sub _write {
    my ($self, @strings) = @_;
    $self->{writer}->write(join("\t", map {$self->_tsv_escape($_)} @strings) . "\n");
    return;
}

sub _tsv_escape {
    my ($class, $str) = @_;

    return '' unless defined $str;

    local $_ = $str;

    s/\\/\\\\/g;
    s/\f/\\f/g;
    s/\n/\\n/g;
    s/\r/\\r/g;
    s/\t/\\t/g;

    return $_;
}

sub end {
    my ($self) = @_;
    $self->{end_marker} = ${$self->{end_marker}} if ref($self->{end_marker}) eq 'SCALAR';
    $self->{end_marker} = $self->_tsv_escape($self->{end_marker});
    $self->{writer}->write($self->{end_marker} . "\n") if length($self->{end_marker});
    return;
}

sub _linearize {
    my ($self, $data, $batch_size) = @_;

    if (ref($data) eq 'ARRAY') {
        push @{$self->{_linearized_}}, {method => 'array_begin'};
        foreach (@$data) {
            push @{$self->{_linearized_}}, {method => 'array_item', args => [$_]};
        }
        push @{$self->{_linearized_}}, {method => 'array_end'};

        return;
    }

    return $self->SUPER::_linearize($data, $batch_size);
}

sub array_item {
    my ($self, $item) = @_;

    throw 'Not in array' unless $self->{_context_}{in_array};
    throw 'Expected hash array item' unless 'HASH' eq ref($item);

    $self->_print_headers_once($item);

    $self->_write(map {$item->{$_}} @{$self->{fields}});

    return;
}

sub _handle_batch {
    my ($self, $data_source, $batch_size) = @_;

    my $bulk = $data_source->get_bulk($batch_size);
    return unless @$bulk;

    $self->_print_headers_once($bulk->[0]);

    $self->{writer}->write(
        tsv_with_fields(
            $bulk,
            $self->{fields},
            without_headers => TRUE,
            end_marker      => FALSE,
            string_escape   => TRUE,
        ),
    );

    return;
}

sub _print_headers_once {
    my ($self, $first_item) = @_;

    unless (defined($self->{fields})) {
        $self->{fields} = [sort keys %$first_item];
        $self->_write(@{$self->{fields}}) if $self->{_print_headers_};
    }
}

1;
